<?php

namespace greenshiftseo\Rest;

use WP_REST_Request;
use WP_REST_Server;

defined( 'ABSPATH' ) OR exit;

class REST {
	private $rest_namespace = 'greenshift/v1';

	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'action_rest_api_init_trait' ) );
	}

	public function action_rest_api_init_trait() {

		register_rest_route(
			$this->rest_namespace,
			"/wc-listing/",
			array(
				'methods'  => WP_REST_Server::CREATABLE,
				'permission_callback' => function ( WP_REST_Request $request ) {
					return current_user_can( 'editor' ) || current_user_can( 'administrator' );
				},
				'callback' => array( $this, 'rest_wc_listing_handler' ),
			)
		);

		register_rest_route(
			$this->rest_namespace,
			"/product/(?P<id>\d+)",
			array(
				'methods'  => WP_REST_Server::READABLE,
				'permission_callback' => function ( WP_REST_Request $request ) {
					return current_user_can( 'editor' ) || current_user_can( 'administrator' );
				},
				'callback' => array( $this, 'rest_product_handler' ),
			)
		);
	}

	public function rest_product_handler( WP_REST_Request $request ) {
		$id   = $request->get_params()['id'];
		$data = array();

		if ( empty( $id ) ) {
			return new \WP_Error( 'empty_data', 'Pass empty data', array( 'status' => 404 ) );
		}

		$product              = wc_get_product( $id );
		$thumbID 			  = $product->get_image_id();
		$copy                 = $product->get_short_description();
		$product_on_sale      = $product->is_on_sale();
		$regular_price        = (float) $product->get_regular_price();
		$sale_price           = (float) $product->get_sale_price();
		$attributes           = $product->get_attributes();
		$gallery_ids          = $product->get_gallery_image_ids();
		$price_label          = '';
		$gallery_images       = array();
		$product_name         = $product->get_title();
		$add_to_cart_text     = $product->add_to_cart_text();
		$product_type         = $product->get_type();
		
		if ( empty( $thumbID ) ) {
			$thumbnail_url = plugin_dir_url( __FILE__ ) . 'assets/icons/noimage-placeholder.png';
		}else{
			$thumbnail_url = wp_get_attachment_image_url($thumbID);
		}
		
		if ( ! empty( $copy ) ) {
			$copy = wp_trim_words($copy, 15);
		}

		if ( $product_on_sale && $regular_price && $sale_price > 0 && $product_type !== 'variable' ) {
			$sale_proc   = 0 - ( 100 - ( $sale_price / $regular_price ) * 100 );
			$sale_proc   = round( $sale_proc );
			$price_label = $sale_proc . '%';
		}

		if ( ! empty( $attributes ) ) {
			ob_start();
			wc_display_product_attributes( $product );
			$attributes = ob_get_contents();
			ob_end_clean();
		}

		if ( ! empty( $gallery_ids ) ) {
			foreach ( $gallery_ids as $key => $value ) {
				$gallery_images[] = wp_get_attachment_url( $value );
			}
		}

		$data['thumbnail'] = array(
			'url' => $thumbnail_url,
			'id'  => $thumbID
		);
		$data['title']             = $product_name;
		$data['copy']              = $copy;
		$data['priceLabel']        = $price_label;
		$data['addToCartText']     = $add_to_cart_text;
		$data['productAttributes'] = $attributes;
		$data['galleryImages']     = $gallery_images;
		$data['userscore']		   = $product->get_average_rating();
		$data['priceHtml']         = $product->get_price_html();
		//$data['id']                = $id;

		return json_encode( $data );
	}

	public function rest_wc_listing_handler( WP_REST_Request $request ) {
		$posts_id = $request['posts_id'];
		$data     = array();

		if ( empty( $posts_id ) || count( $posts_id ) === 0 ) {
			return new \WP_Error( 'empty_data', 'Pass empty data', array( 'status' => 404 ) );
		}

		foreach ( $posts_id as $index => $id ) {
			$product = wc_get_product( $id );

			$thumbID = $product->get_image_id();
			if ( empty( $thumbID ) ) {
				$thumbnail_url = plugin_dir_url( __FILE__ ) . 'assets/icons/noimage-placeholder.png';
			}else{
				$thumbnail_url = wp_get_attachment_image_url($thumbID);
			}
			$copy = $product->get_short_description();
			if ( ! empty( $copy ) ) {
				$copy = wp_trim_words($copy, 15);
			}
			
			$data[$index] = array(
				'userscore'          => $product->get_average_rating(),
				'thumbnail'      => array(
					'url' => $thumbnail_url,
					'id'  => $thumbID
				),
				'title'          => $product->get_title(),
				'copy'           => $copy,
				'currentPrice'   => $product->get_price(),
				'oldPrice'       => $product->get_regular_price(),
				'addToCartText'  => $product->add_to_cart_text(),
				'priceHtml'		 => $product->get_price_html(),
				'id'             => $id
			);
		}

		return json_encode( $data );
	}
	
    public function get_host_name( $url ) {
		$domain = strtolower(str_ireplace('www.', '', parse_url($url, PHP_URL_HOST)));
		
		// remove subdomain
        if (preg_match('/(?P<domain>[a-z0-9][a-z0-9\-]{1,63}\.[a-z\.]{2,6})$/i', $domain, $regs)) {
            $domain = $regs['domain'];
        }
		
		$hostData = explode('.', $domain);
		
		return $hostData[0];
    }
}
new \greenshiftseo\Rest\REST;