<?php
/**
 * Plugin Name:           AutomatorWP Pro
 * Plugin URI:            https://automatorwp.com/add-ons/automatorwp-pro/
 * Description:           All the power of AutomatorWP and all its premium add-ons in a single add-on :)
 * Version:               4.7.3
 * Author:                AutomatorWP
 * Author URI:            https://automatorwp.com/
 * Text Domain:           automatorwp-pro
 * Domain Path:           /languages/
 * Requires at least:     4.4
 * Tested up to:          6.8
 * License:               GNU AGPL v3.0 (http://www.gnu.org/licenses/agpl.txt)
 *
 * @package               AutomatorWP\Pro
 * @author                AutomatorWP
 * @copyright             Copyright (c) AutomatorWP
 */

final class AutomatorWP_Pro {

    /**
     * @var         AutomatorWP_Pro $instance The one true AutomatorWP_Pro
     * @since       1.0.0
     */
    private static $instance;

    /**
     * Get active instance
     *
     * @access      public
     * @since       1.0.0
     * @return      AutomatorWP_Pro self::$instance The one true AutomatorWP_Pro
     */
    public static function instance() {
        if( !self::$instance ) {
            self::$instance = new AutomatorWP_Pro();
            self::$instance->constants();
            self::$instance->includes();
            self::$instance->integrations();
            self::$instance->hooks();
        }

        return self::$instance;
    }

    /**
     * Setup plugin constants
     *
     * @access      private
     * @since       1.0.0
     * @return      void
     */
    private function constants() {
        // Plugin version
        define( 'AUTOMATORWP_PRO_VER', '4.7.3' );

        // Plugin file
        define( 'AUTOMATORWP_PRO_FILE', __FILE__ );

        // Plugin path
        define( 'AUTOMATORWP_PRO_DIR', plugin_dir_path( __FILE__ ) );

        // Plugin URL
        define( 'AUTOMATORWP_PRO_URL', plugin_dir_url( __FILE__ ) );
    }

    /**
     * Include plugin files
     *
     * @access      private
     * @since       1.0.0
     * @return      void
     */
    private function includes() {

        if( $this->meets_requirements() ) {

        }
    }

    /**
     * Include integrations files
     *
     * @access      private
     * @since       1.0.0
     * @return      void
     */
    private function integrations() {

        if( ! $this->meets_requirements() ) {
            return;
        }

        $integrations_dir = AUTOMATORWP_PRO_DIR . 'integrations';

        $integrations = @opendir( $integrations_dir );

        // Setup active plugins
        $active_plugins = array();

        if( function_exists( 'get_option' ) ) {
            $active_plugins = (array) get_option( 'active_plugins', array() );
        }

        // Setup active sitewide plugins
        $active_sitewide_plugins = array();

        if ( is_multisite() && function_exists( 'get_site_option' ) ) {
            $active_sitewide_plugins = get_site_option( 'active_sitewide_plugins' );

            if( ! is_array( $active_sitewide_plugins ) ) {
                $active_sitewide_plugins = array();
            }
        }

        while ( ( $integration = @readdir( $integrations ) ) !== false ) {

            if ( $integration === '.' || $integration === '..' || $integration === 'index.php' ) {
                continue;
            }

            // Skip if integration is already active
            if( $this->is_integration_active( $integration, $active_plugins, $active_sitewide_plugins ) ) {
                continue;
            }

            // /automatorwp-pro/integrations/{integration}/{integration}.php
            $integration_file = $integrations_dir . DIRECTORY_SEPARATOR . $integration . DIRECTORY_SEPARATOR . $integration . '.php';

            // Skip if no file to load
            if( ! file_exists( $integration_file ) ) {
                continue;
            }

            require_once $integration_file;

        }

        closedir( $integrations );

    }

    /**
     * Include integrations files
     *
     * @access      private
     * @since       1.0.0
     * @param       string  $integration
     * @param       array   $active_plugins
     * @param       array   $active_sitewide_plugins
     * @return      bool
     */
    private function is_integration_active( $integration, $active_plugins, $active_sitewide_plugins ) {

        $plugins = array(
            "automatorwp-{$integration}/automatorwp-{$integration}.php",
        );

        if( $integration === 'elementor' ) {
            $plugins = array(
                "automatorwp-{$integration}-forms/automatorwp-{$integration}-forms.php",
            );
        }

        foreach( $plugins as $plugin ) {

            // Bail if plugin is active
            if( in_array( $plugin, $active_plugins, true ) ) {
                return true;
            }

            // Bail if plugin is network wide active
            if ( isset( $active_sitewide_plugins[$plugin] ) ) {
                return true;
            }

            // Consider integration active during it's activation
            if( isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'activate'
                && isset( $_REQUEST['plugin'] ) && $_REQUEST['plugin'] === $plugin ) {
                return true;
            }

            // Support for bulk activate
            if( isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'activate-selected'
                && isset( $_REQUEST['checked'] ) && is_array( $_REQUEST['checked'] )
                && in_array( $plugin, $_REQUEST['checked'] ) ) {
                    return true;
            }

        }

        return false;

    }

    /**
     * Setup plugin hooks
     *
     * @access      private
     * @since       1.0.0
     * @return      void
     */
    private function hooks() {

        add_filter( 'automatorwp_licenses_meta_boxes', array( $this, 'license' ) );

        add_action( 'admin_notices', array( $this, 'admin_notices' ) );

        // Remove recommendations on automations UI
        remove_action( 'automatorwp_automation_ui_add_item_content_bottom', 'automatorwp_automation_ui_integrations_recommendations', 10 );

        add_action( 'init', array( $this, 'load_textdomain' ) );

    }

    /**
     * Licensing
     *
     * @since 1.0.0
     *
     * @param array $meta_boxes
     *
     * @return array
     */
    function license( $meta_boxes ) {

        $meta_boxes['automatorwp-pro-license'] = array(
            'title' => 'AutomatorWP Pro',
            'fields' => array(
                'automatorwp_pro_license' => array(
                    'type' => 'edd_license',
                    'file' => AUTOMATORWP_PRO_FILE,
                    'item_name' => 'AutomatorWP Pro',
                ),
            )
        );

        return $meta_boxes;

    }

    /**
     * Plugin admin notices.
     *
     * @since  1.0.0
     */
    public function admin_notices() {

        if ( ! $this->meets_requirements() && ! defined( 'AUTOMATORWP_ADMIN_NOTICES' ) ) : ?>

            <div id="message" class="notice notice-error is-dismissible">
                <p>
                    <?php printf(
                        __( 'AutomatorWP Pro requires %s in order to work. Please install and activate AutomatorWP.', 'automatorwp-pro' ),
                        '<a href="https://wordpress.org/plugins/automatorwp/" target="_blank">AutomatorWP </a>'
                    ); ?>
                </p>
            </div>

            <?php define( 'AUTOMATORWP_ADMIN_NOTICES', true ); ?>

        <?php endif;

    }

    /**
     * Check if there are all plugin requirements
     *
     * @since  1.0.0
     *
     * @return bool True if installation meets all requirements
     */
    private function meets_requirements() {

        if ( ! class_exists( 'AutomatorWP' ) ) {
            return false;
        }

        return true;

    }

    /**
     * Internationalization
     *
     * @access      public
     * @since       1.0.0
     * @return      void
     */
    public function load_textdomain() {

        // Set filter for language directory
        $lang_dir = AUTOMATORWP_PRO_DIR . '/languages/';
        $lang_dir = apply_filters( 'automatorwp_pro_languages_directory', $lang_dir );

        // Traditional WordPress plugin locale filter
        $locale = apply_filters( 'plugin_locale', get_locale(), 'automatorwp-pro' );
        $mofile = sprintf( '%1$s-%2$s.mo', 'automatorwp-pro', $locale );

        // Setup paths to current locale file
        $mofile_local   = $lang_dir . $mofile;
        $mofile_global  = WP_LANG_DIR . '/automatorwp-pro/' . $mofile;

        if( file_exists( $mofile_global ) ) {
            // Look in global /wp-content/languages/automatorwp-pro/ folder
            load_textdomain( 'automatorwp-pro', $mofile_global );
        } elseif( file_exists( $mofile_local ) ) {
            // Look in local /wp-content/plugins/automatorwp-pro/languages/ folder
            load_textdomain( 'automatorwp-pro', $mofile_local );
        } else {
            // Load the default language files
            load_plugin_textdomain( 'automatorwp-pro', false, $lang_dir );
        }

    }

}

/**
 * The main function responsible for returning the one true AutomatorWP_Pro instance to functions everywhere
 *
 * @since       1.0.0
 * @return      \AutomatorWP_Pro The one true AutomatorWP_Pro
 */
function AutomatorWP_Pro() {
    return AutomatorWP_Pro::instance();
}

$automatorwp_path = '';

if( defined( 'WP_PLUGIN_DIR' ) ) {
    if( file_exists( WP_PLUGIN_DIR . '/automatorwp/automatorwp.php' ) ) {
        $automatorwp_path = WP_PLUGIN_DIR . '/automatorwp/automatorwp.php';
    }
}

if( empty( $automatorwp_path ) && file_exists( plugin_dir_path( __FILE__ ) . '../automatorwp/automatorwp.php' ) ) {
    $automatorwp_path = plugin_dir_path( __FILE__ ) . '../automatorwp/automatorwp.php';
}

// Force to load AutomatorWP core if installed
if ( ! empty( $automatorwp_path ) && ! class_exists( 'AutomatorWP' ) ) {
    require_once $automatorwp_path;
}

AutomatorWP_Pro();
