<?php
/**
 * Anonymous Books Event
 *
 * @package     AutomatorWP\Integrations\Amelia\Triggers\Anonymous_Books_Event
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Amelia_Anonymous_Books_Event extends AutomatorWP_Integration_Trigger {

    public $integration = 'ameliabooking';
    public $trigger = 'ameliabooking_anonymous_books_event';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            'label'             => __( 'Guest books for an event', 'automatorwp-pro' ),
            'select_option'     => __( 'Guest books for an <strong>event</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Number of times. */
            'edit_label'        => sprintf( __( 'Guest books for %1$s %2$s time(s)', 'automatorwp-pro' ), '{event}', '{times}' ),
            'log_label'         => sprintf( __( 'Guest books for %1$s', 'automatorwp-pro' ), '{event}' ),
            'action'            => 'AmeliaBookingAddedBeforeNotify',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'event' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'event',
                    'name'              => __( 'Event:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any event', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_ameliabooking_get_events',
                    'options_cb'        => 'automatorwp_ameliabooking_options_cb_event',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_ameliabooking_get_event_tags(),
                automatorwp_ameliabooking_get_customer_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array       $args    Appointment data
     */
    public function listener( $args ) {

        // Bail if can not find any data
        if ( empty( $args ) ) {
            return;
        }

        $user_id = get_current_user_id( );

        // Bail if user is not logged
        if ($user_id !== 0){
            return;
        }

        // Bail if the booking is not for an appointment
        if ( $args['type'] !== 'event'){
            return;
        }

        // Event tags
        $event_id = absint( $args['event']['id'] );
        $event_name = $args['event']['name'];
        $event_price = $args['event']['price'];
        $event_max_capacity = $args['event']['maxCapacity'];
        $event_status = $args['event']['status'];

        // Customer tags
        $customer_id = absint( $args['booking']['customer']['id'] );
        $customer_email = $args['booking']['customer']['email'];
        $customer_first_name = $args['booking']['customer']['firstName'];
        $customer_last_name = $args['booking']['customer']['lastName'];
        $customer_phone = $args['booking']['customer']['phone'];

        // Trigger user event booking
        automatorwp_trigger_event( array(
            'trigger'               => $this->trigger,
            'event_id'              => $event_id,
            'event_name'            => $event_name,
            'event_price'           => $event_price,
            'event_status'          => $event_status,
            'event_max_capacity'    => $event_max_capacity,
            'customer_id'           => $customer_id,
            'customer_first_name'   => $customer_first_name,
            'customer_last_name'    => $customer_last_name,
            'customer_email'        => $customer_email,
            'customer_phone'        => $customer_phone
        ) );
       
    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */


     public function anonymous_deserves_trigger( $deserves_trigger, $trigger, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['event_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( $trigger_options['event'] !== 'any' &&  absint( $trigger_options['event'] ) !== absint( $event['event_id'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_anonymous_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['event_id'] = ( isset( $event['event_id'] ) ? $event['event_id'] : '' );
        $log_meta['event_name'] = ( isset( $event['event_name'] ) ? $event['event_name'] : '' );
        $log_meta['event_price'] = ( isset( $event['event_price'] ) ? $event['event_price'] : '' );
        $log_meta['event_status'] = ( isset( $event['event_status'] ) ? $event['event_status'] : '' );
        $log_meta['event_max_capacity'] = ( isset( $event['event_max_capacity'] ) ? $event['event_max_capacity'] : '' );
        $log_meta['customer_id'] = ( isset( $event['customer_id'] ) ? $event['customer_id'] : '' );
        $log_meta['customer_first_name'] = ( isset( $event['customer_first_name'] ) ? $event['customer_first_name'] : '' );
        $log_meta['customer_last_name'] = ( isset( $event['customer_last_name'] ) ? $event['customer_last_name'] : '' );
        $log_meta['customer_email'] = ( isset( $event['customer_email'] ) ? $event['customer_email'] : '' );
        $log_meta['customer_phone'] = ( isset( $event['customer_phone'] ) ? $event['customer_phone'] : '' );

        return $log_meta;

    }


}


new AutomatorWP_Amelia_Anonymous_Books_Event();