<?php
/**
 * Add User Notification
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Actions\Add_User_Notification
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Add_User_Notification extends AutomatorWP_Integration_Action {

    public $integration = 'buddyboss';
    public $action = 'buddyboss_add_user_notification';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add a notification to the user', 'automatorwp-pro' ),
            'select_option'     => __( 'Add a <strong>notification</strong> to the user', 'automatorwp-pro' ),
            /* translators: %1$s: Notification. */
            'edit_label'        => sprintf( __( 'Add a %1$s to the user', 'automatorwp-pro' ), '{notification}' ),
            /* translators: %1$s: Notification. */
            'log_label'         => sprintf( __( 'Add a %1$s to the user', 'automatorwp-pro' ), '{notification}' ),
            'options'           => array(
                'notification' => array(
                    'default' => __( 'notification', 'automatorwp-pro' ),
                    'fields' => array(
                        'to' => array(
                            'name' => __( 'To:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that will receive the notification. Leave blank to send the notification to the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Notification\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                        'link' => array(
                            'name' => __( 'Link (Optional):', 'automatorwp-pro' ),
                            'desc' => __( 'URL link associated to the notification. Used for RSS feeds and if option clickable is enabled.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'clickable' => array(
                            'name' => __( 'Make notification clickable?', 'automatorwp-pro' ),
                            'desc' => __( 'Check this option to make the notification clickable. If link option is provided, the notification will redirect to that link, if not will redirect to the same page the user is.', 'automatorwp-pro' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch'
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Bail if notifications component is not active
        if ( ! bp_is_active( 'notifications' ) ) {
            $this->result = __( 'Notifications component is not active.', 'automatorwp-pro' );
            return;
        }

        // Setup the user who will receive the notification
        $to = $action_options['to'];

        if( empty( $to ) ) {
            $to = $user_id;
        }

        if( absint( $to ) === 0 ) {
            $this->result = __( 'Invalid user provided on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Add the notification to the user
        $notification_id = bp_notifications_add_notification(
            array(
                'user_id'           => $to, // User who receives the notification
                'item_id'           => $action->id,
                'secondary_item_id' => $user_id, // User who completed the automation
                'component_name'    => 'automatorwp',
                'component_action'  => 'automatorwp_buddyboss_notification',
                'date_notified'     => bp_core_current_time(),
                'is_new'            => 1,
                'allow_duplicate'   => true,
            )
        );

        // Store content and link as metadata
        if( $notification_id ) {
            bp_notifications_add_meta( $notification_id, 'automatorwp_buddyboss_content', $action_options['content'], true );
            bp_notifications_add_meta( $notification_id, 'automatorwp_buddyboss_link', $action_options['link'], true );
            bp_notifications_add_meta( $notification_id, 'automatorwp_buddyboss_clickable', $action_options['clickable'], true );
        }

        $this->result = __( 'User notified successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Format notifications
        add_filter( 'bp_notifications_get_notifications_for_user', array( $this, 'format_notifications' ), 10, 8 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Filters the notification content for notifications created by plugins.
     * If your plugin extends the {@link BP_Component} class, you should use the
     * 'notification_callback' parameter in your extended
     * {@link BP_Component::setup_globals()} method instead.
     *
     * @since BuddyPress 1.9.0
     * @since BuddyPress 2.6.0 Added $component_action_name, $component_name, $id as parameters.
     *
     * @param string $content               Component action. Deprecated. Do not do checks against this! Use
     *                                      the 6th parameter instead - $component_action_name.
     * @param int    $item_id               Notification item ID.
     * @param int    $secondary_item_id     Notification secondary item ID.
     * @param int    $action_item_count     Number of notifications with the same action.
     * @param string $format                Format of return. Either 'string' or 'object'.
     * @param string $component_action_name Canonical notification action.
     * @param string $component_name        Notification component ID.
     * @param int    $id                    Notification ID.
     *
     * @return string|array If $format is 'string', return a string of the notification content.
     *                      If $format is 'object', return an array formatted like:
     *                      array( 'text' => 'CONTENT', 'link' => 'LINK' )
     */
    public function format_notifications( $content, $item_id, $secondary_item_id, $action_item_count, $format, $component_action_name, $component_name, $id ) {

        if( $component_action_name === 'automatorwp_buddyboss_notification' ) {

            $options = array(
                'content' => bp_notifications_get_meta( $id, 'automatorwp_buddyboss_content', true ),
                'link' => bp_notifications_get_meta( $id, 'automatorwp_buddyboss_link', true ),
                'clickable' => bp_notifications_get_meta( $id, 'automatorwp_buddyboss_clickable', true ),
            );

            // Format the notification
            $formatted = automatorwp_buddyboss_format_notification( $options, $item_id, $format, $component_action_name );

            if( $formatted !== false ) {
                return $formatted;
            }

            // If content has not been stored as meta, try to generate the notification content
            $user_id = absint( $secondary_item_id );

            if( $user_id === 0 ) {
                $user_id = get_current_user_id();
            }

            $action = automatorwp_get_action_object( $item_id );

            // Bail if action can't be found
            if( ! $action ) {
                return $content;
            }

            $automation = automatorwp_get_action_automation( $action->id );

            // Get all action options to parse all replacements
            $action_options = automatorwp_get_action_stored_options( $action->id );

            foreach( $action_options as $option => $value ) {
                // Replace all tags by their replacements
                $action_options[$option] = automatorwp_parse_automation_tags( $automation->id, $user_id, $value );
            }

            // Format the notification
            $formatted = automatorwp_buddyboss_format_notification( $action_options, $item_id, $format, $component_action_name );

            if( $formatted !== false ) {
                return $formatted;
            }

        }

        return $content;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyBoss_Add_User_Notification();