<?php
/**
 * Profile Percentage
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Triggers\Profile_Percentage
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Profile_Percentage extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddyboss';
    public $trigger = 'buddyboss_profile_percentage';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes his profile with a percentage greater than, less than or equal to a specific percentage', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes his profile with a percentage <strong>greater than, less than or equal</strong> to a specific percentage', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Percentage. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes his profile with a percentage %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{condition}', '{percentage}', '{times}' ),
            /* translators: %1$s: Condition. %2$s: Percentage. */
            'log_label'         => sprintf( __( 'User completes his profile with a percentage %1$s %2$s', 'automatorwp-pro' ), '{condition}', '{percentage}' ),
            'filter'            => 'xprofile_pc_user_progress',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'condition' => automatorwp_utilities_condition_option(),
                'percentage' => array(
                    'from' => 'percentage',
                    'fields' => array(
                        'percentage' => array(
                            'name' => __( 'Percentage:', 'automatorwp-woocommerce' ),
                            'desc' => __( 'The percentage required.', 'automatorwp-woocommerce' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                                'max' => '100',
                                'step' => '1',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array $progress_details
     *
     * @return array
     */
    public function listener( $progress_details ) {

        // Bail if no fields configured
        if( $progress_details['total_fields'] === 0 ) {
            return $progress_details;
        }

        $user_id = get_current_user_id();
        $percentage = round( ( $progress_details['completed_fields'] * 100 ) / $progress_details['total_fields'] );

        // Trigger the member type update
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'percentage'   => $percentage,
        ) );

        return $progress_details;

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if order total is not received
        if( ! isset( $event['percentage'] ) ) {
            return false;
        }

        $percentage = (float) $event['percentage'];
        $required_percentage =  (float) $trigger_options['percentage'];

        // Don't deserve if profile percentage completed doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $percentage, $required_percentage, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_BuddyBoss_Profile_Percentage();