<?php
/**
 * Register Specific Field Value
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Triggers\Register_Specific_Field_Value
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_Register_Specific_Field_Value extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddypress';
    public $trigger = 'buddypress_register_specific_field_value';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User registers with a specific field value', 'automatorwp-pro' ),
            'select_option'     => __( 'User registers with a <strong>specific field value</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Field value. %2$s: Field ID or name. %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User registers with the %1$s on the field %2$s %3$s time(s)', 'automatorwp-pro' ), '{field_value}', '{field_name}', '{times}' ),
            /* translators: %1$s: Field value. %2$s: Field ID or name. */
            'log_label'         => sprintf( __( 'User registers with the %1$s on the field %2$s', 'automatorwp-pro' ), '{field_value}', '{field_name}' ),
            'action'            => 'bp_core_signup_user',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 5,
            'options'           => array(
                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'field value', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Field value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'field_name' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'field_name',
                    'name'              => __( 'Field:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'field name', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddypress_get_profile_fields',
                    'options_cb'        => 'automatorwp_buddypress_options_cb_profile_field',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param bool|WP_Error   $user_id       True on success, WP_Error on failure.
     * @param string          $user_login    Login name requested by the user.
     * @param string          $user_password Password requested by the user.
     * @param string          $user_email    Email address requested by the user.
     * @param array           $usermeta      Miscellaneous metadata about the user (blog-specific
     *                                       signup data, xprofile data, etc).
     */
    public function listener( $user_id, $user_login, $user_password, $user_email, $usermeta ) {

        if( ! function_exists( 'xprofile_get_field' ) ) {
            return;
        }

        // Loop all updated fields
        foreach( $usermeta as $field_id => $field_value ) {

            $field_id = str_replace( 'field_', '', $field_id );

            // Skip non profile fields
            if( ! is_numeric( $field_id ) ) {
                continue;
            }

            $field = xprofile_get_field( $field_id );

            // Trigger the register with field value
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'field_id'      => $field_id,
                'field_name'    => $field->name,
                'field_value'   => $field_value,
            ) );

        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post, field name and value are not received
        if( ! isset( $event['field_id'] ) && ! isset( $event['field_name'] ) && ! isset( $event['field_value'] ) ) {
            return false;
        }

        $required_field_name = $trigger_options['field_name'];

        if( is_numeric( $required_field_name ) ) {
            // Don't deserve if field ID doesn't match with the trigger option
            if( absint( $event['field_id'] ) !== absint( $required_field_name ) ) {
                return false;
            }
        } else {
            // Don't deserve if field name doesn't match with the trigger option
            if( $event['field_name'] !== $required_field_name ) {
                return false;
            }
        }

        // Check if field value matches the required one (with support for arrays)
        if( is_array( $event['field_value'] ) ) {

            if( ! in_array( $trigger_options['field_value'], $event['field_value'] ) ) {
                return false;
            }

        } else {

            if( $event['field_value'] !== $trigger_options['field_value'] ) {
                return false;
            }

        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['field_id'] = ( isset( $event['field_id'] ) ? $event['field_id'] : '' );
        $log_meta['field_name'] = ( isset( $event['field_name'] ) ? $event['field_name'] : '' );
        $log_meta['field_value'] = ( isset( $event['field_value'] ) ? $event['field_value'] : '' );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['field_id'] = array(
            'name' => __( 'Field ID', 'automatorwp-pro' ),
            'desc' => __( 'The field ID handled on this profile update.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_name'] = array(
            'name' => __( 'Field name', 'automatorwp-pro' ),
            'desc' => __( 'The field name handled on this profile update.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_value'] = array(
            'name' => __( 'Field value', 'automatorwp-pro' ),
            'desc' => __( 'The field value handled on this profile update.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyPress_Register_Specific_Field_Value();