<?php
/**
 * Add new pretty link
 *
 * @package     AutomatorWP\Integrations\pretty-link\Actions\create-link
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Pretty_Link_Add_New_Link extends AutomatorWP_Integration_Action {

    public $integration = 'pretty_link';
    public $action = 'pretty_link_add_new_link';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create a new pretty link', 'automatorwp-pro' ),
            'select_option'     => __( 'Create a <strong>new pretty link</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: List. */
            'edit_label'        => sprintf( __( 'Create a new %1$s', 'automatorwp-pro' ), '{link}' ),
            /* translators: %1$s: List. */
            'log_label'         => sprintf( __( 'Create a new %1$s', 'automatorwp-pro' ), '{link}' ),
            'options'           => array(
                'link' => array(
                    'from' => 'link',
                    'default' => __('link', 'automatorwp-pro'),
                    'fields' => array(
                        'redirection' => array(
                            'name' => __( 'Redirection: ', 'automatorwp-pro' ),
                            'type' => 'select',
                            'option_none' => false,
                            'options_cb' => 'automatorwp_pretty_link_get_all_redirection_options_cb',
                            'required'  => true
                        ),
                        'name_link' => array(
                            'name' => __( 'Name: ', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                            'required'  => true
                        ),
                        'original_url' => array(
                            'name' => __( 'Target URL: ', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                            'required'  => true
                        ),
                        'new_url' => array(
                            'name' => __( 'pretty Link: ', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                            'desc' => __('A random pretty link will be generated if this field is empty.', 'automatorwp-pro')
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';
        
        $newUrl = $action_options['new_url'];

        if ( empty($action_options['new_url']) ) {
            $newUrl = automatorwp_pretty_link_generate_random_slug();
        }

        if ( empty($action_options['name_link']) ) {
            $this->result = __('Name cannot be empty!', 'automatorwp-pro');
        }

        if ( empty($action_options['original_url']) ) {
            $this->result = __('Target url cannot be empty!', 'automatorwp-pro');
        }

        $values = array(
            'slug' => $newUrl,
            'name' => $action_options['name_link'],
            'url' => $action_options['original_url'],
            'redirect_type' => $action_options['redirection'],
            'track_me' => 1
        );

        $user = get_user_by( 'ID', $user_id );
        $email = $user->user_email;
        
        $links = new PrliLink();

        $link_id = $links->create( $values );

        $this->result = __('Link created', 'automatorwp-pro');
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Pretty_Link_Add_New_Link();