<?php
/**
 * Content Filters
 *
 * @package     AutomatorWP\Restrict_Content\Content_Filters
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Check if user has access to queried post
 *
 * @since 1.0.0
 */
function automatorwp_restrict_content_check_access() {

    global $post;

    $post_types = automatorwp_restrict_content_post_types_slugs();

    if ( $post && in_array( $post->post_type, $post_types ) && is_singular( $post_types )       // Is an allowed post type
        && automatorwp_restrict_content_is_post_access_restricted( $post->ID )                         // Is access restricted
        && ! automatorwp_restrict_content_is_user_granted_to_post( $post->ID ) ) {                        // User is not granted

        $restriction = automatorwp_restrict_content_get_post_meta( $post->ID, 'restriction' );

        // Redirect Page
        if( $restriction === 'redirect_page' ) {
            $redirect_page = absint( automatorwp_restrict_content_get_post_meta( $post->ID, 'redirect_page' ) );

            if( $redirect_page <= 0 ) {
                // WordPress error message
                wp_die( __( 'Access to this page is restricted!', 'automatorwp-pro' ), 'Error' );
            } else {
                $redirect_url = get_permalink( $redirect_page );

                $redirect_url = add_query_arg( 'post', $post->ID, $redirect_url );

                // Redirect to page URL
                wp_redirect( $redirect_url );
                exit;
            }
        }

        // Redirect URL
        if( $restriction === 'redirect_url' ) {
            $redirect_url = automatorwp_restrict_content_get_post_meta( $post->ID, 'redirect_url' );

            if( empty( $redirect_url ) ) {
                // WordPress error message
                wp_die( __( 'Access to this page is restricted!', 'automatorwp-pro' ), 'Error' );
            }

            // Redirect to custom URL
            wp_redirect( $redirect_url );
            exit;
        }

    }

}
add_action( 'template_redirect', 'automatorwp_restrict_content_check_access' );

/**
 * Pass queried post restrictions (content, links and images)
 *
 * @since 1.0.0
 *
 * @param string $content
 *
 * @return string
 */
function automatorwp_restrict_content_pass_restrictions( $content ) {

    global $post;

    $post_types = automatorwp_restrict_content_post_types_slugs();

    if ( $post && in_array( $post->post_type, $post_types ) && is_singular( $post_types )       // Is an allowed post type
        && is_main_query() && ! post_password_required()                                        // Is main query and not pass protected
        && automatorwp_restrict_content_is_post_restricted( $post->ID )                                // Is restricted
        && ! automatorwp_restrict_content_is_user_granted_to_post( $post->ID ) ) {                        // User is not granted

        // Restrict content
        if( automatorwp_restrict_content_is_post_content_restricted( $post->ID ) ) {

            // Replace the post content
            $content = automatorwp_restrict_content_get_post_meta( $post->ID, 'content_replacement' );

        }

        $content = automatorwp_restrict_content_apply_content_filters( $content );

    }

    return $content;
}
add_filter( 'the_content', 'automatorwp_restrict_content_pass_restrictions', 99999 );

/**
 * Apply active WordPress filters to the given content
 *
 * @since 1.0.2
 *
 * @param string $content
 *
 * @return string
 */
function automatorwp_restrict_content_apply_content_filters( $content = '' ) {

    /**
     * Filter available to exclude the filters to pass to the content
     *
     * @since 1.2.2
     *
     * @param array $excluded_filters
     *
     * @return array
     */
    $excluded_filters = apply_filters( 'automatorwp_restrict_content_excluded_content_filters', array() );

    if ( has_filter( 'the_content', 'wptexturize' ) && ! in_array( 'wptexturize', $excluded_filters ) ) {
        $content = wptexturize( $content );
    }

    if ( has_filter( 'the_content', 'convert_smilies' ) && ! in_array( 'convert_smilies', $excluded_filters ) ) {
        $content = convert_smilies( $content );
    }

    if ( has_filter( 'the_content', 'convert_chars' ) && ! in_array( 'convert_chars', $excluded_filters ) ) {
        $content = convert_chars( $content );
    }

    if ( has_filter( 'the_content', 'wpautop' ) && ! in_array( 'wpautop', $excluded_filters ) ) {
        $content = wpautop( $content );
    }

    if ( has_filter( 'the_content', 'shortcode_unautop' ) && ! in_array( 'shortcode_unautop', $excluded_filters ) ) {
        $content = shortcode_unautop( $content );
    }

    if ( has_filter( 'the_content', 'prepend_attachment' ) && ! in_array( 'prepend_attachment', $excluded_filters ) ) {
        $content = prepend_attachment( $content );
    }

    if ( has_filter( 'the_content', 'capital_P_dangit' ) && ! in_array( 'capital_P_dangit', $excluded_filters ) ) {
        $content = capital_P_dangit( $content );
    }

    if ( ! in_array( 'do_shortcode', $excluded_filters ) ) {
        $content = do_shortcode( $content );
    }

    if ( ! in_array( 'do_blocks', $excluded_filters ) ) {
        $content = do_blocks( $content );
    }

    return $content;

}