<?php
/**
 * Complete Lesson Category
 *
 * @package     AutomatorWP\Integrations\Tutor_LMS\Triggers\Complete_Lesson_Category
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Tutor_LMS_Complete_Lesson_Category extends AutomatorWP_Integration_Trigger {

    public $integration = 'tutor';
    public $trigger = 'tutor_complete_lesson_category';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a lesson of a category', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes <strong>a lesson</strong> of a category', 'automatorwp-pro' ),
            /* translators: %1$s: Category. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes a lesson of %1$s category %2$s time(s)', 'automatorwp-pro' ), '{term}', '{times}' ),
            /* translators: %1$s: Category. */
            'log_label'         => sprintf( __( 'User completes a lesson of %1$s category', 'automatorwp-pro' ), '{term}' ),
            'action'            => 'tutor_lesson_completed_after',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'taxonomy' => 'course-category',
                )),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $lesson_id
     */
    public function listener( $lesson_id ) {

        $user_id = get_current_user_id();

        $course_id = tutor_utils()->get_course_id_by_lesson( $lesson_id );

        $terms_ids = automatorwp_get_term_ids( $course_id, 'course-category' );

        // Bail if post isn't assigned to any category
        if( empty( $terms_ids ) ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'post_id'   => $lesson_id,
            'course_id' => $course_id,
            'terms_ids' => $terms_ids,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Tutor_LMS_Complete_Lesson_Category();