<?php
/**
 * Rest API
 *
 * @package     AutomatorWP\Webhooks\Rest_API
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Register receive data from webhooks endpoints on the WordPress Rest API
 *
 * @since 1.0.0
 */
function automatorwp_webhooks_rest_api_init() {

    $triggers = automatorwp_webhooks_get_receive_triggers();

    ct_setup_table( 'automatorwp_triggers' );

    foreach( $triggers as $trigger ) {

        $slug = ct_get_object_meta( $trigger->id, 'slug', true );

        if ( !empty ( $slug ) ){

            register_rest_route( 'automatorwp/webhooks', $slug, array(
                'methods' => array( 'GET', 'POST', 'PUT' ),
                'callback' => 'automatorwp_webhooks_rest_api_cb',
                'permission_callback' => '__return_true',
            ) );
            
        }
        

    }

    ct_reset_setup_table();

}
add_action( 'rest_api_init', 'automatorwp_webhooks_rest_api_init');

/**
 * Callback used to handle webhooks received requests
 *
 * @since 1.0.0
 *
 * @param WP_REST_Request $data
 *
 * @return WP_REST_Response
 */
function automatorwp_webhooks_rest_api_cb( $data ) {

    $params = $data->get_params();
    $slug = str_replace( '/automatorwp/webhooks/', '', $data->get_route() );
    $trigger = automatorwp_webhooks_get_trigger_by_slug( $slug );

    if( ! $trigger ) {
        return new WP_REST_Response( array( 'code' => 'automatorwp_rest_no_route', 'message' => 'No webhook was found matching the URL.', 'success' => false ), 404 );
    }

    /**
     * Available filter to override the parameters received
     *
     * @since 1.0.4
     *
     * @param array             $params     Request parameters
     * @param stdClass          $trigger    Trigger assigned to this request
     * @param WP_REST_Request   $data       Request object
     */
    $params = apply_filters( 'automatorwp_webhooks_received_parameters', $params, $trigger, $data );

    // Pull all array values
    $params = automatorwp_utilities_pull_array_values( $params );

    // Get the trigger stored options
    $trigger_options = automatorwp_get_trigger_stored_options( $trigger->id );

    $fields = array();
    $user_id = get_current_user_id();

    // Sanitize fields
    if( isset( $trigger_options['fields'] ) && is_array( $trigger_options['fields'] ) ) {

        foreach( $trigger_options['fields'] as $field ) {

            // Shorthand
            $key = ( isset( $field['key'] ) ? $field['key'] : '' );
            $type = ( isset( $field['type'] ) ? $field['type'] : '' );
            $required = ( isset( $field['required'] ) ? (bool) $field['required'] : false );
            $required_value = ( isset( $field['required_value'] ) ? $field['required_value'] : '' );

            if( isset( $params[$key] ) ) {

                if( ! empty( $required_value ) && $params[$key] !== $required_value ) {
                    return new WP_REST_Response( array( 'code' => 'automatorwp_rest_incorrect_parameter_value', 'message' => 'Incorrect value for parameter \'' . $key . '\'.', 'success' => false ), 403 );
                }

                switch( $type ) {
                    case 'integer':
                    case 'user_id':
                        $fields[$key] = (int) $params[$key];
                        break;
                    case 'float':
                        $fields[$key] = (float) $params[$key];
                        break;
                    case 'text':
                    case 'array':
                        $fields[$key] = is_array( $params[$key] ) ? array_map( 'wp_kses_post', $params[$key] ) : wp_kses_post( $params[$key] );
                        break;
                    case 'user_email':
                    default:
                        $fields[$key] = sanitize_text_field( $params[$key] );
                    break;
                }


                if( $type === 'user_id' ) {

                    // Set the user ID from this field
                    $user_id = $fields[$key];

                } else if( $type === 'user_email' ) {

                    // Set the user ID from this email
                    $user = get_user_by_email( $fields[$key] );

                    if( $user ) {
                        $user_id = $user->ID;
                    }

                }

            } else if( $required ) {
                return new WP_REST_Response( array( 'code' => 'automatorwp_rest_missing_parameter', 'message' => 'Missing required parameter \'' . $key . '\'.', 'success' => false ), 403 );
            }
        }

    }

    // Sanitize the rest of fields not defined as text
    foreach( $params as $key => $value ) {

        if( ! isset( $fields[$key] ) ) {
            $fields[$key] = sanitize_text_field( $value );
        }

    }

    // Insert the webhook log
    automatorwp_webhooks_insert_log( array(
        'url'               => str_replace( '//automatorwp', '/automatorwp', get_rest_url() . $data->get_route() ),
        'method'            => $data->get_method(),
        'data'              => $fields,
        'logged_in_user_id' => get_current_user_id(),
        'user_id'           => $user_id,
    ) );

    /**
     * Action to meet when a webhook has been requested
     *
     * @since 1.0.0
     *
     * @param string            $slug       The slug matched
     * @param stdClass          $trigger    The trigger object
     * @param int               $user_id    The user ID
     * @param array             $fields     Fields received
     * @param WP_REST_Request   $data       WordPress Rest API request object
     */
    do_action( 'automatorwp_webhook_received', $slug, $trigger, $user_id, $fields, $data );

    $response = array( 'success' => true );
	
    /**
     * To customize webhook response
     *
     * @since 1.0.0
     *
     * @param array             $response   Response
     * @param string            $slug       The slug matched
     * @param stdClass          $trigger    The trigger object
     * @param int               $user_id    The user ID
     * @param array             $fields     Fields received
     * 
     * @return array
     */
	$response = apply_filters( 'automatorwp_webhooks_custom_response', $response, $slug, $trigger, $user_id, $fields, $data );

	return new WP_REST_Response( $response , 200 );

}