<?php
/**
 * Get Comment Activity Group
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Triggers\Get_Comment_Activity_Group
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Get_Comment_Activity_Group extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddyboss';
    public $trigger = 'buddyboss_get_comment_activity_group';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets reply to an activity stream message in group', 'automatorwp-pro' ),
            'select_option'     => __( 'User gets <strong>reply</strong> to an activity stream message in <strong>group</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets reply to an activity stream message in %1$s %2$s time(s)', 'automatorwp-pro' ), '{group}', '{times}' ),
            /* translators: %1$s: Group. */
            'log_label'         => sprintf( __( 'User gets reply to an activity stream message in %1$s', 'automatorwp-pro' ), '{group}' ),
            'action'            => 'bp_activity_comment_posted',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'group' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group',
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any group', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddyboss_get_groups',
                    'options_cb'        => 'automatorwp_buddyboss_options_cb_group',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                array(
                    'activity_comment_content' => array(
                        'label'     => __( 'Comment content', 'automatorwp-pro' ),
                        'type'      => 'text',
                        'preview'   => 'The comment content',
                    )
                ),
                automatorwp_buddyboss_get_activity_tags(),
                automatorwp_buddyboss_get_group_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $comment_id
     * @param array $args
     * @param BP_Activity_Activity $activity
     */
    public function listener( $comment_id, $args, $activity ) {

        $user_id = bp_loggedin_user_id();

        // Bail if not comment in group
        if ( $activity->component !== 'groups' ) {
            return;
        }

        $replied_comment = new BP_Activity_Activity( $args['parent_id'] );

        // Bail if reply is from the same user
        if ( absint( $replied_comment->user_id ) === absint( $user_id ) ) {
            return;
        }

        // Trigger the comment on an activity
        automatorwp_trigger_event( array(
            'trigger'                   => $this->trigger,
            'user_id'                   => absint( $replied_comment->user_id ),
            'group_id'                  => $activity->item_id,
            'activity_id'               => $activity->id,
            'activity_content'          => $activity->content,
            'activity_author_id'        => $activity->user_id,
            'activity_comment_id'       => $comment_id,
            'activity_comment_content'  => $args['content'],
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if group is not received
        if( ! isset( $event['group_id'] ) ) {
            return false;
        }

        // Don't deserve if group doesn't match with the trigger option
        if( $trigger_options['group'] !== 'any' && absint( $event['group_id'] ) !== absint( $trigger_options['group'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['group_id'] = ( isset( $event['group_id'] ) ? $event['group_id'] : '' );
        $log_meta['activity_id'] = ( isset( $event['activity_id'] ) ? $event['activity_id'] : '' );
        $log_meta['activity_content'] = ( isset( $event['activity_content'] ) ? $event['activity_content'] : '' );
        $log_meta['activity_author_id'] = ( isset( $event['activity_author_id'] ) ? $event['activity_author_id'] : '' );
        $log_meta['activity_comment_content'] = ( isset( $event['activity_comment_content'] ) ? $event['activity_comment_content'] : '' );
        $log_meta['activity_media_urls'] = automatorwp_buddyboss_get_activity_media_urls( $event['activity_id'] );
        $log_meta['activity_video_urls'] = automatorwp_buddyboss_get_activity_video_urls( $event['activity_id'] );
        $log_meta['activity_document_urls'] = automatorwp_buddyboss_get_activity_document_urls( $event['activity_id'] );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['activity_comment_content'] = array(
            'name' => __( 'Comment Content', 'automatorwp-pro' ),
            'desc' => __( 'The comment content.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyBoss_Get_Comment_Activity_Group();