<?php
/**
 * Submit Quiz
 *
 * @package     AutomatorWP\Integrations\Forminator\Triggers\Submit_Quiz
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Forminator_Submit_Quiz extends AutomatorWP_Integration_Trigger {

    public $integration = 'forminator';
    public $trigger = 'forminator_submit_quiz';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User attempts, passes or fails a quiz', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>attempts, passes or fails</strong> a quiz', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (attempts, passes or fails). %2$s: Post title. %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{operation}', '{post}', '{times}' ),
            /* translators: %1$s: Operation (attempts, passes or fails). %2$s: Post title. */
            'log_label'         => sprintf( __( 'User %1$s %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'action'            => 'forminator_quizzes_submit_before_set_fields',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'attempt'     => __( 'attempts', 'automatorwp-pro' ),
                                'pass'    => __( 'passes', 'automatorwp-pro' ),
                                'fail'    => __( 'fails', 'automatorwp-pro' ),
                            ),
                            'default' => 'attempt'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Quiz:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any quiz', 'automatorwp-pro' ),
                    'post_type' => 'forminator_quizzes'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                array(
                    'form_field:FIELD_NAME' => array(
                        'label'     => __( 'Form field value', 'automatorwp-pro' ),
                        'type'      => 'text',
                        'preview'   => __( 'Form field value, replace "FIELD_NAME" by the field name', 'automatorwp-pro' ),
                    ),
                ),
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param Forminator_Form_Entry_Model $entry      - the entry model
     * @param int                         $form_id    - the form id
     * @param array                       $field_data - the entry data
     */
    public function listener( $entry, $form_id, $field_data ) {

        $user_id = get_current_user_id();

        // Bail if user is not logged in
        if( $user_id === 0 ) {
            return;
        }

        $answers = $field_data;

        if( isset( $field_data[0] )
            && isset( $field_data[0]['value'] )
            && isset( $field_data[0]['value']['answers'] ) ) {
            $answers = $field_data[0]['value']['answers'];
        }

        $form_fields = automatorwp_forminator_get_form_fields_values( $answers, 'question', 'answer' );

        // Trigger submit form event
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $form_id,
            'field_data'    => $field_data,
            'form_fields'   => $form_fields,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Shorthand
        $operation = $trigger_options['operation'];

        // Don't deserve if post and field data are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['field_data'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        // Loop all answers to meet if all of them are correct
        $all_correct = true;

        foreach( $event['field_data'] as $data ) {
            if( isset( $data['isCorrect'] ) && $data['isCorrect'] === false ) {
                $all_correct = false;
                break;
            }
        }

        switch ( $operation ) {
            case 'attempt':
                // Attempt doesn't requires any extra check
                break;
            case 'pass':
                // Don't deserve if user hasn't completed correctly all quizzes
                if( ! $all_correct ) {
                    return false;
                }
                break;
            case 'fail':
                // Don't deserve if user has completed correctly all quizzes
                if( $all_correct ) {
                    return false;
                }
                break;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['form_fields'] = ( isset( $event['form_fields'] ) ? $event['form_fields'] : array() );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['form_fields'] = array(
            'name' => __( 'Fields Submitted', 'automatorwp-pro' ),
            'desc' => __( 'Information about the fields values sent on this form submission.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_Forminator_Submit_Quiz();