<?php
/**
 * User Points
 *
 * @package     AutomatorWP\Integrations\GamiPress\Actions\User_Points
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_User_Points extends AutomatorWP_Integration_Action {

    public $integration = 'gamipress';
    public $action = 'gamipress_user_points';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Award or deduct points to user', 'automatorwp-pro' ),
            'select_option'     => __( 'Award or deduct <strong>points</strong> to user', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (award or deduct). %2$s: Points amount. %3$s: Post title. %4$s: User. */
            'edit_label'        => sprintf( __( '%1$s %2$s %3$s to %4$s', 'automatorwp-pro' ), '{operation}', '{points}', '{points_type}', '{user}' ),
            /* translators: %1$s: Operation (award or deduct). %2$s: Points amount. %3$s: Post title. %4$s: User. */
            'log_label'         => sprintf( __( '%1$s %2$s %3$s to %4$s', 'automatorwp-pro' ), '{operation}', '{points}', '{points_type}', '{user}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'award'     => __( 'Award', 'automatorwp-pro' ),
                                'deduct'    => __( 'Deduct', 'automatorwp-pro' ),
                            ),
                            'default' => 'award'
                        ),
                    )
                ),
                'points' => array(
                    'from' => 'points',
                    'fields' => array(
                        'points' => array(
                            'name' => __( 'Points amount:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '1'
                        ),
                        'earning_text' => array(
                            'name' => __( 'User earning text:', 'automatorwp-pro' ),
                            'desc' => __( 'Enter the text for the user earning entry. Leave blank to do not register a new user earning entry.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'points_type' => array(
                    'from' => 'points_type',
                    'default' => __( 'Choose a points type', 'automatorwp-pro' ),
                    'fields' => array(
                        'points_type' => array(
                            'name' => __( 'Points Type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'option_none' => false,
                            'options_cb' => 'automatorwp_gamipress_points_types_options_cb'
                        )
                    )
                ),
                'user' => array(
                    'from' => 'user',
                    'default' => __( 'user', 'automatorwp-pro' ),
                    'fields' => array(
                        'user' => array(
                            'name' => __( 'User ID:', 'automatorwp-pro' ),
                            'desc' => __( 'User ID that will receive this points. Leave blank to award the points to the user that completes the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $points = absint( $action_options['points'] );
        $points_type = $action_options['points_type'];
        $earning_text = $action_options['earning_text'];
        $user_id_to_award = absint( $action_options['user'] );

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'award';
        }

        if( $user_id_to_award === 0 ) {
            $user_id_to_award = $user_id;
        }

        $user = get_userdata( $user_id_to_award );

        // Bail if user does not exists
        if( ! $user ) {
            return;
        }

        // Bail if no points to award
        if( $points === 0 ) {
            return;
        }

        // Bail if post is not a points type
        if( ! gamipress_get_points_type( $points_type ) ) {
            return;
        }

        switch ( $operation ) {
            case 'award':
                // Award the points
                gamipress_award_points_to_user( $user_id_to_award, $points, $points_type );
                break;
            case 'deduct':
                // Deduct the points
                gamipress_deduct_points_to_user( $user_id_to_award, $points, $points_type );
                break;
        }

        if( ! empty( $earning_text ) ) {

            // Make the points amount negative
            if( $operation === 'deduct' ) {
                $points *= -1;
            }

            // Insert the custom user earning
            gamipress_insert_user_earning( $user_id, array(
                'title'	        => $earning_text,
                'user_id'	    => $user_id_to_award,
                'post_id'	    => gamipress_get_points_type_id( $points_type ),
                'post_type' 	=> 'points-type',
                'points'	    => $points,
                'points_type'	=> $points_type,
                'date'	        => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
            ) );

        }

    }

}

new AutomatorWP_GamiPress_User_Points();