<?php
/**
 * User Achievement
 *
 * @package     AutomatorWP\Integrations\GamiPress\Filters\User_Achievement
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_User_Achievement_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'gamipress';
    public $filter = 'gamipress_user_achievement';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User achievement', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>achievement</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Achievement. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{post}'  ),
            /* translators: %1$s: Condition. %2$s: Achievement. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{post}' ),
            'options'           => array(
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'earned'             => __( 'has earned', 'automatorwp-pro' ),
                                'not_earned'         => __( 'has not earned', 'automatorwp-pro' ),
                            ),
                            'default' => 'earned'
                        )
                    )
                ),
                'post' => automatorwp_gamipress_utilities_post_option( array(
                    'name' => __( 'Achievement:', 'automatorwp-pro' ),
                    'option_default'        => __( 'any achievement', 'automatorwp-pro' ),
                    'option_none_label'     => __( 'any achievement', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Achievement ID', 'automatorwp-pro' ),
                    'post_type_cb'          => 'gamipress_get_achievement_types_slugs'
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $condition = $filter_options['condition'];
        $post_id = $filter_options['post'];

        // Any achievement
        if( $post_id === 'any' ) {

            $earned_times = gamipress_get_earnings_count( array(
                'user_id'   => absint( $user_id ),
                'post_type' => gamipress_get_achievement_types_slugs(),
            ) );

            switch ( $condition ) {
                case 'earned':
                    // Don't deserve if user has not earned any achievement
                    if( $earned_times === 0 ) {
                        $this->result = __( 'Filter not passed. User has not earned any achievement.', 'automatorwp-pro' );
                        return false;
                    } else {
                        $this->result = sprintf( __( 'Filter passed. User has earned %1$s achievement(s).', 'automatorwp-pro' ),
                            $earned_times
                        );
                    }
                    break;
                case 'not_earned':
                    // Don't deserve if user has earned any achievement
                    if( $earned_times > 0 ) {
                        $this->result = sprintf( __( 'Filter not passed. User has earned %1$s achievement(s).', 'automatorwp-pro' ),
                            $earned_times
                        );
                        return false;
                    } else {
                        $this->result = __( 'Filter passed. User has not earned any achievement.', 'automatorwp-pro' );
                    }
                    break;
            }

            return $deserves_filter;

        }

        // Single achievement
        $post_id = absint( $post_id );

        // Bail if post id not provided
        if( $post_id === 0 ) {
            $this->result = __( 'Filter not passed. Achievement option is not configured.', 'automatorwp-pro' );
            return false;
        }

        $post = get_post( $post_id );

        // Bail if post doesn't exists
        if( ! $post ) {
            $this->result = __( 'Filter not passed. Achievement configured does not exists.', 'automatorwp-pro' );
            return false;
        }

        // Bail if not is an achievement
        if( ! in_array( $post->post_type, gamipress_get_achievement_types_slugs() ) ) {
            $this->result = __( 'Filter not passed. Achievement configured is not a GamiPress achievement.', 'automatorwp-pro' );
            return false;
        }

        $earned_times = gamipress_get_earnings_count( array(
            'user_id'   => absint( $user_id ),
            'post_id'   => absint( $post_id ),
        ) );

        switch ( $condition ) {
            case 'earned':
                // Don't deserve if user has not earned the achievement
                if( $earned_times === 0 ) {
                    $this->result = sprintf( __( 'Filter not passed. User has not earned the achievement %1$s.', 'automatorwp-pro' ),
                        $post->post_title
                    );
                    return false;
                } else {
                    $this->result = sprintf( __( 'Filter passed. User has earned the achievement %1$s.', 'automatorwp-pro' ),
                        $post->post_title
                    );
                }
                break;
            case 'not_earned':
                // Don't deserve if user has earned the achievement
                if( $earned_times > 0 ) {
                    $this->result = sprintf( __( 'Filter not passed. User has earned the achievement %1$s.', 'automatorwp-pro' ),
                        $post->post_title
                    );
                    return false;
                } else {
                    $this->result = sprintf( __( 'Filter passed. User has not earned the achievement %1$s.', 'automatorwp-pro' ),
                        $post->post_title
                    );
                }
                break;
        }

        return $deserves_filter;

    }

}

new AutomatorWP_GamiPress_User_Achievement_Filter();