<?php
/**
 * User Course
 *
 * @package     AutomatorWP\Integrations\MasterStudy_LMS\Actions\User_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_MasterStudy_LMS_User_Course extends AutomatorWP_Integration_Action {

    public $integration = 'masterstudy_lms';
    public $action = 'masterstudy_lms_user_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Enroll or unroll user from a course', 'automatorwp-pro' ),
            'select_option'     => __( 'Enroll or unroll user from <strong>a course</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'enroll'     => __( 'Enroll', 'automatorwp-pro' ),
                                'unroll'    => __( 'Unroll', 'automatorwp-pro' ),
                            ),
                            'default' => 'enroll'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all courses', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type'         => 'stm-courses',
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unroll' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{post}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $course_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'enroll';
        }

        $courses = array();

        // Check specific course
        if( $course_id !== 'any' ) {

            $course = get_post( $course_id );

            // Bail if course doesn't exists
            if( ! $course ) {
                return;
            }

            $courses = array( $course_id );

        }

        switch ( $operation ) {
            case 'enroll':
                // If enrolling to all courses, get all courses
                if( $course_id === 'any' ) {

                    $query = new WP_Query( array(
                        'post_type'		=> 'stm-courses',
                        'post_status'	=> 'publish',
                        'fields'        => 'ids',
                        'nopaging'      => true,
                    ) );

                    $courses = $query->get_posts();
                }

                // Enroll user in courses
                foreach( $courses as $course_id ) {
                    STM_LMS_Course::add_user_course( $course_id, $user_id, 0, 0 );
                    STM_LMS_Course::add_student( $course_id );
                }
                break;
            case 'unroll':
                // If unrolling from all courses, get all user courses
                if( $course_id === 'any' ) {

                    $user_courses = stm_lms_get_user_courses( $user_id, '', '', array( 'course_id' ) );

                    if( is_array( $user_courses ) ) {

                        // Loop all user courses to get their IDs
                        foreach( $user_courses as $user_course ) {
                            $courses[] = $user_course['course_id'];
                        }

                    }

                }

                // Unroll user from courses
                foreach( $courses as $course_id ) {
                    stm_lms_get_delete_user_course( $user_id, $course_id );
                    STM_LMS_Course::remove_student( $course_id );
                }
                break;
        }

    }

}

new AutomatorWP_MasterStudy_LMS_User_Course();