<?php
/**
 * Commission
 *
 * @package     AutomatorWP\Integrations\SliceWP\Triggers\Commission
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_SliceWP_Commission extends AutomatorWP_Integration_Trigger {

    public $integration = 'slicewp';
    public $trigger = 'slicewp_commission';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a commission of type paid or rejected', 'automatorwp-pro' ),
            'select_option'     => __( 'User earns a commission of type <strong>paid or rejected</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Commission type. %2$s: Operation (paid or rejected). %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a commission %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{type}', '{operation}', '{times}' ),
            /* translators: %1$s: Commission type. %2$s: Operation (paid or rejected). */
            'log_label'         => sprintf( __( 'User gets a commission %1$s %2$s', 'automatorwp-pro' ), '{type}', '{operation}' ),
            'action'            => 'slicewp_update_commission',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'type' => array(
                    'from' => 'type',
                    'fields' => array(
                        'type' => array(
                            'name' => __( 'Commission type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => array( $this, 'commission_types_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'paid'      => __( 'paid', 'automatorwp-pro' ),
                                'rejected'  => __( 'rejected', 'automatorwp-pro' ),
                            ),
                            'default' => 'paid'
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_slicewp_commission_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Get commission types options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function commission_types_options_cb() {

        $options = array(
            'any' => __( 'of any type', 'automatorwp-pro' )
        );

        foreach( slicewp_get_commission_types() as $type_id => $type ) {
            $options[$type_id] = $type['label'];
        }

        return $options;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int   $commission_id          The commission ID
     * @param array $commission_data        The commission data
     * @param array $commission_old_data    The commission old data
     */
    public function listener( $commission_id, $commission_data, $commission_old_data ) {

        // Get the affiliate user ID
        $affiliate = slicewp_get_affiliate( $commission_data['affiliate_id'] );
        $user_id =  $affiliate->get( 'user_id' );

        $new_status = $commission_data['status'];
        $old_status = $commission_old_data['status'];

        if( $new_status === 'paid' && $old_status !== 'paid' ) {

            // Trigger the commission paid
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'commission_id' => $commission_id,
                'type'          => $commission_data['type'],
                'amount'        => $commission_data['amount'],
                'operation'     => 'paid',
            ) );

        }

        if( $new_status === 'rejected' && $old_status !== 'rejected' ) {

            // Trigger the commission rejected
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'type'          => $commission_data['type'],
                'amount'        => $commission_data['amount'],
                'operation'     => 'rejected',
            ) );

        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if type and operation are not received
        if( ! isset( $event['type'] ) && ! isset( $event['operation'] ) ) {
            return false;
        }

        // Don't deserve if operation doesn't match with the trigger option
        if( $event['operation'] !== $trigger_options['operation'] ) {
            return false;
        }

        // Don't deserve if type doesn't match with the trigger option
        if( $trigger_options['type'] !== 'any' && $event['type'] !== $trigger_options['type'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['commission_id'] = ( isset( $event['commission_id'] ) ? $event['commission_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_SliceWP_Commission();