<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Integrations\Twilio\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $phone_number
 * @param string $message
 *
 * @return array
 */
function automatorwp_twilio_send_sms( $phone_number, $message ) {

    $sid = automatorwp_twilio_get_option( 'sid', '' );
    $token = automatorwp_twilio_get_option( 'token', '' );
    $number = automatorwp_twilio_get_option( 'number', '' );

    // Check that Twilio settings are correctly setup
    if( empty( $sid ) || empty( $token ) || empty( $number ) ) {
        return array( 'result' => false, 'message' => __( 'Please, configure your Twilio details on AutomatorWP settings.', 'automatorwp-pro' ) );
    }

    // Sanitize the phone number
    $phone_number = automatorwp_twilio_validate_phone_number( $phone_number );

    if( ! $phone_number ) {
        return array( 'result' => false, 'message' => __( 'Invalid phone number.', 'automatorwp-pro' ) );
    }

    // Sanitize message
    $message = wp_strip_all_tags( $message );

    try {

        $twilio  = new Twilio\Rest\Client( $sid, $token );

        $response = $twilio->messages->create( $phone_number,
            array(
                'body' => $message,
                'from' => $number,
            )
        );

        return array(
            'result' => true,
            'message' => __( 'Message sent successfully.', 'automatorwp-pro' )
        );

    } catch ( Twilio\Exceptions\RestException $exception ) {
        return array(
            'result' => false,
            'message' => sprintf( __( 'Could not send message, error received: %s', 'automatorwp-pro' ), $exception->getMessage() )
        );
    }
}

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @return string|false|WP_Error
 */
function automatorwp_twilio_validate_phone_number( $phone_number ) {

    // Remove spaces
    $phone_number = str_replace( ' ', '', $phone_number );
    // Remove -
    $phone_number = str_replace( '-', '', $phone_number );

    //Sanitize phone number
    $phone_number = filter_var( $phone_number, FILTER_SANITIZE_NUMBER_INT );

    // Check the length of the number
    if ( strlen( $phone_number ) < 10 || strlen( $phone_number ) > 14 ) {
        return false;
    } else {
        return $phone_number;
    }

}