<?php
/**
 * Logs
 *
 * @package     AutomatorWP\User Lists\Logs
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Adds user to list
 *
 * @since 1.0.0
 *
 * @return bool true on delete, false on failure
 */
function automatorwp_user_lists_add_user_to_list( $user_id, $list_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $list_id = absint( $list_id );

    if( automatorwp_user_lists_is_user_on_list( $user_id, $list_id ) ) {
        return false; // User already on list
    }

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );

    ct_insert_object( array(
        'user_id' => $user_id,
        'list_id' => $list_id,
    ) );

    ct_reset_setup_table();

    return true;

}

/**
 * Removes user from list
 *
 * @since 1.0.0
 *
 * @return bool true on delete, false on failure
 */
function automatorwp_user_lists_remove_user_from_list( $user_id, $list_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $list_id = absint( $list_id );

    $user_list = automatorwp_user_lists_get_user_list_entry( $user_id, $list_id );

    if( ! $user_list ) {
        return false; // User is not in list so can not remove it
    }

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );
    
    ct_delete_object( absint( $user_list->id ) );
    $deleted = $wpdb->delete( $ct_table->db->table_name, array( 'id' => $user_list->id ), array( '%d' ) );

    ct_reset_setup_table();

    return true;

}

/**
 * Checks if user is on the given list
 *
 * @since 1.0.0
 *
 * @return bool
 */
function automatorwp_user_lists_is_user_on_list( $user_id, $list_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $list_id = absint( $list_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );

    $count = absint( $wpdb->get_var(
        "SELECT COUNT(*) 
                FROM {$ct_table->db->table_name} AS ulu 
                WHERE ulu.user_id = {$user_id}
                AND ulu.list_id = {$list_id}"
    ) );

    ct_reset_setup_table();

    return (bool) ( $count >= 1 );

}

/**
 * Checks if user is on the given list
 *
 * @since 1.0.0
 *
 * @return stdClass The user list entry object (id, list_id, user_id)
 */
function automatorwp_user_lists_get_user_list_entry( $user_id, $list_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $list_id = absint( $list_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );

    $user_list = $wpdb->get_row(
        "SELECT *
                FROM {$ct_table->db->table_name} AS ulu 
                WHERE ulu.user_id = {$user_id}
                AND ulu.list_id = {$list_id}"
    );

    ct_reset_setup_table();

    return $user_list;

}

/**
 * Checks if user is on the given list
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_user_lists_get_user_lists( $user_id ) {

    global $wpdb;

    $user_id = absint( $user_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );
    $user_list_user_table = $ct_table->db->table_name;
    ct_reset_setup_table();

    $ct_table = ct_setup_table( 'automatorwp_user_lists' );

    $user_list = $wpdb->get_results(
        "SELECT ul.*
                FROM {$ct_table->db->table_name} AS ul
                LEFT JOIN {$user_list_user_table} AS ulu ON ulu.list_id = ul.id AND ulu.user_id = {$user_id}"
    );

    ct_reset_setup_table();

    return $user_list;

}

/**
 * Get all custom user fields
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_user_lists_get_all_lists() {

    global $wpdb, $automatorwp_user_lists_cache;

    if( ! $automatorwp_user_lists_cache ) {

        // Setup table
        $ct_table = ct_setup_table( 'automatorwp_user_lists' );

        // Get all fields
        $automatorwp_user_lists_cache = $wpdb->get_results( "SELECT * FROM {$ct_table->db->table_name} AS ul" );

        ct_reset_setup_table();

    }

    return $automatorwp_user_lists_cache;

}

/**
 * Get the count of all custom user fields
 *
 * @since 1.0.0
 *
 * @return int
 */
function automatorwp_user_lists_get_user_count( $list_id ) {

    global $wpdb;

    $list_id = absint( $list_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists_users' );

    $count = absint( $wpdb->get_var( "SELECT COUNT(*) FROM {$ct_table->db->table_name} AS ulu WHERE ulu.list_id = {$list_id}" ) );

    ct_reset_setup_table();

    return $count;

}

/**
 * Find custom user field that matches with a meta key
 *
 * @since 1.0.0
 *
 * @param string $name The list name to search
 *
 * @return array
 */
function automatorwp_user_lists_get_list_by_name( $name ) {

    global $wpdb;

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists' );

    // Search custom field by meta key
    $custom_field = $wpdb->get_row( $wpdb->prepare(
        "SELECT *
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.name = %s",
        $name,
    ) );

    ct_reset_setup_table();

    return $custom_field;

}

/**
 * Find custom user field that matches with a list ID
 *
 * @since 1.0.0
 *
 * @param int $list_id The list ID to search
 *
 * @return array
 */
function automatorwp_user_lists_get_list_by_id( $list_id ) {

    global $wpdb;

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_user_lists' );

    // Search list by list ID
    $custom_field = $wpdb->get_row( $wpdb->prepare(
        "SELECT *
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.id = %s",
        $list_id,
    ) );

    ct_reset_setup_table();

    return $custom_field;

}
