<?php
/**
 * Add Row
 *
 * @package     AutomatorWP\Integrations\WhatsApp\Actions\Send_Message_Template
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_WhatsApp_Send_Message_Template extends AutomatorWP_Integration_Action {
    
    public $integration = 'whatsapp';
    public $action = 'whatsapp_send_message_template';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Send message template to phone number', 'automatorwp-pro' ),
            'select_option' => __( 'Send <strong>message</strong> <strong>template</strong> to phone number', 'automatorwp-pro' ),
            /* translators: %1$s: Row. */
            'edit_label' => sprintf( __( 'Send %1$s to %2$s', 'automatorwp-pro' ), '{template}', '{phone}'  ),
            /* translators: %1$s: Row. */
            'log_label' => sprintf( __( 'Send %1$s to %2$s', 'automatorwp-pro' ), '{template}', '{phone}'),
            'options' => array(
                'phone' => array(
                    'from' => 'phone',
                    'default' => __( 'phone', 'automatorwp-pro' ),
                    'fields' => array(
                        'prefix' => array(
                            'name' => __( 'Country code:', 'automatorwp-pro' ),
                            'desc' => __( 'The country code. This field can be left empty if the country code is included in the Phone field.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'phone' => array(
                            'name' => __( 'Phone:', 'automatorwp-pro' ),
                            'desc' => __( 'The phone where you want to send the message.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                    )
                    ),
                'template' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'template',
                    'option_default'    => __( 'template', 'automatorwp-pro' ),
                    'name'              => __( 'Template:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'action_cb'         => 'automatorwp_whatsapp_get_templates',
                    'options_cb'        => 'automatorwp_whatsapp_options_cb_template',
                    'placeholder'       => 'Select a template',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){
        
        //Shorthand
        $prefix = $action_options['prefix'];
        $phone =  $action_options['phone'];
        $template = $action_options['template'];
        $this->result = '';


        // Bail if phone or template empty
        if( empty( $phone ) || empty( $template ) ) {
            $this->result = __( 'Both fields must be filled', 'automatorwp-pro' );
            return;
        }

        // Bail if WhatsApp not configured
        if( ! automatorwp_whatsapp_get_api() ) {
            $this->result = __( 'WhatsApp integration not configured in AutomatorWP settings.', 'automatorwp-pro' );
            return;
        }

        $phone = automatorwp_whatsapp_validate_phone_number( $prefix, $phone );

        // Send the message
        $response = automatorwp_whatsapp_send_message_template( $phone, $template );

        if ( isset( $response['code'] ) && $response["code"] === 200 ) {
            $this->result = sprintf ( __( 'Message sent to +%s', 'automatorwp-pro' ), $phone );
        } else {
            $response = json_decode( wp_remote_retrieve_body( $response ), true  );
            if ( isset( $response['error']['message'] ) ) {
                $this->result = sprintf( __( 'Message couldnt be sent to %s. Error: %s', 'automatorwp-pro' ), $phone, $response['error']['message'] );
            } else {
                $this->result = sprintf( __( 'Message couldnt be sent. Check the number: %s', 'automatorwp-pro' ), $phone );
            }
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' || $object->type !== $this->action ) {
            return;
        }

        // Warn user if the authorization has not been setup from settings
        if( ! automatorwp_whatsapp_get_api() ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">WhatsApp settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-whatsapp'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/whatsapp/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_WhatsApp_Send_Message_Template();