<?php
namespace BricksUltimate\Elements;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use BricksUltimate\Helpers;

class WooBuyNow extends \Bricks\Element {
	public $category 		= 'ultimate';
	public $name 			= 'bu-buy-now';
	public $icon 			= 'ti-bag';

	// Return localized element label
	public function get_label() {
		return esc_html__( '(BU)Buy Now Button', 'bricksultimate' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['link']    = array(
			'title' => esc_html__( 'Link', 'bricks' ),
			'tab'   => 'content',
		);
		$this->control_groups['extra']    = array(
			'title' => esc_html__( 'Extra Settings', 'bricksultimate' ),
			'tab'   => 'content',
		);
		$this->control_groups['icon']  = array(
			'title' => esc_html__( 'Icon', 'bricks' ),
			'tab'   => 'content',
		);
		$this->control_groups['style'] = array(
			'title' => esc_html__( 'Button Styles', 'bricks' ),
			'tab'   => 'content',
		);
	}

	// Set builder controls
	public function set_controls() {
		$this->controls['link_button'] = [
			'tab' 		=> 'content',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Button type', 'ziultimate'),
			'default' 	=> 'button',
			'options' 	=> [
				'button' 	=> esc_html__('Button'),
				'link' 		=> esc_html__('Plain Link'),
			],
			'inline' 	=> true,
			'medium' 	=> true
		];

		$this->controls['button_text'] = [
			'tab'         => 'content',
			'type'        => 'text',
			'label'       => esc_html__( 'Button Text', 'bricksultimate' ),
			'placeholder' => esc_html__( 'Buy Now', 'bricksultimate' ),
		];

		$this->controls['size'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Size', 'bricks' ),
			'type'        => 'select',
			'options'     => $this->control_options['buttonSizes'],
			'inline'      => true,
			'reset'       => true,
			'placeholder' => esc_html__( 'Default', 'bricks' ),
			'required' 	  => ['link_button', '=', 'button']
		];

		$this->controls['style'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Style', 'bricks' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'bricks' ),
			'required' 	  => ['link_button', '=', 'button']
		];

		$this->controls['circle'] = [
			'tab'   	=> 'content',
			'label' 	=> esc_html__( 'Circle', 'bricks' ),
			'type'  	=> 'checkbox',
			'reset' 	=> true,
			'required' 	=> ['link_button', '=', 'button']
		];

		$this->controls['outline'] = [
			'tab'   	=> 'content',
			'label' 	=> esc_html__( 'Outline', 'bricks' ),
			'type'  	=> 'checkbox',
			'reset' 	=> true,
			'required' 	=> ['link_button', '=', 'button']
		];

		$this->controls['redirect_link'] = [
			'tab'       => 'content',
			'group' 	=> 'link',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Redirect to', 'bricksultimate'),
			'info' 		=> esc_html__( 'Redirecting the customer after adding the product to cart.', 'bricksultimate' ),
			'default' 	=> 'checkout',
			'options' 	=> [
				'checkout' 	=> esc_html__( 'Checkout Page', 'ziultimate' ),
				'custom' 	=> esc_html__( 'Custom URL', 'ziultimate' )
			],
			'inline' 	=> true,
			'medium' 	=> true
		];

		$this->controls['page_link'] = [
			'tab' 		=> 'content',
			'group' 	=> 'link',
			'type' 		=> 'text',
			'label' 	=> esc_html__( 'URL', 'bricks' ),
			'placeholder' => 'https://redirecturl.com',
			'required' 	=> [ 'redirect_link', '=', 'custom']
		];

		$this->controls['rel'] = [
			'tab' 		=> 'content',
			'group' 	=> 'link',
			'type'      => 'text',
			'label'     => esc_html__('Rel Attribute', 'bricksultimate'),
			'placeholder' => 'noopener noreferrer nofollow'
		];

		$this->controls['aria_label'] = [
			'tab' 		=> 'content',
			'group' 	=> 'link',
			'type'      => 'text',
			'label'     => esc_html__('Aria label', 'bricksultimate'),
		];

		$this->controls['title'] = [
			'tab' 		=> 'content',
			'group' 	=> 'link',
			'type'      => 'text',
			'label'     => esc_html__('Title', 'bricksultimate'),
		];

		$this->controls['keep_cart_items'] = [
			'tab'       => 'content',
			'group' 	=> 'extra',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Keep cart items?', 'bricksultimate'),
			'info' 		=> esc_html__( 'Will keep the previously added cart items?', 'bricksultimate' ),
			'default' 	=> 'no',
			'options' 	=> [
				'no' 	=> esc_html__( 'No' ),
				'yes' 	=> esc_html__( 'Yes' )
			],
			'inline' 	=> true,
		];

		$this->controls['using_in_loop'] = [
			'tab'       => 'content',
			'group' 	=> 'extra',
			'type' 		=> 'select',
			'label' 	=> esc_html__('Using inside the query loop?', 'bricksultimate'),
			'default' 	=> 'no',
			'options' 	=> [
				'no' 	=> esc_html__( 'No' ),
				'yes' 	=> esc_html__( 'Yes' )
			]
		];

		$this->controls['exclude_products'] = [
			'tab' 			=> 'content',
			'group' 		=> 'extra',
			'type' 			=> 'text',
			'label' 		=> esc_html__( 'Disable button for these products', 'bricksultimate' ),
			'placeholder' 	=> esc_html__( 'Enter product ids with comma', 'bricksultimate' ),
		];

		$this->controls['exclude_product_types'] = [
			'tab' 			=> 'content',
			'group' 		=> 'extra',
			'type' 			=> 'select',
			'label' 		=> esc_html__( 'Disable button for these product types', 'bricksultimate' ),
			'placeholder' 	=> esc_html__('You can select multiple product types', 'bricksultimate'),
			'multiple' 		=> true,
			'options' 		=> [
				'grouped' 	=> esc_html__('Grouped'),
				'simple' 	=> esc_html__('Simple'),
				'variable' 	=> esc_html__('Variable'),
				'external' 	=> esc_html__('External'),
			]
		];

		$this->controls['icon'] = [
			'tab'   => 'content',
			'group' => 'icon',
			'label' => esc_html__( 'Icon', 'bricks' ),
			'type'  => 'icon',
		];

		$this->controls['iconTypography'] = [
			'tab'      => 'content',
			'group' 	=> 'icon',
			'label'    => esc_html__( 'Typography', 'bricks' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => 'i',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'required' => [ 'icon.icon', '!=', '' ],
		];

		$this->controls['iconPosition'] = [
			'tab'         => 'content',
			'group' 	  => 'icon',
			'label'       => esc_html__( 'Position', 'bricks' ),
			'type'        => 'select',
			'options'     => [
				'row-reverse' 		=> esc_html__('Left', 'bricks'),
				'row' 				=> esc_html__('Right', 'bricks'),
				'column-reverse' 	=> esc_html__('Top', 'bricks'),
			],
			'inline'      	=> true,
			'placeholder' 	=> esc_html__( 'Right', 'bricks' ),
			'default' 		=> 'row',
			'css' 			=> [
				[
					'selector' => '',
					'property' => 'flex-direction'
				]
			],
			'required'    	=> [ 'icon.icon', '!=', '' ],
		];

		$this->controls['iconGap'] = [
			'tab' 		=> 'content',
			'group' 	=> 'icon',
			'label'    	=> esc_html__( 'Gap', 'bricksultimate' ),
			'type'     	=> 'number',
			'units' 	=> true,
			'min' 		=> 0,
			'max' 		=> 100,
			'step' 		=> 1,
			'inline' 	=> true,
			'css'      	=> [
				[
					'property' => 'gap',
					'selector' => '',
				],
			],
			'default' 	=> '10px',
			'required' => [ 'icon.icon', '!=', '' ],
		];

		$this->controls['iconMrg'] = [
			'tab' 		=> 'content',
			'group' 	=> 'icon',
			'label'    	=> esc_html__( 'Margin', 'bricks' ),
			'type'     	=> version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'css'      	=> [
				[
					'property' => 'margin',
					'selector' => 'i',
				],
			],
			'required' => [ 'icon.icon', '!=', '' ],
		];

		$this->controls['iconHColor'] = [
			'tab' 		=> 'content',
			'group' 	=> 'icon',
			'label'    	=> esc_html__( 'Hover color', 'bricksultimate' ),
			'type'     	=> 'color',
			'css'      	=> [
				[
					'property' => 'color',
					'selector' => '&:hover i',
				],
			],
			'required' => [ 'icon.icon', '!=', '' ],
		];

		$selector = '&';
		$this->controls['sub_width'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'number',
			'label'     => esc_html__('Width', 'bricksultimate'),
			'units' 	=> true,
			'min' 		=> 0,
			'step' 		=> 1,
			'inline' 	=> true,
			'placeholder' => '100%',
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'width'
				]
			]
		];

		$this->controls['sub_sp_sep'] = [
			'tab'   	=> 'content',
			'group' 	=> 'style',
			'label' 	=> esc_html__( 'Spacing', 'bricks' ),
			'type'  	=> 'separator',
		];

		//margin field
		$this->controls['sub_mrg'] = [
			'tab'   	=> 'content',
			'group' 	=> 'style',
			'label' 	=> esc_html__( 'Margin', 'bricks' ),
			'type'  	=> version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'css'   	=> [
				[
					'property' => 'margin',
					'selector' => $selector,
				]
			],
		];

		//padding field
		$this->controls['sub_pad'] = [
			'tab'   	=> 'content',
			'group' 	=> 'style',
			'label' 	=> esc_html__( 'Padding', 'bricks' ),
			'type'  	=> version_compare( BRICKS_VERSION, '1.5', '>') ? 'spacing' : 'dimensions',
			'css' 		=> [
				[
					'property' => 'padding',
					'selector' => $selector,
				]
			],
		];

		$this->controls['sub_sp_sep_close'] = [
			'tab'   	=> 'content',
			'group' 	=> 'style',
			'type'  	=> 'separator',
		];

		$this->controls['sub_bg'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'color',
			'label'     => esc_html__('Background Color', 'bricksultimate'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['sub_font'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'label' 	=> esc_html__( 'Typography', 'bricks' ),
			'type' 		=> 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => $selector,
				],
			]
		];

		$this->controls['sub_brd'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'border',
			'label'     => esc_html__('Border', 'bricksultimate'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'border'
				]
			]
		];

		$this->controls['sub_shadow'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'box-shadow',
			'label'     => esc_html__('Box shadow', 'bricks'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> $selector,
					'property' 	=> 'box-shadow'
				]
			]
		];

		$this->controls['sub_hover_sep'] = [
			'tab'   	=> 'content',
			'group' 	=> 'style',
			'type'  	=> 'separator',
			'label'     => esc_html__('Hover State', 'bricksultimate'),
		];

		$this->controls['sub_hbg'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'color',
			'label'     => esc_html__('Background Color', 'bricksultimate'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> "{$selector}:hover",
					'property' 	=> 'background-color'
				]
			]
		];

		$this->controls['sub_hclr'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'color',
			'label'     => esc_html__('Color', 'bricksultimate'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> "{$selector}:hover",
					'property' 	=> 'color'
				]
			]
		];

		$this->controls['sub_hbrd'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'border',
			'label'     => esc_html__('Border', 'bricksultimate'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> "{$selector}:hover",
					'property' 	=> 'border'
				]
			]
		];

		$this->controls['sub_hshadow'] = [
			'tab' 		=> 'content',
			'group' 	=> 'style',
			'type'      => 'box-shadow',
			'label'     => esc_html__('Box shadow', 'bricks'),
			'inline' 	=> true,
			'small' 	=> true,
			'css' 		=> [
				[
					'selector' 	=> "{$selector}:hover",
					'property' 	=> 'box-shadow'
				]
			]
		];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		/*wp_register_style( 'bu-buy-now', Helpers::get_asset_url('css') . 'buynow.min.css', [], filemtime(Helpers::get_asset_path('css') . 'buynow.min.css'), 'all' );*/
		wp_register_script( 'bu-buy-now', Helpers::get_asset_url('js') . 'buynow.min.js', ['wc-add-to-cart-variation'], filemtime(Helpers::get_asset_path('js') . 'buynow.min.js'), true );
	}

	// Render element HTML
	public function render() {
		if( ! $this->hasPermission() )
			return;

		wp_enqueue_style( 'bu-buy-now' );
		wp_enqueue_script( 'bu-buy-now' );

		$settings = $this->settings;

		$using_in_loop 		= Helpers::get_value( $settings, 'using_in_loop', 'no');
		$keep_cart 			= Helpers::get_value( $settings, 'keep_cart_items', 'no');
		$redirect_link 		= Helpers::get_value( $settings, 'redirect_link', 'checkout');
		$link 				= Helpers::get_value( $settings, 'page_link', false );
		$button_text 		= Helpers::get_value( $settings, 'button_text', __('Buy Now', 'bricksultimate') );
		$isButton 			= Helpers::get_value( $settings, 'link_button', 'button' );

		$url = ( $redirect_link == 'custom' && ! empty( $link['page_link'] ) ) ? esc_url( $link['page_link'] ) : wc_get_checkout_url();

		if( $using_in_loop == 'yes' ) {
			$url = add_query_arg([ 'add_to_cart' => $this->post_id, 'keep_cart_items' => $keep_cart, 'bu_buy_now' => 'yes' ], $url );
		}

		if( $isButton == 'button' ) {
			$this->set_attribute( '_root', 'class', 'bricks-button' );
			$this->set_attribute( '_root', 'role', 'button' );

			if ( ! empty( $settings['size'] ) ) {
				$this->set_attribute( '_root', 'class', $settings['size'] );
			}

			if ( ! empty( $settings['style'] ) ) {
				// Outline
				if ( isset( $settings['outline'] ) ) {
					$this->set_attribute( '_root', 'class', 'outline' );
					$this->set_attribute( '_root', 'class', "bricks-color-{$settings['style']}" );
				}

				// Fill (= default)
				else {
					$this->set_attribute( '_root', 'class', "bricks-background-{$settings['style']}" );
				}
			}

			// Button circle
			if ( isset( $settings['circle'] ) ) {
				$this->set_attribute( '_root', 'class', 'circle' );
			}

			if ( isset( $settings['block'] ) ) {
				$this->set_attribute( '_root', 'class', 'block' );
			}
		}

		$aria_label = Helpers::get_value( $settings, 'aria_label', false );
		if( $aria_label ) {
			$this->set_attribute( '_root', 'aria-label', esc_html( $aria_label ) );
		}

		$title = Helpers::get_value( $settings, 'title', false );
		if( $title ) {
			$this->set_attribute( '_root', 'title', esc_html( $title ) );
		}

		$rel = Helpers::get_value( $settings, 'rel', false );
		if( $rel ) {
			$this->set_attribute( '_root', 'rel', esc_html( $rel ) );
		}

		$this->set_attribute( '_root', 'data-in-loop', $using_in_loop );
		$this->set_attribute( '_root', 'data-keep-cart', $keep_cart );
		$this->set_attribute( '_root', 'data-product_id', $this->post_id );

		echo "<a href='{$url}' {$this->render_attributes('_root')}>";

		echo "<span class='button-text'>{$button_text}</span>";
		
		$icon = ! empty( $settings['icon'] ) ? self::render_icon( $settings['icon'] ) : false;
		echo $icon;

		echo '</a>';
	}

	public static function render_builder() { ?>
		<script type="text/x-template" id="tmpl-bricks-bu-buy-now">
			<component
				:is="'a'"
				:class="[
					settings.link_button == 'button' ? 'bricks-button' : null,
					settings.size ? settings.size : null,
					settings.style ? settings.outline ? `outline bricks-color-${settings.style}` : `bricks-background-${settings.style}` : null,
					settings.circle ? 'circle' : null,
					settings.block ? 'block' : null
				]">
				<contenteditable tag="span" :name="name" controlKey="button_text" class="button-text" toolbar="style" :settings="settings"/>
				<icon-svg v-if="settings.icon" :iconSettings="settings.icon"/>
			</component>
		</script>
		<?php
	}

	public function hasPermission() {
		global $product;

		$product = WC()->product_factory->get_product( $this->post_id );

		if( $product === false )
			return false;

		if ( ! $product->is_in_stock() )
			return false;

		$settings = $this->settings;

		$used_in_loop 			= Helpers::get_value( $settings, 'using_in_loop', 'no');
		$exclude_products 		= Helpers::get_value( $settings, 'exclude_products', false);
		$exclude_product_types 	= Helpers::get_value( $settings, 'exclude_product_types', false);

		if( ! empty( $exclude_products ) ) {
			$exclude_products = explode( ",", $exclude_products );

			if( in_array( $product->get_id(), $exclude_products) )
				return false;
		}

		if( ! empty( $exclude_product_types ) ) {
			if( in_array( $product->get_type(), $exclude_product_types) )
				return false;
		}

		if( is_singular( 'product' ) && $used_in_loop == 'no' ) {
			return true;
		}

		return ( 'yes' == $used_in_loop && 'simple' !== $product->get_type() ) ? false : true;
	}
}