<?php

if( ! class_exists( 'Chart_Builder_Actions' ) ){

    /**
     * Class Chart_Builder_Actions
     * Class contains chart data filtering and hooks related to chart
     *
     * Main functionality belong to chart source and source type
     * Also chart settings and options
     *
     * Hooks used in the class
     *
     * @hooks           @actions        ays_chart_get_chart_data
     *
     *                  @filters        ays_chart_source_type
     *                                  ays_chart_source
     *                                  ays_chart_source_{source_type}
     *                                  ays_chart_get_settings
     *                                  ays_chart_get_options
     *                                  ays_chart_get_title
     *                                  ays_chart_get_type
     *                                  ays_chart_get_status
     *
     * @param           $plugin_name
     *
     * @since           1.0.0
     * @package         Chart_Builder
     * @subpackage      Chart_Builder/includes
     * @author          Chart Builder Team <info@ays-pro.com>
     */
    class Chart_Builder_Actions {

        /**
         * The ID of this plugin.
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $plugin_name    The ID of this plugin.
         */
        private $plugin_name;

        /**
         * The object of the database.
         *
         * @since    1.0.0
         * @access   private
         * @var      string    $db_obj    The database object of the chart.
         */
        private $db_obj;

        /**
         * The constructor of the class
         *
         * @since   1.0.0
         *
         * @param   $plugin_name
         */
        public function __construct( $plugin_name ) {

	        /**
	         * Assigning $plugin_name to the @plugin_name property
	         */
            $this->plugin_name = $plugin_name;

	        /**
	         * Assigning database object to the @db_obj property
	         *
	         * Creating instance of Chart_Builder_DB_Actions class
	         */
            $this->db_obj = new Chart_Builder_DB_Actions( $this->plugin_name );
        }

        /**
         * Get instance of this class
         *
         * @since   1.0.0
         * @access  public
         *
         * @param   $plugin_name
         *
         * @return  Chart_Builder_Actions
         */
        public static function get_instance( $plugin_name ){
            return new self( $plugin_name );
        }

        /**
         * Get chart data by id
         *
         * @since   1.0.0
         * @access  public
         *
         * @param   $id
         *
         * @return  null|non-empty-array
         */
        public function get_chart_data( $id ){

            /**
             * Checking if given valid @id
             *
             * @return null if not valid
             */
            if( is_null( $id ) || intval( $id ) === 0 ){
                return null;
            }

            /**
             * Retrieving chart row from database
             *
             * Passing @id as a parameter
             */
            $chart = $this->db_obj->get_item( $id );
            if( !$chart ){
                return null;
            }


            /**
             * Validate chart general data by calling validation function
             *
             * Function must @return array
             */
            $chart = $this->validate_item_data( $chart );


            /**
             * Getting source chart type from @chart object
             */
            $source_chart_type = $chart['source_chart_type'];
            $source_chart_type = apply_filters( 'ays_chart_source_chart_type', $source_chart_type );


            /**
             * Getting source type from @chart object
             */
            $source_type = $chart['source_type'];


            /**
             * Applying filter to source type for development purposes
             *
             * @accept manual, custom, etc.
             */
            $source_type = apply_filters( 'ays_chart_source_type', $source_type );

            /**
             * Getting source from @chart object
             */
            $source = $chart['source'];

            /**
             * Getting chart author id
             */
            $author_id = $chart['author_id'];

            /**
             * Getting @settings data from database by calling get_metadata function from db object
             *
             * get_metadata function must return an array that contains chart @settings
             */
            $chart_settings = $this->db_obj->get_metadata( $id );

            /**
             * Settings mustn't be empty or not array type data
             * So checking is array and not empty
             *
             * If validation not passed assigning empty array
             */
            if( ! is_array( $chart_settings ) || empty( $chart_settings ) ){
                $chart_settings = array();
            }

            /**
             * Converting @settings data to usable format
             *
             * convert_metadata function must return an array that contains formatted chart @settings
             */
            $chart_settings = $this->db_obj->convert_metadata( $chart_settings );

            /**
             * Applying defaults to @settings
             *
             * apply_default_metadata function must return an array
             * Expecting @settings with default values applied
             */
            $chart_settings = $this->db_obj->apply_default_metadata( $chart_settings );


            /**
             * Applying filter to @settings data for development purposes
             */
            $chart_settings = apply_filters( 'ays_chart_get_settings', $chart_settings );

            /**
             * Defining filter function name that will filter source data
             */
            $filter_function = 'chart_data_' . $source_type;

            /**
             * Calling filter function
             *
             * Before calling checking if the function exists and is callable
             *
             * Passing @source as a parameter
             * Expecting an array which must be returned after filter
             */
            $source_type_data = array();
            $gsheet_column_ordering = array();
            if( method_exists( $this , $filter_function ) && is_callable( array( $this, $filter_function ) ) ) {
                if ($source_type == 'import_from_external_db') {
                    $source_type_data = $this->$filter_function( $source, $id, $chart, $chart_settings);
                } else {
                    $source_type_data = $this->$filter_function( $source, $id );
                }

                if ($source_type == 'google_sheet') {
                    $filter_function_data = $this->$filter_function( $source, $id );
                    $source_type_data = $filter_function_data['data'];
                    $gsheet_column_ordering = $filter_function_data['column_order'];
                }
            }

            /**
             * Checking that returning data was array
             */
            if( ! is_array( $source_type_data ) ){
                $source_type_data = array();
            }

            /**
             * Checking availability of data in the returned value from filter function
             */
            if( empty( $source_type_data ) ){
                $source_type_data = array();
            }

            /**
             * Final source after validations and filters
             * And final filter that applies to the source for giving access to change the data after all
             *
             * Applying filter to @source for development purposes
             *
             * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
             * Please don't use this filter often when you need to change the source
             * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
             */
            $final_source = apply_filters( 'ays_chart_source', $source_type_data, $source_type );

            /**
             * Getting @options from @chart object
             */
            $chart_options = $chart['options'];

            /**
             * After getting @options from @chart object
             */
            unset( $chart['options'] );

            /**
             * Applying filter to @options for development purposes
             */
            $chart_options = apply_filters( 'ays_chart_get_options', $chart_options );

            /**
             * Getting external database custom settings data
             */
            $database = isset($chart['database']) ? $chart['database'] : "";

            /**
             * Getting quiz data
             */
            $quiz_query = $chart['quiz_query'];
            $quiz_id =  $chart['quiz_id'];
            $quiz_cat_id =  $chart['quiz_cat_id'];
            $quiz_question_cat_id =  $chart['quiz_question_cat_id'];

            /**
             * Collecting chart whole data into one variable
             *
             * @array $chart_data
             */
            $chart_data = array(
                'chart' => $chart,
                'author_id' => $author_id,
                'source_chart_type' => $source_chart_type,
                'source_type' => $source_type,
                'source' => $final_source,
                'gsheet_column_ordering' => $gsheet_column_ordering,
                'settings' => $chart_settings,
                'options' => $chart_options,
                'database' => $database,
                'quiz_query' => $quiz_query,
                'quiz_id' => $quiz_id,
                'quiz_cat_id' => $quiz_cat_id,
                'quiz_question_cat_id' => $quiz_question_cat_id,
            );

            /**
             * Action for get chart data function
             */
            do_action( 'ays_chart_get_chart_data', $chart_data );

            /**
             * Returning chart whole data
             *
             * @return $chart_data
             */
            return $chart_data;
        }

        /**
         * Chart data type manual
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_manual( $source, $chart_id ){

            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){

                /**
                 * Data are expected to be of type JSON
                 * Trying to parse the data
                 */
                $decoded_data = json_decode( $source, true );

                /**
                 * @decoded_data mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if( is_array( $decoded_data ) && ! empty( $decoded_data ) ){

                    /**
                     * Assigning parsed data to the @filtered_data variable
                     */
                    $filtered_data = $decoded_data;
                }

            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_manual', $filtered_data );
        }

        /**
         * Chart data type import_from_db
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_import_from_db( $source, $chart_id ){

            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){

                /**
                 * Data are expected to be of type array
                 * Trying to parse the data
                 */
	            $db_query_data_option_name = 'ays_chart_db_query_results_' . $chart_id;
	            $db_query_data = get_option( $db_query_data_option_name, array() );
	            $decoded_data = isset( $db_query_data['source'] ) && $db_query_data['source'] !== '' ? $db_query_data['source'] : '';

                /**
                 * @decoded_data mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if( $decoded_data !== '' ){

	                $source     = new Chart_Builder_DB_Query( stripslashes( $decoded_data ), $chart_id, $db_query_data );
	                $data       = $source->fetch( false );
	                $error      = $source->get_error();

                    /**
                     * Assigning parsed data to the @filtered_data variable
                     */
					if( empty( $error ) ) {
						$filtered_data = $data;
					}
                }
            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_import_from_db', $filtered_data );
        }

        /**
         * Chart data type import_from_external_db
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_import_from_external_db( $source, $chart_id, $chart_data, $chart_settings ){
            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();
            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
            */
            if( is_string( $source ) && trim( $source ) !== '' ){

                /**
                 * Data are expected to be of type array
                 * Trying to parse the data
                */
                $external_db_query_data_option_name = 'ays_chart_external_db_query_results_' . $chart_id;
                $external_db_query_data = get_option( $external_db_query_data_option_name, array() );
	            $decoded_data = isset( $external_db_query_data['source'] ) && $external_db_query_data['source'] !== '' ? $external_db_query_data['source'] : '';

                /**
                 * @decoded_data mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if( $decoded_data !== '' ){
                    $use_chart_specific_db = isset($chart_settings['use_chart_specific_db']) && $chart_settings ['use_chart_specific_db'] == 'on' ? true : false;

                    if ($use_chart_specific_db) {
                        $custom_db_settings = isset($chart_data['database']) && $chart_data['database'] != '' ? json_decode($chart_data['database'], ARRAY_A) : array();
                        $db_host = isset($custom_db_settings['host']) ? esc_attr($custom_db_settings['host']) : '';
                        $db_name = isset($custom_db_settings['db_name']) ? esc_attr($custom_db_settings['db_name']) : '';
                        $db_user = isset($custom_db_settings['user']) ? esc_attr($custom_db_settings['user']) : '';
                        $db_password = isset($custom_db_settings['password']) ? openssl_decrypt(base64_decode($custom_db_settings['password']), 'aes-256-cbc', SECURE_AUTH_KEY, 0, substr(md5(SECURE_AUTH_KEY), 0, 16)) : '';
                        $db_port = isset($custom_db_settings['port']) ? esc_attr($custom_db_settings['port']) : '3306';
                    } else {
                        $settings_obj = new Chart_Builder_Settings_DB_Actions( $this->plugin_name );
                        // Database settings
                        $databaseb_res = ($settings_obj->get_setting('database') === false) ? json_encode(array()) : $settings_obj->get_setting('database');
                        $db_settings = json_decode($databaseb_res, true);
                        $db_host = isset($db_settings['host']) ? esc_attr($db_settings['host']) : '';
                        $db_name = isset($db_settings['db_name']) ? esc_attr($db_settings['db_name']) : '';
                        $db_user = isset($db_settings['user']) ? esc_attr($db_settings['user']) : '';
                        $db_password = isset($db_settings['password']) ? openssl_decrypt(base64_decode($db_settings['password']), 'aes-256-cbc', SECURE_AUTH_KEY, 0, substr(md5(SECURE_AUTH_KEY), 0, 16)) : '';
                        $db_port = isset($db_settings['port']) ? esc_attr($db_settings['port']) : '3306';
                    }

                    $source = new Chart_Builder_External_DB_Query( stripslashes( $decoded_data ), $chart_id, $db_host, $db_name, $db_user, $db_password, $db_port );
	                $data       = $source->fetch( false, false, false, $source->external_db_connection );
	                $error      = $source->get_error();

                    /**
                     * Assigning parsed data to the @filtered_data variable
                     */
					if( empty( $error ) ) {
						$filtered_data = $data;
					}
                }
            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_import_from_external_db', $filtered_data );
        }

        /**
         * Chart data type google_sheet
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_google_sheet( $source, $chart_id ){

            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){

                /**
                 * Data are expected to be of type array
                 * Trying to parse the data
                 */
	            $gsheet_data_option_name = 'ays_chart_google_sheet_results_' . $chart_id;
	            $gsheet_query_data = get_option( $gsheet_data_option_name, array() );
	            $decoded_data = isset( $gsheet_query_data['source'] ) && $gsheet_query_data['source'] !== '' ? $gsheet_query_data['source'] : '';
                $column_order = isset($gsheet_query_data['column_order']) ? $gsheet_query_data['column_order'] : [];

                /**
                 * @decoded_data mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if( $decoded_data !== '' ){

	                $integrations_obj = new Chart_Builder_Integrations( CHART_BUILDER_NAME, CHART_BUILDER_VERSION );
	                $settings_obj = new Chart_Builder_Settings_DB_Actions( CHART_BUILDER_NAME );

	                $google_res           = ($settings_obj->get_setting('google') === false) ? json_encode(array()) : $settings_obj->get_setting('google');
	                $google               = json_decode( $google_res, true );
	                $google_client        = isset($google['client']) ? $google['client'] : '';
	                $google_secret        = isset($google['secret']) ? $google['secret'] : '';
	                $google_email         = isset($google['user_email']) ? $google['user_email'] : '';
	                $google_refresh_token = isset($google['refresh_token']) ? $google['refresh_token'] : '';

					$result = false;
	                if( $google ) {
		                $google_sheet_data = array(
			                "refresh_token"     => $google_refresh_token,
			                "client_id"         => $google_client,
			                "client_secret"     => $google_secret,
			                "user_email"        => $google_email,
			                "spreadsheet_id"    => $decoded_data,
		                );

		                $result = $integrations_obj->ays_chart_get_google_spreadsheet_data( $google_sheet_data );
	                }

                    /**
                     * Assigning parsed data to the @filtered_data variable
                     */

	                if ( $result !== false ) {
                        if( isset($result['valueRanges'][0] ) && is_array( $result['valueRanges'][0]) ){
                            $result_values = ( isset($result['valueRanges'][0]['values']) && is_array( $result['valueRanges'][0]['values']) ) ? $result['valueRanges'][0]['values'] : array();			
                        }
                        else{
                            $result_values = array();
                        }

		                // $headers = $result_values[0];
		                // unset( $result_values[0] );
                        if(!empty($result_values)){
                            foreach ($result_values as $key => $value) {
                                if (!isset($value) || count($value) <= 1) {
                                    unset($result_values[$key]);
                                }
                            }

                            $result_values = array_values( $result_values );

                            $data = $result_values;
                            if (!empty($column_order)) {
                                $sorted_data = [];

                                foreach ($result_values as $rowIndex => $row) {
                                    $sorted_row = [];
                                    foreach ($column_order as $colIndex => $column) {
                                        if ($column['enabled'] === 'true') {
                                            $sorted_row[] = $row[$column['columnNumber'] - 1];
                                        }
                                    }
                                    $sorted_data[] = $sorted_row;
                                }
                                
                                $data = $sorted_data;
                            }

                            $filtered_data = [
                                'data' => $data,
                                'column_order' => $column_order,
                            ];

                        }
	                }
                }
            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_google_sheet', $filtered_data );
        }

        /**
         * Chart data type file_import
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_file_import( $source, $chart_id ){

            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){
                 /**
                 * Data are expected to be of type JSON
                 * Trying to parse the data
                 */
                $decoded_data = json_decode( $source, true );

                /**
                 * @decoded_data mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if (is_array($decoded_data) && !empty($decoded_data)) {
	                
                    /**
                     * Assigning parsed data to the @filtered_data variable
                     */
		            foreach ($decoded_data as $key => $value) {
                        if (!isset($value) || count($value) <= 1 && $key != 0) {
                            unset($decoded_data[$key]);
                        }
                    }
                    
                    $ready_data = array_values( $decoded_data );
		            $filtered_data = $ready_data;
                }
            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_file_import', $filtered_data );
        }

        /**
         * Chart data type quiz_maker
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_quiz_maker( $source, $chart_id ){

            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){

                /**
                 * Data are expected to be of type array
                 * Trying to parse the data
                 */
	            // $quiz_maker_data_option_name = 'ays_chart_quiz_maker_results_' . $chart_id;
	            // $quiz_maker_data = get_option( $quiz_maker_data_option_name, array() );
	            // $decoded_data = isset( $quiz_maker_data['source'] ) && $quiz_maker_data['source'] !== '' ? $quiz_maker_data['source'] : '';

                $quiz_maker_data_option_name = 'ays_chart_quiz_maker_quiz_data_' . $chart_id;
                $quiz_query_data = get_option( $quiz_maker_data_option_name );
                $quiz_query = isset( $quiz_query_data['quiz_query'] ) && $quiz_query_data['quiz_query'] !== '' ? $quiz_query_data['quiz_query'] : ''; 
                /**
                 * @quiz_query mustn't be empty or not array type data
                 * So checking is array and not empty
                 */
                if( $quiz_query !== '' ){

                    $quiz_id = isset( $quiz_query_data['quiz_id'] ) && $quiz_query_data['quiz_id'] !== '' ? $quiz_query_data['quiz_id'] : 0;

                    $quiz_cat_id = isset( $quiz_query_data['quiz_cat_id'] ) && $quiz_query_data['quiz_cat_id'] !== '' ? $quiz_query_data['quiz_cat_id'] : 0;

                    $quiz_question_cat_id = isset( $quiz_query_data['quiz_question_cat_id'] ) && $quiz_query_data['quiz_question_cat_id'] !== '' ? $quiz_query_data['quiz_question_cat_id'] : 0;
                    
                    $return_results = CBFunctions()->get_quiz_query( $quiz_query, $quiz_id, $quiz_cat_id, $quiz_question_cat_id, get_current_user_id() );
                
			        $result_values = $return_results['result'];
                
                    $results = array();
			        if ( !empty($result_values) ) {
			        	if (count($return_results['result'][0]) != 1 ) {
                            $results = array();
                            $headers = array();
                            if ( $result_values ) {
                                $row_num = 0;
                                foreach ( $result_values as $row ) {
                                    $result = array();
			        			    $col_num = 0;
                                    foreach ( $row as $k => $v ) {
                                        $result[] = $v;
                                        if ( $row_num === 0 ) {
                                            $headers[]  = $k;
                                        }
                                    }
                                    $results[] = $result;
                                    $row_num++;
                                }
                            }
                        } else if (count($return_results['result'][0]) == 1) {
                            $results = array();
                            $headers = array();
                            if ( $result_values ) {
                                $row_num = 0;
                                foreach ( $result_values as $index => $row ) {
                                    $result = array();
                                    foreach ( $row as $k => $v ) {
                                        $result[] = strval($index) + 1;
                                        $result[] = $v;
                                        if ( $row_num === 0 ) {
                                            $headers[] = 'Quiz';
                                            $headers[] = $k;
                                        }
                                    }
                                    $results[] = $result;
                                    $row_num++;
                                }
                            }
                        }
                        array_unshift($results, $headers);
			        }

					$filtered_data = $results;
                }
            }

            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_quiz_maker', $filtered_data );
        }

        /**
         * Chart data type Woocommerce_data
         * Validating the data and returning validated array
         *
         * Returning applied filter to array for development purposes
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  JSON, empty string
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_woocommerce_data( $source, $chart_id ){
            global $wpdb;
            $table_name = $wpdb->prefix . "ayschart_charts";
            /**
             * Defining an empty array that will return after validation
             */
            $filtered_data = array();

            
            
            /**
             * @source mustn't be empty or not string type data
             * So checking is string and not empty
             *
             * If validation not passed function will return an empty array
             */
            if( is_string( $source ) && trim( $source ) !== '' ){
                 

                $woocommerce_data_option_name = 'ays_chart_woocommerce_results_saved_' . $chart_id;
                $woocommerce_data_option = get_option( $woocommerce_data_option_name );
                $woocommerce_data_type = isset($woocommerce_data_option['query']) ? $woocommerce_data_option['query'] : "";
               
                if ($woocommerce_data_type != ""){
                    $return_results = CBFunctions()->get_woocommerce_query_data($woocommerce_data_type); //data_type

                    $result_values = $return_results['result'];
                    /**
                     * @result_values mustn't be empty or not array type data
                    * So checking is array and not empty
                    */
                    if (is_array($result_values) && !empty($result_values)) {
                        
                        /**
                         * Assigning parsed data to the @filtered_data variable
                         */

                        $headers = array_keys($result_values[0]);
                        
                        $data = array($headers);
                        foreach($result_values as $key => $value){
                            $data[] = array_values($value);
                        }
                        
                        foreach ($data as $key => $value) {
                            if (!isset($value) || count($value) <= 1 && $key != 0) {
                                unset($data[$key]);
                            }
                        }
                        // every time it has to update data
                        $ready_data = array_values( $data );
                        $filtered_data = $ready_data;
                    }
                }
            }
            /**
             * Returning the data that has passed the validation
             * Returning applied filter to array for development purposes
             *
             * @return array
             */
            return apply_filters( 'ays_chart_source_woocommerce', $filtered_data );
        }

        /**
         * Chart data type custom
         * For developers
         * Returning data must be an array
         *
         * @since   1.0.0
         * @access  protected
         *
         * @accept  mixed
         * @param   $source
         *
         * @return  array
         */
	    protected function chart_data_custom( $source, $chart_id ){

            /**
             * Getting the data that will some developers pass via this filter
             */
            $filtered_data = apply_filters( 'ays_chart_source_custom', array(), $chart_id );

            /**
             * Checking that returning data was array
             */
            if( ! is_array( $filtered_data ) ){
                return array();
            }

            /**
             * Checking availability of data in array
             */
            if( empty( $filtered_data ) ){
                return array();
            }

            /**
             * Returning the data that has passed the validation
             *
             * @return array
             */
            return $filtered_data;
        }

        /**
         * Validation function for chart general data
         *
         * Validating title, description, type, status, etc.
         *
         * @since   1.0.0
         * @access  protected
         *
         * @param   $chart
         *
         * @return  array
         */
        protected function validate_item_data( $chart ){

            /**
             * @chart mustn't be empty or not array type data
             * So checking is array and not empty
             *
             * If validation not passed returning an empty array
             */
            if( ! is_array( $chart ) || empty( $chart ) ){
                return array();
            }

            /**
             * Defining an empty array that will return after validation
             *
             * During validation array will be filled with valid data
             */
            $chart_data = array();


            /**
             * Validating the @title
             *
             * Title mustn't be empty
             *
             * Applying esc_html filtering function to escape HTML from the title
             * Then applying filter to @title for development purposes
             *
             * @default "Chart example"
             */
            $chart_title = __( 'Chart example', $this->plugin_name );
            if( isset( $chart['title'] ) && $chart['title'] !== '' ) {
                $chart_title = esc_html( $chart['title'] );
            }

            $chart_data['title'] = apply_filters( 'ays_chart_get_title', $chart_title );


            /**
             * Validating the @description
             *
             * Description is optional, so it can be empty
             * Applying stripslashes function to escape unnecessary slashes
             *
             * @default empty string
             */
            $chart_data['description'] = '';
            if( isset( $chart['description'] ) && $chart['description'] !== '' ){
                $chart_data['description'] = stripslashes( $chart['description'] );
            }

            /**
             * Validating the @author_id
             */
            $chart_data['author_id'] = ( isset( $chart['author_id'] ) && $chart['author_id'] !== '' ) ? esc_html( intval($chart['author_id']) ) : get_current_user_id();


            /**
             * Validating the @type
             *
             * Type is required, it can't be empty
             * Applying esc_html filtering function to escape HTML from the type
             *
             * Then applying filter to @type for development purposes
             *
             * @accept values are google-charts, etc.
             * @default google-charts
             */
            $chart_type = 'google-charts';
            if( isset( $chart['type'] ) && $chart['type'] !== '' ){
                $chart_type = esc_html( $chart['type'] );
            }

            $chart_data['type'] = apply_filters( 'ays_chart_get_type', $chart_type );

            /**
             * Validating the @source_chart_type
             *
             * Source chart type is required, it can't be empty
             * Applying esc_html filtering function to escape HTML from the @source_type
             *
             * @accept values are manual, file, WordPress, database, etc.
             * @default manual
             */
            $chart_data['source_chart_type'] = ( isset( $chart['source_chart_type'] ) && $chart['source_chart_type'] !== '' ) ? esc_html( $chart['source_chart_type'] ) : 'pie_chart';

            /**
             * Validating the @source_type
             *
             * Source type is required, it can't be empty
             * Applying esc_html filtering function to escape HTML from the @source_type
             *
             * @accept values are manual, file, WordPress, database, etc.
             * @default manual
             */
            $chart_data['source_type'] = 'manual';
            if( isset( $chart['source_type'] ) && $chart['source_type'] !== '' ){
                $chart_data['source_type'] = esc_html( $chart['source_type'] );
            }

            $chart_data['database'] = isset($chart['database']) && $chart['database'] != '' ? sanitize_text_field($chart['database']) : '';

            $chart_data['quiz_query'] = isset($chart['quiz_query']) && $chart['quiz_query'] != '' ? sanitize_text_field($chart['quiz_query']) : '';
            $chart_data['quiz_id'] = isset( $chart['quiz_id'] ) && $chart['quiz_id'] != '' ? intval( $chart['quiz_id'] ) : 0;
            $chart_data['quiz_cat_id'] = isset( $chart['quiz_cat_id'] ) && $chart['quiz_cat_id'] != '' ? intval( $chart['quiz_cat_id'] ) : 0;
            $chart_data['quiz_question_cat_id'] = isset( $chart['quiz_question_cat_id'] ) && $chart['quiz_question_cat_id'] != '' ? intval( $chart['quiz_question_cat_id'] ) : 0;


            /**
             * Validating the @source
             *
             * Source is required, but it can be empty
             * Applying esc_html filtering function to escape HTML from the @source
             *
             * @accept values are JSON, CSV, serialized data, etc.
             * @default empty string
             */
            $chart_data['source'] = '';
            if( isset( $chart['source'] ) && $chart['source'] !== '' ){
                $chart_data['source'] =  $chart['source'] ;
            }


            /**
             * Validating the @status
             *
             * Status is required, it can't be empty
             * Applying esc_html filtering function to escape HTML from the @status
             *
             * Then applying filter to type for development purposes
             *
             * @accept values are published and draft
             * @default draft
             */
            $chart_status = 'draft';
            if( isset( $chart['status'] ) && $chart['status'] !== '' ){
                $chart_status = esc_html( $chart['status'] );
            }

            $chart_data['status'] = apply_filters( 'ays_chart_get_status', $chart_status );


            /**
             * Validating the @options
             * Options is optional, it can be empty
             *
             * @accept JSON
             * @default empty string
             * @return array
             */
	        $chart_data['options'] = array();
            if( isset( $chart['options'] ) && $chart['options'] !== '' ){

                /**
                 * Data are expected to be of type JSON
                 * Trying to parse the data
                 */
                $chart_options = json_decode( $chart['options'], true );

                /**
                 * Options mustn't be empty or not array type data
                 * So checking is array and not empty
                 *
                 * If validation not passed assigning empty array
                 */
                if( ! is_array( $chart_options ) || empty( $chart_options ) ){
                    $chart_options = array();
                }

                /**
                 * Pushing the result to the returning variable
                 */
                $chart_data['options'] = $chart_options;
            }

            /**
             * Returning validated data
             * Returning an array
             *
             * @return array
             */
            return $chart_data;
        }

        /** 
         * Returns default array for charts  
         */ 
        public function get_charts_default_data(){
            return  array(
                "commonTypeCharts" => array(
                    '0' => array(
                        'Country',
                        'Population',
                        'Url'
                    ),   
                    '1' => array(
                        'United States',
                        '189',
                        ''
                    ),
                    '2' => array(
                        'China',
                        '43',
                        ''
                    ),
                    '3' => array(
                        'Egypt',
                        '256',
                        ''
                    ),
                    '4' => array(
                        'France',
                        '123',
                        ''
                    ),
                    '5' => array(
                        'Australia',
                        '88',
                        ''
                    )        
                ),
                "orgTypeChart" => array(
                    '0' => array(
                        'Name',
                        'Description',
                        'Image',
                        'Parent name',
                        'Tooltip',
                        'Url',
                        'Parent id',
                        'Level'
                    ),
                    '1' => array(
                        'Mike',
                        'President',
                        '',
                        '',
                        'The President',
                        '',
                        '',
                        '1'
                    ),
                    '2' => array(
                        'Jim',
                        'Vice President',
                        '',
                        'Mike',
                        'VP',
                        '',
                        '1',
                        '2'
                    ),
                    '3' => array(
                        'Bob',
                        '',
                        '',
                        'Jim',
                        'Bob Sponge',
                        '',
                        '2',
                        '3'
                    ),
                    '4' => array(
                        'Carol',
                        '',
                        '',
                        'Bob',
                        '',
                        '',
                        '3',
                        '4'
                    ),
                    '5' => array(
                        'Alice',
                        '',
                        '',
                        'Mike',
                        '',
                        '',
                        '1',
                        '2'
                    ),
                )
            );
        }
        
        /** 
         * Returns default array for charts by chart type
         */ 
        public function get_chart_data_by_types(){
            return  array(
                "bubble_chart" => array(
                    '0' => array(
                        'ID',
                        'Life Expectancy (y-axis)',
                        'Fertility Rate (x-axis)',
                        'Region (category)',
                        'Population (value/size)',
                    ),
                    '1' => array(
                        'CAN',
                        80.66,
                        1.67,
                        'North America',
                        33739900,
                    ),
                    '2' => array(
                        'DEU',
                        79.84,
                        1.36,
                        'Europe',
                        33739900,
                    ),
                    '3' => array(
                        'DNK',
                        78.6,
                        1.84,
                        'Europe',
                        5523095,
                    ),
                    '4' => array(
                        'EGY',
                        72.73,
                        2.78,
                        'Middle East',
                        79716203,
                    ),
                    '5' => array(
                        'GBR',
                        80.05,
                        2,
                        'Europe',
                        61801570,
                    ),
                    '6' => array(
                        'IRN',
                        72.49,
                        1.7,
                        'Middle East',
                        73137148,
                    ),
                    '7' => array(
                        'IRQ',
                        68.09,
                        4.77,
                        'Middle East',
                        31090763,
                    ),
                    '8' => array(
                        'ISR',
                        81.55,
                        2.96,
                        'Middle East',
                        7485600,
                    ),
                    '9' => array(
                        'RUS',
                        68.6,
                        1.54,
                        'Europe',
                        141850000,
                    ),
                    '10' => array(
                        'USA',
                        78.09,
                        2.05,
                        'North America',
                        307007000,
                    )
                ),
                "scatter_chart" => array(
                    '0' => array(
                        'X-values',
                        'Series 1 Y-Values',
                        'Series 2 Y-Values'
                    ),
                    '1' => array(
                        10,
                        null,
                        77
                    ),
                    '2' => array(
                        20,
                        null,
                        20
                    ),
                    '3' => array(
                        33,
                        null,
                        24
                    ),
                    '4' => array(
                        55,
                        16,
                        null
                    ),
                    '5' => array(
                        14,
                        61,
                        null
                    ),
                    '6' => array(
                        48,
                        3,
                        null
                    ),
                    '7' => array(
                        14,
                        null,
                        9
                    )
                ),
                "table_chart" => array(
                    '0' => array(
                        'Name',
                        'Salary',
                        'Full Time Employee',
                    ),
                    '1' => array(
                        'Mike',
                        '$10,000',
                        true
                    ),
                    '2' => array(
                        'Jim',
                        '$8,000',
                        false
                    ),
                    '3' => array(
                        'Alice',
                        '$12,500',
                         true
                    ),
                    '4' => array(
                        'Bob',
                        '$7,000',
                        true
                    ),
                    '5' => array(
                        'Sam',
                        '$9,000',
                        true
                    ),
                    '6' => array(
                        'Caroline',
                        '$11,000',
                        true
                    ),
                ),
                "timeline_chart" => array(
                    "0" => array(
                        'President',
                        'Start',
                        'End',
                    ),
                    "1" => array(
                        'Washington',
                        '1789-04-30',
                        '1797-03-04',
                    ),
                    "2" => array(
                        'Adams',
                        '1797-03-04',
                        '1801-03-04',
                    ),
                    "3" => array(
                        'Jefferson',
                        '1801-03-04',
                        '1809-03-04',
                    ),
                ),
                "candlestick_chart" => array(
                    "0" => array(
                        'Mon',
                        20,
                        28,
                        38,
                        45
                    ),
                    "1" => array(
                        'Tue',
                        31,
                        38,
                        55,
                        66
                    ),
                    "2" => array(
                        'Wed',
                        50,
                        55,
                        77,
                        80
                    ),
                    "3" => array(
                        'Thu',
                        77,
                        77,
                        66,
                        50
                    ),
                    "4" => array(
                        'Fri',
                        68,
                        66,
                        22,
                        15
                    ),
                ),
                "gantt_chart" => array(
                    "0" => array(
                        'Task ID',
                        'Task Name',
                        'Resource ID',
                        'Start',
                        'End',
                        'Duration',
                        'Percent Complete',
                        'Dependencies'
                    ),
                    "1" => array(
                        'Research',
                        'Find sources',
                        '',
                        '2022-01-01',
                        '2022-01-05',
                        null,
                        107,
                        null
                    ),
                    "2" => array(
                        'Write',
                        'Write paper',
                        '',
                        null,
                        '2022-01-09',
                        3,
                        25,
                        'Research,Outline'
                    ),
                    "3" => array(
                        'Cite',
                        'Create bibliography',
                        '',
                        null,
                        '2022-01-07',
                        1,
                        20,
                        'Research'
                    ),
                    "4" => array(
                        'Complete',
                        'Hand in paper',
                        '',
                        null,
                        '2022-01-10',
                        1,
                        0,
                        'Cite,Write'
                    ),
                    "5" => array(
                        'Outline',
                        'Outline paper',
                        '',
                        null,
                        '2022-01-06',
                        1,
                        100,
                        'Research'
                    ),
                ),
                "sankey_diagram" => array(
                    "0" => array(
                        'From',
                        'To',
                        'Weight'
                    ),
                    "1" => array(
                        'A',
                        'X',
                        5
                    ),
                    "2" => array(
                        'A',
                        'Y',
                        7
                    ),
                    "3" => array(
                        'A',
                        'Z',
                        6
                    ),
                    "4" => array(
                        'B',
                        'X',
                        2
                    ),
                    "5" => array(
                        'B',
                        'Y',
                        9
                    ),
                    "6" => array(
                        'B',
                        'Z',
                        4
                    )
                ),
                "treemap" => array(
                    "0" => array(
                        'Location',
                        'Parent',
                        'Market trade volume (size)',
                        'Market increase/decrease (color)'
                    ),
                    "1" => array(
                        'Global',
                        '',
                        0,
                        0
                    ),
                    "2" => array(
                        'America',
                        'Global',
                        0,
                        0
                    ),
                    "3" => array(
                        'Europe',
                        'Global',
                        0,
                        0
                    ),
                    "4" => array(
                        'Asia',
                        'Global',
                        0,
                        0
                    ),
                    "5" => array(
                        'Australia',
                        'Global',
                        0,
                        0
                    ),
                    "6" => array(
                        'Africa',
                        'Global',
                        0,
                        0
                    ),
                    "7" => array(
                        'Brazil',
                        'America',
                        11,
                        10
                    ),
                    "8" => array(
                        'USA',
                        'America',
                        52,
                        31
                    ),
                    "9" => array(
                        'Mexico',
                        'America',
                        24,
                        12
                    ),
                    "10" => array(
                        'Canada',
                        'America',
                        16,
                        -23
                    ),
                    "11" => array(
                        'France',
                        'Europe',
                        42,
                        -11
                    ),
                    "12" => array(
                        'Germany',
                        'Europe',
                        31,
                        -2
                    ),
                    "13" => array(
                        'Sweden',
                        'Europe',
                        22,
                        -13
                    ),
                    "14" => array(
                        'Italy',
                        'Europe',
                        17,
                        4
                    ),
                    "15" => array(
                        'UK',
                        'Europe',
                        21,
                        -5
                    ),
                    "16" => array(
                        'China',
                        'Asia',
                        36,
                        4
                    ),
                    "17" => array(
                        'Japan',
                        'Asia',
                        20,
                        -12
                    ),
                    "18" => array(
                        'India',
                        'Asia',
                        40,
                        63
                    ),
                    "19" => array(
                        'Laos',
                        'Asia',
                        4,
                        34
                    ),
                    "20" => array(
                        'Mongolia',
                        'Asia',
                        1,
                        -5
                    ),
                    "21" => array(
                        'Israel',
                        'Asia',
                        12,
                        24
                    ),
                    "22" => array(
                        'Iran',
                        'Asia',
                        18,
                        13
                    ),
                    "23" => array(
                        'Pakistan',
                        'Asia',
                        11,
                        -52
                    ),
                    "24" => array(
                        'Egypt',
                        'Africa',
                        21,
                        0
                    ),
                    "25" => array(
                        'S. Africa',
                        'Africa',
                        30,
                        43
                    ),
                    "26" => array(
                        'Sudan',
                        'Africa',
                        12,
                        2
                    ),
                    "27" => array(
                        'Congo',
                        'Africa',
                        10,
                        12
                    ),
                    "28" => array(
                        'Zaire',
                        'Africa',
                        8,
                        10
                    ),
                ),
                "word_tree" => array(
                    "0" => array('Phrases'),
                    "1" => array('cats are better than dogs'),
                    "2" => array('cats eat kibble'),
                    "3" => array('cats are better than hamsters'),
                    "4" => array('cats are awesome'),
                    "5" => array('cats are people too'),
                    "6" => array('cats eat mice'),
                    "7" => array('cats meowing'),
                    "8" => array('cats in the cradle'),
                    "9" => array('cats eat mice'),
                    "10" => array('cats in the cradle lyrics'),
                    "11" => array('cats eat kibble'),
                    "12" => array('cats for adoption'),
                    "13" => array('cats are family'),
                    "14" => array('cats eat mice'),
                    "15" => array('cats are better than kittens'),
                    "16" => array('cats are evil'),
                    "17" => array('cats are weird'),
                    "18" => array('cats eat mice'),
                ),
            );
        }

        public function get_charts($ordering = ''){
            global $wpdb;
            $charts_table = esc_sql( $wpdb->prefix . CHART_BUILDER_DB_PREFIX ) . "charts";
    
            $sql = "SELECT id,title
                    FROM {$charts_table} WHERE `status` = 'published' OR `status` = 'draft'";
    
    
            if($ordering != ''){
                $sql .= ' ORDER BY id '.$ordering;
            }
    
            $charts = $wpdb->get_results( $sql , "ARRAY_A" );
    
            return $charts;
        }
    }
}

if( ! function_exists( 'CBActions' ) ){
	/**
	 * Function for quick access to Chart_Builder_Actions class
	 *
	 * @since   1.0.0
	 * @return  Chart_Builder_Actions
	 */
	function CBActions(){

		static $instance = null;

		if( $instance === null ){
			$instance = Chart_Builder_Actions::get_instance( CHART_BUILDER_NAME );
		}

		if( $instance instanceof Chart_Builder_Actions ){
			return $instance;
		}

		return Chart_Builder_Actions::get_instance( CHART_BUILDER_NAME );
	}
}
