<?php
/**
 * Plugin Name:       AMS - WooCommerce Amazon
 * Plugin URI:        https://affiliatepro.org/
 * Description:       Transform your WooCommerce store into a powerful Amazon affiliate platform with advanced product import, automated updates, and comprehensive analytics.
 * Version:           11.0.1
 * Requires at least: 5.6
 * Tested up to:      6.7.2
 * Requires PHP:      7.4
 * Author:            AffiliateProSaaS
 * Author URI:        https://affiliatepro.org/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       ams-wc-amazon
 * Domain Path:       /languages/
 * WC requires at least: 5.0
 * WC tested up to:   9.4.2
 * Update URI:        https://affiliatepro.org/updates/
 * Network:           true
 * 
 * @package           AMS_WC_Amazon
 * @author            AffiliateProSaaS
 * @copyright         2025 AffiliateProSaaS
 * @since             1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// ========================================
// PLUGIN VERSION & CORE CONSTANTS
// ========================================

// Main plugin version - READ FROM PLUGIN HEADER (SINGLE SOURCE OF TRUTH)
if (!defined('AMS_PLUGIN_VERSION')) {
    // Use get_file_data() which works regardless of folder name
    if (!function_exists('get_file_data')) {
        // Manual parsing fallback (rare edge case)
        $plugin_content = file_get_contents(__FILE__);
        preg_match('/\*\s*Version:\s*(.+)/', $plugin_content, $matches);
        define('AMS_PLUGIN_VERSION', isset($matches[1]) ? trim($matches[1]) : '1.0.0');
    } else {
        $plugin_data = get_file_data(__FILE__, array('Version' => 'Version'));
        define('AMS_PLUGIN_VERSION', $plugin_data['Version'] ?: '1.0.0');
    }
}

// Plugin identification
if (!defined('AMS_PLUGIN_UNIQUE_ID')) {
    define('AMS_PLUGIN_UNIQUE_ID', 'ams-woocommerce-amazon-affiliate');
}

// Plugin slug for updates and identification
if (!defined('AMS_PLUGIN_SLUG')) {
    define('AMS_PLUGIN_SLUG', 'ams-wc-amazon');
}

// Minimum requirements
if (!defined('AMS_MIN_WP_VERSION')) {
    define('AMS_MIN_WP_VERSION', '5.6');
}

if (!defined('AMS_MIN_WC_VERSION')) {
    define('AMS_MIN_WC_VERSION', '5.0');
}

if (!defined('AMS_MIN_PHP_VERSION')) {
    define('AMS_MIN_PHP_VERSION', '7.4');
}

// ========================================
// REQUIREMENTS CHECK
// ========================================

// Check PHP version
if (version_compare(PHP_VERSION, AMS_MIN_PHP_VERSION, '<')) {
    add_action('admin_notices', 'ams_php_version_notice');
    return;
}

// Check WordPress version
if (version_compare(get_bloginfo('version'), AMS_MIN_WP_VERSION, '<')) {
    add_action('admin_notices', 'ams_wp_version_notice');
    return;
}

require_once __DIR__ . '/vendor/autoload.php';

// Check if WooCommerce is active
if (!function_exists('is_plugin_active')) {
    require_once(ABSPATH . '/wp-admin/includes/plugin.php');
}

if (!is_plugin_active('woocommerce/woocommerce.php')) {
    add_action('admin_notices', 'ams_woocommerce_missing');
    return;
}

// Check WooCommerce version (but don't return - let WooCommerce handle compatibility)
if (defined('WC_VERSION') && version_compare(WC_VERSION, AMS_MIN_WC_VERSION, '<')) {
    add_action('admin_notices', 'ams_wc_version_notice');
}

// ========================================
// WOOCOMMERCE COMPATIBILITY DECLARATION
// ========================================

// Declare WooCommerce HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
    }
});

if (!class_exists('AmsWcAmazon')) {
    /**
     * The main plugin class
     */

    final class AmsWcAmazon {

        /**
         * AmsWcAmazon constructor.
         */

        private function __construct() {
            $this->define_constants();
            register_activation_hook(__FILE__, array($this, 'activate'));
            register_deactivation_hook(__FILE__, array($this, 'deactivate'));
            add_action('plugins_loaded', array($this, 'init_plugin'));
            add_action('plugins_loaded', array($this, 'plugins_loaded_text_domain'));
            add_action('admin_post_nopriv_cart_redirected_count', array($this, 'cart_redirected_count'));
            add_action('admin_post_cart_redirected_count', array($this, 'cart_redirected_count'));
            add_filter('cron_schedules', array($this, 'every_day_cron'));
        }

        /**
         * Initializes a single instance
         */

        public static function init() {
            static $instance = false;
            if (!$instance) {
                $instance = new self();
            }
            return $instance;
        }

        /**
         * Plugin text domain loaded
         */

        public function plugins_loaded_text_domain() {
            load_plugin_textdomain( 'ams-wc-amazon', false, AMS_PLUGIN_PATH . 'languages/' );
        }

        /**
         * Define plugin path and url constants
         */

        public function define_constants() {
            if (!defined('AMS_PLUGIN_FILE')) {
                define('AMS_PLUGIN_FILE', __FILE__);
            }
            if (!defined('AMS_PLUGIN_PATH')) {
                define('AMS_PLUGIN_PATH', plugin_dir_path(__FILE__));
            }
            if (!defined('AMS_PLUGIN_URL')) {
                define('AMS_PLUGIN_URL', plugin_dir_url(__FILE__));
            }
            // Version is already defined at the top of the file
            if (!defined('AMS_PLUGIN_NAME')) {
                define('AMS_PLUGIN_NAME', 'AMS - WooCommerce Amazon');
            }
            if (!defined('AMS_BRAND_NO_LOGO')) {
                define('AMS_BRAND_NO_LOGO', AMS_PLUGIN_URL . 'assets/plugin/img/brand/no-logo.png');
            }
        }


        /**
         *  Init plugin
         */

        public function init_plugin() {
            new \Amazon\Affiliate\Assets();

            // Always initialize admin and front-end classes
            if (is_admin()) {
                new \Amazon\Affiliate\Admin();
                add_action('admin_footer-plugins.php', 'ams_deactivation_popup_script');
                add_action('woocommerce_product_options_inventory_product_data', 'display_gtin_in_inventory_tab');
                add_filter('plugins_api', 'ams_plugin_information_content', 20, 3);
                add_action('admin_head', 'ams_admin_styles');
            } else {
                new \Amazon\Affiliate\Frontend();
                add_action('wp', 'show_page_load_time');
                add_shortcode('ams_display_products', 'ams_display_products_grid_shortcode');
                add_filter('the_content', 'append_custom_notification_to_content');
                add_action('woocommerce_single_product_summary', 'display_last_updated_date', 25);
                add_action('woocommerce_review_before_comment_text', 'custom_review_title', 10);
                add_filter('get_avatar_url', 'custom_reviewer_image_url', 10, 3);
                add_filter('get_avatar_data', 'custom_reviewer_image_data', 10, 2);
                add_action('wp_enqueue_scripts', 'ams_enqueue_image_fit_css');
            }

            //Code for variants gallery custom theme
            // Run the script on init
            add_action('init', 'update_custom_variation_images');
            // Add a filter to handle both attachment IDs and URLs when displaying images
            add_filter('wp_get_attachment_image_src', 'handle_url_based_attachment_images', 10, 4);
            //Code for variants gallery custom theme
            
            // Setup daily dashboard statistics tracking
            add_action('init', 'ams_setup_dashboard_cron');
            add_action('ams_daily_dashboard_stats', 'ams_track_daily_dashboard_stats');
            

            // Brands
            add_action('wp_ajax_ams_track_brand_click', 'ams_track_brand_click');
            add_action('wp_ajax_nopriv_ams_track_brand_click', 'ams_track_brand_click');
            
            // Enhanced Brand Admin AJAX Handlers
            add_action('wp_ajax_ams_reset_selected_brand_stats', 'ams_reset_selected_brand_stats');
            add_action('wp_ajax_ams_reset_single_brand_stats', 'ams_reset_single_brand_stats');
            add_action('wp_ajax_ams_brand_bulk_operations', 'ams_brand_bulk_operations');
            add_action('wp_ajax_ams_get_brand_performance_data', 'ams_get_brand_performance_data');
            
            // SEO Optimization Hooks
            add_action('wp_head', 'ams_add_brand_seo_meta');
            add_action('init', function() {
                if (isset($_GET['ams_brand_sitemap']) && $_GET['ams_brand_sitemap'] === '1') {
                    ams_generate_brand_sitemap();
                }
            });
            

            // AMS Replace Product AJAX Handler
            add_action('wp_ajax_ams_replace_product', array($this, 'ams_handle_replace_product_ajax'));
            
            // Reset functionality AJAX handlers
            add_action('wp_ajax_ams_reset_tracking_data', array($this, 'ams_handle_reset_tracking_data'));
            add_action('wp_ajax_ams_reset_dashboard_stats', array($this, 'ams_handle_reset_dashboard_stats'));
            add_action('wp_footer', 'ams_add_brand_tracking');
            add_action('init', 'ams_register_product_brand_taxonomy');
            add_action('product_brand_edit_form_fields', 'ams_add_brand_logo_field');
            add_action('product_brand_add_form_fields', 'ams_add_brand_logo_field');
            add_action('edited_product_brand', 'ams_save_brand_logo');
            add_action('create_product_brand', 'ams_save_brand_logo');
            add_action('admin_init', 'ams_register_brand_settings');
            
            // Add enhanced variation image management
            if (is_admin()) {
                add_action('woocommerce_product_after_variable_attributes', 'ams_display_variation_images_readonly', 10, 3);
                add_action('admin_enqueue_scripts', 'ams_enqueue_variation_admin_scripts');
            }
            add_action('ams_brand_settings_page', 'ams_show_brand_statistics');
            add_action('wp_ajax_ams_reset_stats', 'ams_reset_stats_handler');
            add_action('wp_head', 'ams_inject_custom_css');
            add_action('admin_menu', 'ams_add_brand_admin_menu');
            add_action('woocommerce_single_product_summary', 'ams_display_product_brand', 25);
            add_shortcode('brand_filter', 'ams_brand_filter_shortcode');
            add_action('woocommerce_before_shop_loop', 'ams_display_brand_filter', 20);
            // Brands

            add_action('admin_enqueue_scripts', 'ams_enqueue_admin_scripts');

            //Backend code for product meta box - admin side
            add_action('add_meta_boxes', 'ams_modify_product_image_meta_boxes', 40);

            // Add URL-based images to the product gallery
            add_action('woocommerce_product_write_panel_tabs', 'ams_add_url_images_to_product_gallery');

            // Handle deletion of URL-based images
            add_action('wp_ajax_woocommerce_remove_images', 'ams_handle_url_image_deletion', 5);

            // Add custom CSS to ensure proper display
            add_action('admin_head', 'ams_add_product_image_styles');

            //Backend code for product meta box - admin side

            add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'ams_add_action_links');
            add_filter('plugin_row_meta', 'ams_plugin_row_meta', 10, 4);

            //debug code
            //add_action('woocommerce_after_single_product', 'display_variation_image_debug_info');
            //add_action('woocommerce_after_single_product', 'display_simple_product_debug_info');
            //debug code

            // Initialize shared global variable
            $GLOBALS['amswoofiu'] = $this->amswoofiu();

            // Two-way sync: When WooCommerce currency changes, update Amazon country
            add_action('update_option_woocommerce_currency', array($this, 'sync_amazon_country_with_currency'), 10, 2);
        }

        /**
         * Init Fiu - Featured Image URL
         */

        function amswoofiu() {
            return \Amazon\Affiliate\Fiu\Fiu::instance();
        }

        /**
         * It's count cart amazon redirect
         */

        public function cart_redirected_count() {
            if (isset($_GET['url']) && isset($_GET['id'])) {
                $post_id = intval($_GET['id']);
                
                if (!$post_id || !get_post($post_id)) {
                    wp_die('Invalid product ID', 'Security Error', array('response' => 403));
                }
                
                $stored_product_url = get_post_meta($post_id, '_ams_product_url', true);
                if (empty($stored_product_url)) {
                    wp_die('Product has no Amazon URL', 'Security Error', array('response' => 403));
                }
                
                $incoming_url = urldecode_deep(sanitize_text_field($_GET['url']));
                $stored_asin = '';
                $incoming_asin = '';
                
                // Extract ASIN from stored URL
                if (preg_match('/\/dp\/([A-Z0-9]{10})/', $stored_product_url, $matches)) {
                    $stored_asin = $matches[1];
                }
                
                // Extract ASIN from incoming URL
                if (preg_match('/\/dp\/([A-Z0-9]{10})/', $incoming_url, $matches)) {
                    $incoming_asin = $matches[1];
                }
                
                if (empty($stored_asin) || empty($incoming_asin) || $stored_asin !== $incoming_asin) {
                    error_log("AMS Security: ASIN mismatch for product {$post_id}. Stored: {$stored_asin}, Incoming: {$incoming_asin}");
                    wp_die('URL validation failed', 'Security Error', array('response' => 403));
                }
                
                // Get user's IP address and user agent for better tracking
                $user_ip = $this->get_user_ip();
                $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '';
                $user_fingerprint = md5($user_ip . $user_agent);
                
                $current_time = time();
                $click_window = 3600; // 1 hour window to prevent duplicate counting
                
                // Check if this user (by IP+UA fingerprint) clicked this product recently
                $should_count = $this->should_count_interaction('click', $post_id, $user_fingerprint, $click_window);
                
                $url = '';
                if ('redirect' === get_option('ams_buy_action_btn')) {
                    // Only count if not clicked recently by this user
                    if ($should_count) {
                        $count_key = 'ams_product_direct_redirected';
                        $count = get_post_meta($post_id, $count_key, true);
                        $count = is_numeric($count) ? (int)$count : 0;
                        $count++;
                        update_post_meta($post_id, $count_key, $count);
                        
                        // Record this click in database
                        $this->record_interaction('click', $post_id, $user_fingerprint, $current_time);
                    }
                    // SECURITY FIX: Use validated URL
                    $url = $incoming_url;
                } elseif ('cart_page' === get_option('ams_buy_action_btn')) {
                    // Only count if not clicked recently by this user
                    if ($should_count) {
                        $count_key = 'ams_product_added_to_cart';
                        $count = get_post_meta($post_id, $count_key, true);
                        $count = is_numeric($count) ? (int)$count : 0;
                        $count++;
                        update_post_meta($post_id, $count_key, $count);
                        
                        // Record this click in database
                        $this->record_interaction('click', $post_id, $user_fingerprint, $current_time);
                    }
                    // SECURITY FIX: Use validated URL
                    $url = $incoming_url;
                }
                
                if (!empty($url)) {
                    wp_redirect(esc_url_raw($url));
                    exit();
                } else {
                    wp_die('Invalid redirect action', 'Configuration Error', array('response' => 500));
                }
            } else {
                wp_die('Missing required parameters', 'Invalid Request', array('response' => 400));
            }
        }

        /**
         * Get user's real IP address
         */
        private function get_user_ip() {
            $ip_keys = array('HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR');
            
            foreach ($ip_keys as $key) {
                if (array_key_exists($key, $_SERVER) === true) {
                    foreach (explode(',', $_SERVER[$key]) as $ip) {
                        $ip = trim($ip);
                        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                            return $ip;
                        }
                    }
                }
            }
            
            return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
        }

        /**
         * Check if we should count this interaction based on recent activity
         */
        private function should_count_interaction($type, $post_id, $user_fingerprint, $time_window) {
            $option_key = 'ams_tracking_' . $type . '_' . $post_id . '_' . $user_fingerprint;
            $stored_data = get_option($option_key, null);
            
            if (!$stored_data) {
                return true;
            }
            
            // Handle both old format (timestamp) and new format (JSON)
            $data = json_decode($stored_data, true);
            $last_interaction = is_array($data) ? ($data['timestamp'] ?? 0) : (int)$stored_data;
            
            if (!$last_interaction) {
                return true;
            }
            
            $current_time = time();
            
            return ($current_time - $last_interaction) > $time_window;
        }

        /**
         * Record interaction in database with enhanced analytics
         */
        private function record_interaction($type, $post_id, $user_fingerprint, $timestamp) {
            $option_key = 'ams_tracking_' . $type . '_' . $post_id . '_' . $user_fingerprint;
            
            // Enhanced tracking data
            $tracking_data = [
                'timestamp' => $timestamp,
                'ip' => $this->get_user_ip(),
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'referrer' => $_SERVER['HTTP_REFERER'] ?? '',
                'device_info' => $this->get_device_info(),
                'geo_info' => $this->get_geo_info(),
                'page_url' => $_SERVER['REQUEST_URI'] ?? '',
                'session_id' => session_id() ?: $user_fingerprint
            ];
            
            update_option($option_key, json_encode($tracking_data));
            
            // Clean up old tracking data (older than 7 days) to prevent database bloat
            $this->cleanup_old_tracking_data();
        }
        
        /**
         * Get detailed device information
         */
        private function get_device_info() {
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
            
            $device_info = [
                'is_mobile' => wp_is_mobile(),
                'browser' => $this->detect_browser($user_agent),
                'os' => $this->detect_os($user_agent),
                'device_type' => wp_is_mobile() ? 'mobile' : 'desktop'
            ];
            
            return $device_info;
        }
        
        /**
         * Detect browser from user agent
         */
        private function detect_browser($user_agent) {
            $browsers = [
                'Chrome' => '/Chrome/i',
                'Firefox' => '/Firefox/i',
                'Safari' => '/Safari/i',
                'Edge' => '/Edge/i',
                'Opera' => '/Opera/i',
                'Internet Explorer' => '/MSIE/i'
            ];
            
            foreach ($browsers as $browser => $pattern) {
                if (preg_match($pattern, $user_agent)) {
                    return $browser;
                }
            }
            
            return 'Unknown';
        }
        
        /**
         * Detect operating system from user agent
         */
        private function detect_os($user_agent) {
            $os_array = [
                'Windows 11' => '/Windows NT 10.0.*Win64.*x64/i',
                'Windows 10' => '/Windows NT 10.0/i',
                'Windows 8.1' => '/Windows NT 6.3/i',
                'Windows 8' => '/Windows NT 6.2/i',
                'Windows 7' => '/Windows NT 6.1/i',
                'macOS' => '/Mac OS X/i',
                'iOS' => '/iPhone|iPad/i',
                'Android' => '/Android/i',
                'Linux' => '/Linux/i'
            ];
            
            foreach ($os_array as $os => $pattern) {
                if (preg_match($pattern, $user_agent)) {
                    return $os;
                }
            }
            
            return 'Unknown';
        }
        
        /**
         * Get geographic information from IP
         */
        private function get_geo_info() {
            $ip = $this->get_user_ip();
            
            // Basic geo detection
            $geo_info = [
                'ip' => $ip,
                'country' => 'Unknown',
                'country_code' => 'XX',
                'city' => 'Unknown',
                'region' => 'Unknown'
            ];
            
            // Try to get country from IP
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                $geo_data = $this->get_ip_geolocation($ip);
                if ($geo_data) {
                    $geo_info = array_merge($geo_info, $geo_data);
                }
            }
            
            return $geo_info;
        }
        
        /**
         * Get IP geolocation using free service
         */
        private function get_ip_geolocation($ip) {
            // FOR TESTING: Log what IP we're checking
            error_log("AMS Debug: Checking geolocation for IP: " . $ip);
            // Use free IP geolocation service
            $api_url = "http://ip-api.com/json/{$ip}?fields=status,country,countryCode,region,city";
            
            $response = wp_remote_get($api_url, [
                'timeout' => 5,
                'headers' => [
                    'User-Agent' => 'AMS-WooCommerce-Amazon-Plugin'
                ]
            ]);
            
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $data = json_decode(wp_remote_retrieve_body($response), true);
                
                if ($data && $data['status'] === 'success') {
                    return [
                        'country' => $data['country'] ?? 'Unknown',
                        'country_code' => $data['countryCode'] ?? 'XX',
                        'city' => $data['city'] ?? 'Unknown',
                        'region' => $data['region'] ?? 'Unknown'
                    ];
                }
            }
            
            return null;
        }

        /**
         * TEST FUNCTION: Check geolocation with public IP (REMOVE IN PRODUCTION)
         */
        public function test_geolocation() {
            $test_ip = '8.8.8.8'; // Google DNS - public IP
            $result = $this->get_ip_geolocation($test_ip);
            error_log("AMS Debug: Test geolocation result: " . print_r($result, true));
            return $result;
        }

        /**
         * Clean up old tracking data to prevent database bloat
         */
        private function cleanup_old_tracking_data() {
            // Only run cleanup 1% of the time to avoid performance impact
            if (rand(1, 100) !== 1) {
                return;
            }
            
            global $wpdb;
            $cutoff_time = time() - (7 * 24 * 3600); // 7 days ago
            
            // Get all tracking options to check timestamps
            $tracking_options = $wpdb->get_results(
                "SELECT option_name, option_value FROM {$wpdb->options} 
                 WHERE option_name LIKE 'ams_tracking_%'"
            );
            
            foreach ($tracking_options as $option) {
                $data = json_decode($option->option_value, true);
                $timestamp = is_array($data) ? ($data['timestamp'] ?? 0) : (int)$option->option_value;
                
                if ($timestamp < $cutoff_time) {
                    delete_option($option->option_name);
                }
            }
        }

        /**
         * Do Stuff Plugin activation
         */

        public function activate() {
        	flush_rewrite_rules();
            $installer = new \Amazon\Affiliate\Installer();
            $installer->run();
        }

        /**
         * Every day cron interval register
         */

        // Define the schedule
        public function every_day_cron($schedules) {
            $schedules['ams_every_day'] = array(
                'interval' => 86400,
                'display'  => esc_html__('Once Daily', 'ams-wc-amazon'),
            );
            
            // Add minute intervals for automation
            $schedules['every_minute'] = array(
                'interval' => 60, // 1 minute = 60 seconds
                'display'  => esc_html__('Every Minute', 'ams-wc-amazon'),
            );
            
            $schedules['every_5_minutes'] = array(
                'interval' => 300, // 5 minutes = 5 * 60
                'display'  => esc_html__('Every 5 Minutes', 'ams-wc-amazon'),
            );
            
            $schedules['every_15_minutes'] = array(
                'interval' => 900, // 15 minutes = 15 * 60
                'display'  => esc_html__('Every 15 Minutes', 'ams-wc-amazon'),
            );
            
            $schedules['every_30_minutes'] = array(
                'interval' => 1800, // 30 minutes = 30 * 60
                'display'  => esc_html__('Every 30 Minutes', 'ams-wc-amazon'),
            );
            
            // Add hourly intervals for automation
            $schedules['every_2_hours'] = array(
                'interval' => 7200, // 2 hours = 2 * 60 * 60
                'display'  => esc_html__('Every 2 Hours', 'ams-wc-amazon'),
            );
            
            $schedules['every_6_hours'] = array(
                'interval' => 21600, // 6 hours = 6 * 60 * 60
                'display'  => esc_html__('Every 6 Hours', 'ams-wc-amazon'),
            );
            
            $schedules['every_12_hours'] = array(
                'interval' => 43200, // 12 hours = 12 * 60 * 60
                'display'  => esc_html__('Every 12 Hours', 'ams-wc-amazon'),
            );
            
            // Add weekly schedule for automation
            $schedules['weekly'] = array(
                'interval' => 604800, // 7 days = 7 * 24 * 60 * 60
                'display'  => esc_html__('Once Weekly', 'ams-wc-amazon'),
            );
            
            // Add monthly schedule for automation  
            $schedules['monthly'] = array(
                'interval' => 2635200, // 30.5 days = 30.5 * 24 * 60 * 60
                'display'  => esc_html__('Once Monthly', 'ams-wc-amazon'),
            );
            
            return $schedules;
        }

        /**
         * Plugin deactivate
         */

        public function deactivate() {
            flush_rewrite_rules();
            wp_clear_scheduled_hook( 'ams_product_availability' );
        }

        public function ams_handle_replace_product_ajax() {
            $productsearch = new \Amazon\Affiliate\Admin\Productsearch();
            $productsearch->ams_replace_product();
        }

        /**
         * Handle reset tracking data AJAX request
         */
        public function ams_handle_reset_tracking_data() {
            // Verify nonce and permissions
            if (!wp_verify_nonce($_POST['nonce'], 'ams_reset_tracking_nonce') || !current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Security check failed or insufficient permissions.', 'ams-wc-amazon')]);
                return;
            }

            $reset_type = sanitize_text_field($_POST['reset_type'] ?? 'all');
            $result = $this->reset_tracking_data($reset_type);

            if ($result['success']) {
                wp_send_json_success([
                    'message' => $result['message'],
                    'deleted_count' => $result['deleted_count']
                ]);
            } else {
                wp_send_json_error(['message' => $result['message']]);
            }
        }

        /**
         * Handle reset dashboard statistics AJAX request
         */
        public function ams_handle_reset_dashboard_stats() {
            // Verify nonce and permissions
            if (!wp_verify_nonce($_POST['nonce'], 'ams_reset_dashboard_nonce') || !current_user_can('manage_options')) {
                wp_send_json_error(['message' => __('Security check failed or insufficient permissions.', 'ams-wc-amazon')]);
                return;
            }

            $reset_type = sanitize_text_field($_POST['reset_type'] ?? 'all');
            $result = $this->reset_dashboard_statistics($reset_type);

            if ($result['success']) {
                wp_send_json_success([
                    'message' => $result['message'],
                    'reset_stats' => $result['reset_stats']
                ]);
            } else {
                wp_send_json_error(['message' => $result['message']]);
            }
        }

        /**
         * Reset tracking data based on type
         */
        private function reset_tracking_data($type = 'all') {
            global $wpdb;
            
            $deleted_count = 0;
            $message = '';

            try {
                switch ($type) {
                    case 'clicks':
                        $deleted_count = $wpdb->query(
                            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ams_tracking_click_%'"
                        );
                        $message = sprintf(__('Successfully deleted %d click tracking records.', 'ams-wc-amazon'), $deleted_count);
                        break;

                    case 'views':
                        $deleted_count = $wpdb->query(
                            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ams_tracking_view_%'"
                        );
                        $message = sprintf(__('Successfully deleted %d view tracking records.', 'ams-wc-amazon'), $deleted_count);
                        break;

                    case 'geographic':
                        // Delete all tracking data (since geographic info is part of tracking records)
                        $deleted_count = $wpdb->query(
                            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ams_tracking_%'"
                        );
                        $message = sprintf(__('Successfully reset geographic data (%d records deleted).', 'ams-wc-amazon'), $deleted_count);
                        break;

                    case 'all':
                    default:
                        $deleted_count = $wpdb->query(
                            "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ams_tracking_%'"
                        );
                        $message = sprintf(__('Successfully deleted all tracking data (%d records).', 'ams-wc-amazon'), $deleted_count);
                        break;
                }

                return [
                    'success' => true,
                    'message' => $message,
                    'deleted_count' => $deleted_count
                ];

            } catch (Exception $e) {
                return [
                    'success' => false,
                    'message' => __('Error occurred while resetting tracking data: ', 'ams-wc-amazon') . $e->getMessage()
                ];
            }
        }

        /**
         * Reset dashboard statistics based on type
         */
        private function reset_dashboard_statistics($type = 'all') {
            $reset_stats = [];
            $message = '';

            try {
                switch ($type) {
                    case 'products':
                        // Reset product view counts
                        global $wpdb;
                        $wpdb->query("UPDATE {$wpdb->postmeta} SET meta_value = '0' WHERE meta_key = 'ams_product_views_count'");
                        $reset_stats['products'] = 0;
                        $message = __('Product view counts have been reset to zero.', 'ams-wc-amazon');
                        break;

                    case 'cart':
                        // Reset cart addition counts
                        global $wpdb;
                        $wpdb->query("UPDATE {$wpdb->postmeta} SET meta_value = '0' WHERE meta_key = 'ams_product_added_to_cart'");
                        $reset_stats['cart'] = 0;
                        $message = __('Cart addition counts have been reset to zero.', 'ams-wc-amazon');
                        break;

                    case 'redirects':
                        // Reset redirect counts
                        global $wpdb;
                        $wpdb->query("UPDATE {$wpdb->postmeta} SET meta_value = '0' WHERE meta_key = 'ams_product_direct_redirected'");
                        $reset_stats['redirects'] = 0;
                        $message = __('Amazon redirect counts have been reset to zero.', 'ams-wc-amazon');
                        break;

                    case 'all':
                    default:
                        // Reset all dashboard statistics
                        global $wpdb;
                        $wpdb->query("UPDATE {$wpdb->postmeta} SET meta_value = '0' WHERE meta_key IN ('ams_product_views_count', 'ams_product_added_to_cart', 'ams_product_direct_redirected')");
                        
                        // Also clear daily tracking data
                        delete_option('ams_dashboard_last_tracked');
                        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'ams_daily_stats_%'");
                        
                        $reset_stats = [
                            'products' => 0,
                            'views' => 0,
                            'cart' => 0,
                            'redirects' => 0
                        ];
                        $message = __('All dashboard statistics have been reset to zero.', 'ams-wc-amazon');
                        break;
                }

                return [
                    'success' => true,
                    'message' => $message,
                    'reset_stats' => $reset_stats
                ];

            } catch (Exception $e) {
                return [
                    'success' => false,
                    'message' => __('Error occurred while resetting statistics: ', 'ams-wc-amazon') . $e->getMessage()
                ];
            }
        }

        /**
         * Sync Amazon country when WooCommerce currency changes
         */
        public function sync_amazon_country_with_currency($old_value, $new_value) {
            // Currency to Amazon country mapping
            $currency_to_country_map = array(
                'AUD' => 'com.au',  // Australia
                'BRL' => 'com.br',  // Brazil
                'CAD' => 'ca',      // Canada
                'CNY' => 'cn',      // China
                'EUR' => 'de',      // Germany (default for EUR)
                'INR' => 'in',      // India
                'JPY' => 'jp',      // Japan
                'MXN' => 'mx',      // Mexico
                'SAR' => 'sa',      // Saudi Arabia
                'SGD' => 'sg',      // Singapore
                'TRY' => 'com.tr',  // Turkey
                'AED' => 'ae',      // United Arab Emirates
                'GBP' => 'co.uk',   // United Kingdom
                'USD' => 'com',     // United States
                'PLN' => 'pl'       // Poland
            );
            
            // Handle currency change
            if (isset($currency_to_country_map[$new_value])) {
                // Currency is supported - update Amazon country
                update_option('ams_amazon_country', $currency_to_country_map[$new_value]);
            } else {
                // Currency is NOT supported by Amazon - clear Amazon country or set to default
                // Option 1: Clear the setting
                // delete_option('ams_amazon_country');
                
                // Option 2: Set to default (US) - uncomment if preferred
                // update_option('ams_amazon_country', 'com');
                
                // Option 3: Do nothing - keep current Amazon country (current behavior)
                // This allows users to manually manage unsupported currencies
            }
        }
    }
}

// Additional functions

// ========================================
// VERSION & UPDATE FUNCTIONS
// ========================================

/**
 * Check for plugin updates from remote server
 */
function ams_check_version() {
    $url = add_query_arg(
        array(
            'slug' => AMS_PLUGIN_SLUG,
            'client_version' => AMS_PLUGIN_VERSION
        ),
        'https://buy.affiliatepro.org/version/check'
    );
    
    $response = wp_remote_get($url, array(
        'timeout' => 15,
        'headers' => array(
            'User-Agent' => 'AMS-Plugin/' . AMS_PLUGIN_VERSION . '; ' . home_url()
        )
    ));
    
    if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
        $update_data = json_decode(wp_remote_retrieve_body($response));
        if (isset($update_data->version) && !empty($update_data->version)) {
            return $update_data->version;
        }
    }
    
    return false;
}

// ========================================
// ADMIN NOTICE FUNCTIONS
// ========================================

/**
 * Display PHP version requirement notice
 */
function ams_php_version_notice() {
    $message = sprintf(
        esc_html__('AMS - WooCommerce Amazon requires PHP version %s or higher. You are running version %s. Please upgrade PHP to activate this plugin.', 'ams-wc-amazon'),
        AMS_MIN_PHP_VERSION,
        PHP_VERSION
    );
    printf('<div class="notice notice-error"><p><strong>%s</strong></p></div>', $message);
}

/**
 * Display WordPress version requirement notice
 */
function ams_wp_version_notice() {
    $message = sprintf(
        esc_html__('AMS - WooCommerce Amazon requires WordPress version %s or higher. You are running version %s. Please upgrade WordPress to activate this plugin.', 'ams-wc-amazon'),
        AMS_MIN_WP_VERSION,
        get_bloginfo('version')
    );
    printf('<div class="notice notice-error"><p><strong>%s</strong></p></div>', $message);
}

/**
 * Display WooCommerce version requirement notice
 */
function ams_wc_version_notice() {
    $message = sprintf(
        esc_html__('AMS - WooCommerce Amazon requires WooCommerce version %s or higher. You are running version %s. Please upgrade WooCommerce to use all features.', 'ams-wc-amazon'),
        AMS_MIN_WC_VERSION,
        defined('WC_VERSION') ? WC_VERSION : 'Unknown'
    );
    printf('<div class="notice notice-error"><p><strong>%s</strong></p></div>', $message);
}

// ========================================
// PLUGIN INFORMATION FUNCTIONS
// ========================================

/**
 * Get plugin information array
 */
function ams_get_plugin_info() {
    return array(
        'name' => 'AMS - WooCommerce Amazon',
        'version' => AMS_PLUGIN_VERSION,
        'slug' => AMS_PLUGIN_SLUG,
        'file' => __FILE__,
        'path' => plugin_dir_path(__FILE__),
        'url' => plugin_dir_url(__FILE__),
        'text_domain' => 'ams-wc-amazon',
        'author' => 'AffiliateProSaaS',
        'author_uri' => 'https://affiliatepro.org/',
        'plugin_uri' => 'https://affiliatepro.org/',
        'min_wp' => AMS_MIN_WP_VERSION,
        'min_wc' => AMS_MIN_WC_VERSION,
        'min_php' => AMS_MIN_PHP_VERSION
    );
}

/**
 * Check if plugin requirements are met
 */
function ams_requirements_met() {
    return (
        version_compare(PHP_VERSION, AMS_MIN_PHP_VERSION, '>=') &&
        version_compare(get_bloginfo('version'), AMS_MIN_WP_VERSION, '>=') &&
        is_plugin_active('woocommerce/woocommerce.php')
    );
}

/**
 * Check if WooCommerce version requirements are met
 */
function ams_wc_requirements_met() {
    return (!defined('WC_VERSION') || version_compare(WC_VERSION, AMS_MIN_WC_VERSION, '>='));
}

// ========================================
// PLUGIN INITIALIZATION
// ========================================

/**
 * Initializes the main plugin
 *
 * @return \AmsWcAmazon
 */
function ams_wc_amazon() {
    return AmsWcAmazon::init();
}

/**
 * Initialize plugin - Always load if basic requirements met
 * WooCommerce version check is handled separately to avoid compatibility issues
 */
if (ams_requirements_met()) {
    ams_wc_amazon();
}