<?php

namespace Amazon\Affiliate\Admin;

class AutomationLogs {
    
    public function __construct() {
        add_action('wp_ajax_ams_get_automation_logs', [$this, 'get_automation_logs']);
        add_action('wp_ajax_ams_clear_automation_logs', [$this, 'clear_automation_logs']);
        add_action('wp_ajax_ams_export_automation_logs', [$this, 'export_automation_logs']);
    }
    
    /**
     * Log an automation event
     */
    public static function log($level, $automation_type, $process, $message, $details = '', $product_id = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        // Create table if it doesn't exist
        self::create_logs_table();
        
        $wpdb->insert(
            $table_name,
            [
                'timestamp' => current_time('mysql'),
                'level' => strtoupper($level),
                'automation_type' => strtoupper($automation_type),
                'process' => strtoupper($process),
                'message' => $message,
                'details' => $details,
                'product_id' => $product_id
            ],
            ['%s', '%s', '%s', '%s', '%s', '%s', '%d']
        );
        

        
        // Clean old logs (keep last 1000 entries)
        self::cleanup_old_logs();
    }
    
    /**
     * Create automation logs table
     */
    private static function create_logs_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id int(11) NOT NULL AUTO_INCREMENT,
            timestamp datetime NOT NULL,
            level varchar(20) NOT NULL,
            automation_type varchar(50) NOT NULL,
            process varchar(50) NOT NULL,
            message text NOT NULL,
            details longtext,
            product_id int(11),
            PRIMARY KEY (id),
            KEY timestamp (timestamp),
            KEY level (level),
            KEY automation_type (automation_type),
            KEY process (process)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Get automation logs with filtering
     */
    public function get_automation_logs() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        // Get filter parameters
        $automation_type = sanitize_text_field($_POST['automation_type'] ?? 'all');
        $level = sanitize_text_field($_POST['level'] ?? 'all');
        $process = sanitize_text_field($_POST['process'] ?? 'all');
        $search = sanitize_text_field($_POST['search'] ?? '');
        $date_range = sanitize_text_field($_POST['date_range'] ?? '7');
        $page = intval($_POST['page'] ?? 1);
        // Read per_page from request with sane bounds (10-200), default 50
        $per_page = intval($_POST['per_page'] ?? 50);
        if ($per_page < 10) { $per_page = 10; }
        if ($per_page > 200) { $per_page = 200; }
        $offset = ($page - 1) * $per_page;
        
        // Build WHERE clause
        $where_conditions = ['1=1'];
        $where_values = [];
        
        if ($automation_type !== 'all') {
            $where_conditions[] = 'automation_type = %s';
            $where_values[] = strtoupper($automation_type);
        }
        
        if ($level !== 'all') {
            $where_conditions[] = 'level = %s';
            $where_values[] = strtoupper($level);
        }
        
        if ($process !== 'all') {
            $where_conditions[] = 'process = %s';
            $where_values[] = strtoupper($process);
        }
        
        if (!empty($search)) {
            $where_conditions[] = '(message LIKE %s OR details LIKE %s)';
            $where_values[] = '%' . $wpdb->esc_like($search) . '%';
            $where_values[] = '%' . $wpdb->esc_like($search) . '%';
        }
        
        if ($date_range !== 'all') {
            $where_conditions[] = 'timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)';
            $where_values[] = intval($date_range);
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        // Get total count
        $count_sql = "SELECT COUNT(*) FROM $table_name WHERE $where_clause";
        if (!empty($where_values)) {
            $count_sql = $wpdb->prepare($count_sql, $where_values);
        }
        $total_logs = $wpdb->get_var($count_sql);
        
        // Get logs
        $sql = "SELECT * FROM $table_name WHERE $where_clause ORDER BY timestamp DESC LIMIT %d OFFSET %d";
        $where_values[] = $per_page;
        $where_values[] = $offset;
        
        $logs = $wpdb->get_results($wpdb->prepare($sql, $where_values));
        
        // Format logs for display
        $formatted_logs = [];
        foreach ($logs as $log) {
            $formatted_logs[] = [
                'id' => $log->id,
                'timestamp' => $log->timestamp,
                'timestamp_formatted' => date('H:i:s', strtotime($log->timestamp)),
                'date_formatted' => date('M j, Y', strtotime($log->timestamp)),
                'time_ago' => human_time_diff(strtotime($log->timestamp)) . ' ago',
                'level' => $log->level,
                'automation_type' => $log->automation_type,
                'process' => $log->process,
                'message' => $log->message,
                'details' => $log->details,
                'product_id' => $log->product_id,
                'level_class' => $this->get_level_class($log->level),
                'type_display' => $this->get_type_display($log->automation_type),
                'process_display' => $this->get_process_display($log->process)
            ];
        }
        
        wp_send_json_success([
            'logs' => $formatted_logs,
            'total' => $total_logs,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total_logs / $per_page)
        ]);
    }
    
    /**
     * Clear automation logs
     */
    public function clear_automation_logs() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        $clear_type = sanitize_text_field($_POST['clear_type'] ?? 'all');
        
        if ($clear_type === 'old') {
            // Clear logs older than 30 days
            $result = $wpdb->query("DELETE FROM $table_name WHERE timestamp < DATE_SUB(NOW(), INTERVAL 30 DAY)");
        } else {
            // Clear all logs
            $result = $wpdb->query("TRUNCATE TABLE $table_name");
        }
        
        if ($result !== false) {
            self::log('INFO', 'SYSTEM', 'MAINTENANCE', 'Automation logs cleared by user');
            wp_send_json_success('Logs cleared successfully');
        } else {
            wp_send_json_error('Failed to clear logs');
        }
    }
    
    /**
     * Export automation logs
     */
    public function export_automation_logs() {
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        $logs = $wpdb->get_results("SELECT * FROM $table_name ORDER BY timestamp DESC LIMIT 1000");
        
        $filename = 'automation-logs-' . date('Y-m-d-H-i-s') . '.txt';
        
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        echo "AMS WooCommerce Amazon Plugin - Automation Logs\n";
        echo "Generated: " . current_time('mysql') . "\n";
        echo str_repeat("=", 80) . "\n\n";
        
        foreach ($logs as $log) {
            echo "[{$log->timestamp}] [{$log->level}] [{$log->automation_type}] [{$log->process}] {$log->message}\n";
            if (!empty($log->details)) {
                echo "Details: {$log->details}\n";
            }
            if ($log->product_id) {
                echo "Product ID: {$log->product_id}\n";
            }
            echo "\n";
        }
        
        exit;
    }
    
    /**
     * Clean up old logs
     */
    private static function cleanup_old_logs() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        // Keep only the latest 1000 logs
        $wpdb->query("DELETE FROM $table_name WHERE id NOT IN (SELECT id FROM (SELECT id FROM $table_name ORDER BY timestamp DESC LIMIT 1000) AS temp)");
    }
    
    /**
     * Get CSS class for log level
     */
    private function get_level_class($level) {
        switch ($level) {
            case 'SUCCESS': return 'text-success';
            case 'ERROR': return 'text-danger';
            case 'WARNING': return 'text-warning';
            case 'INFO': return 'text-info';
            default: return 'text-muted';
        }
    }
    
    /**
     * Get display name for automation type
     */
    private function get_type_display($type) {
        switch ($type) {
            case 'REVIEW_IMPORT': return 'Review Import';
            case 'PRODUCT_UPDATE': return 'Product Update';
            case 'PRICE_MONITOR': return 'Price Monitor';
            case 'INVENTORY_SYNC': return 'Inventory Sync';
            case 'EMAIL_AUTOMATION': return 'Email Automation';
            case 'SYSTEM': return 'System';
            default: return ucwords(str_replace('_', ' ', strtolower($type)));
        }
    }
    
    /**
     * Get display name for process
     */
    private function get_process_display($process) {
        switch ($process) {
            case 'SCRAPING': return 'Scraping';
            case 'API_CALL': return 'API Call';
            case 'DATABASE': return 'Database';
            case 'SCHEDULING': return 'Scheduling';
            case 'IMPORT': return 'Import';
            case 'EXPORT': return 'Export';
            case 'SYNC': return 'Sync';
            case 'MAINTENANCE': return 'Maintenance';
            default: return ucwords(str_replace('_', ' ', strtolower($process)));
        }
    }
    
    /**
     * Get available automation types for filtering
     */
    public static function get_automation_types() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        $types = $wpdb->get_col("SELECT DISTINCT automation_type FROM $table_name ORDER BY automation_type");
        
        return $types ?: [];
    }
    
    /**
     * Get available processes for filtering
     */
    public static function get_processes() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ams_automation_logs';
        
        $processes = $wpdb->get_col("SELECT DISTINCT process FROM $table_name ORDER BY process");
        
        return $processes ?: [];
    }
}
