<?php
namespace Amazon\Affiliate\Admin;

use Amazon\Affiliate\Admin\AutomationLogs;

/**
 * Automation Page Class
 * 
 * Handles all automation services including review sync, product updates, etc.
 *
 * @package Amazon\Affiliate\Admin
 */
class AutomationPage {
    
    public function __construct() {
        // Register AJAX handlers
        add_action('wp_ajax_ams_save_review_automation', array($this, 'save_review_automation'));
        add_action('wp_ajax_ams_get_review_automation_status', array($this, 'get_review_automation_status'));
        add_action('wp_ajax_ams_test_review_automation', array($this, 'test_review_automation'));
        add_action('wp_ajax_ams_toggle_review_automation', array($this, 'toggle_review_automation'));
        add_action('wp_ajax_ams_get_product_amazon_url', array($this, 'get_product_amazon_url'));
        add_action('wp_ajax_ams_trigger_cron_manually', array($this, 'trigger_cron_manually'));
        add_action('wp_ajax_ams_test_product_availability', array($this, 'test_product_availability'));
        add_action('wp_ajax_ams_run_cron_hook', array($this, 'run_cron_hook'));
        add_action('wp_ajax_ams_delete_cron_hook', array($this, 'delete_cron_hook'));
        add_action('wp_ajax_ams_schedule_cron_hook', array($this, 'schedule_cron_hook'));
        
        // Register the automation cron hook (same pattern as existing AMS cron)
        add_action('ams_review_automation_event', array($this, 'execute_review_automation'));
    }

    /**
     * Render the automation page
     */
    public function automation_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'ams-wc-amazon'));
        }

        // Handle form submissions (same pattern as settings page)
        if (isset($_POST['action']) && $_POST['action'] === 'ams_review_automation') {
            $this->save_review_automation();
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'ams_product_availability_toggle') {
            $this->save_product_availability_cron();
        }

        // Include the view file
        include dirname(__FILE__) . '/views/automation.php';
    }

    /**
     * Get automation statistics
     */
    private function get_automation_statistics() {
        global $wpdb;
        
        // Get total products with Amazon URLs
        $total_products = $wpdb->get_var("
            SELECT COUNT(DISTINCT p.ID) 
            FROM {$wpdb->posts} p 
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
            WHERE p.post_type = 'product' 
            AND p.post_status = 'publish' 
            AND pm.meta_key = '_ams_product_url' 
            AND pm.meta_value != ''
        ");

        // Get total imported reviews
        $total_reviews = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->comments} c
            INNER JOIN {$wpdb->commentmeta} cm ON c.comment_ID = cm.comment_id
            WHERE c.comment_type = 'review' 
            AND cm.meta_key = 'review_hash'
        ");

        // Get products with reviews
        $products_with_reviews = $wpdb->get_var("
            SELECT COUNT(DISTINCT c.comment_post_ID)
            FROM {$wpdb->comments} c
            INNER JOIN {$wpdb->commentmeta} cm ON c.comment_ID = cm.comment_id
            WHERE c.comment_type = 'review' 
            AND cm.meta_key = 'review_hash'
        ");

        // Get next scheduled run (same pattern as existing daily cron)
        $next_run = wp_next_scheduled('ams_review_automation_event');
        

        
        $last_run = get_option('ams_automation_last_run', '');
        

        
        // Simple next run detection (same pattern as existing AMS cron)
        $final_next_run = $next_run;
        
        return array(
            'total_products' => (int)$total_products,
            'total_reviews' => (int)$total_reviews,
            'products_with_reviews' => (int)$products_with_reviews,
            'next_run' => $final_next_run,
            'last_run' => $last_run,
            'is_scheduled' => ($final_next_run !== false) && ($final_next_run !== null)
        );
    }

    /**
     * Save review automation settings
     */
    public function save_review_automation() {
        // Security check
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'ams-wc-amazon'));
        }

        if (!wp_verify_nonce($_POST['automation_nonce'], 'automation_setting_nonce')) {
            wp_die(__('Security check failed.', 'ams-wc-amazon'));
        }

        // Get the enable/disable option (same pattern as settings page)
        $enable_review_cron = isset($_POST['enable_review_cron']) ? '1' : '0';

        // Save the Enable/Disable Review Cron setting (same pattern as settings page)
        update_option('enable_review_cron', $enable_review_cron);

        // Manage cron job scheduling (EXACT same pattern as settings page)
        if ($enable_review_cron === '1') {
            if (!wp_next_scheduled('ams_review_automation_event')) {
                wp_schedule_event(time(), 'ams_every_day', 'ams_review_automation_event');
            }
        } else {
            if (wp_next_scheduled('ams_review_automation_event')) {
                wp_clear_scheduled_hook('ams_review_automation_event');
            }
        }
        
        wp_redirect('admin.php?page=automation&settings=saved');
        exit;
    }

    /**
     * Save product availability cron settings (same pattern as review automation)
     */
    public function save_product_availability_cron() {
        // Security check
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'ams-wc-amazon'));
        }

        if (!wp_verify_nonce($_POST['automation_nonce'], 'automation_setting_nonce')) {
            wp_die(__('Security check failed.', 'ams-wc-amazon'));
        }

        // Get the enable/disable option (same pattern as settings page)
        $enable_product_cron = isset($_POST['enable_product_cron']) ? '1' : '0';

        // Save the Enable/Disable Product Cron setting
        update_option('enable_product_cron', $enable_product_cron);

        // Manage cron job scheduling (same pattern as existing crons)
        if ($enable_product_cron === '1') {
            if (!wp_next_scheduled('ams_product_availability')) {
                wp_schedule_event(time(), 'ams_every_day', 'ams_product_availability');
            }
        } else {
            if (wp_next_scheduled('ams_product_availability')) {
                wp_clear_scheduled_hook('ams_product_availability');
            }
        }
        
        wp_redirect('admin.php?page=automation&settings=saved');
        exit;
    }



    /**
     * Get current automation status
     */
    public function get_review_automation_status() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        $settings = get_option('ams_review_automation_settings', array());
        $stats = $this->get_automation_statistics();

        wp_send_json_success(array(
            'settings' => $settings,
            'stats' => $stats
        ));
    }

    /**
     * Toggle automation on/off quickly
     */
    public function toggle_review_automation() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $settings = get_option('ams_review_automation_settings', array('enabled' => false));
        $settings['enabled'] = !$settings['enabled'];

        update_option('ams_review_automation_settings', $settings);

        if ($settings['enabled']) {
            $this->schedule_review_automation($settings);
            $message = __('Review automation enabled successfully!', 'ams-wc-amazon');
        } else {
            $this->unschedule_review_automation();
            $message = __('Review automation disabled successfully!', 'ams-wc-amazon');
        }

        wp_send_json_success(array(
            'message' => $message,
            'enabled' => $settings['enabled']
        ));
    }

    /**
     * Test automation (run once manually)
     */
    public function test_review_automation() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        // Run automation once for testing
        $result = $this->execute_review_automation(true);

        wp_send_json_success(array(
            'message' => __('Test automation completed successfully!', 'ams-wc-amazon'),
            'result' => $result
        ));
        }
    
    /**
     * Auto-activate review automation cron (same pattern as existing AMS cron)
     */
    public function ams_review_automation_activation() {
        $settings = get_option('ams_review_automation_settings', array());
        
        if (!empty($settings['enabled']) && $settings['enabled'] === 'yes') {
            // If automation is enabled but no cron is scheduled, schedule it
            if (!wp_next_scheduled('ams_review_automation_event')) {
                $this->schedule_review_automation();
            }
        }
    }
    
    /**
     * Get product Amazon URL for logs
     */
    public function get_product_amazon_url() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_get_product_url')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $product_id = intval($_POST['product_id'] ?? 0);
        if (!$product_id) {
            wp_send_json_error(array('message' => __('Invalid product ID.', 'ams-wc-amazon')));
            return;
        }

        $amazon_url = get_post_meta($product_id, '_ams_product_url', true);
        
        wp_send_json_success(array(
            'amazon_url' => $amazon_url,
            'product_id' => $product_id
        ));
    }

    /**
     * Manually trigger cron execution for debugging
     */
    public function trigger_cron_manually() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        // Trigger the cron manually (use correct hook name)
        error_log('AMS Debug - Manually triggering ams_review_automation_event');
        do_action('ams_review_automation_event');
        
        wp_send_json_success(array(
            'message' => __('Cron triggered manually!', 'ams-wc-amazon')
        ));
    }

    /**
     * Test product availability cron (lightweight - only 3 products)
     */
    public function test_product_availability() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_product_availability')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        // Get only 3 products for testing
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => 3,
            'meta_query' => array(
                array(
                    'key' => '_ams_product_url',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        $products = get_posts($args);
        
        if (empty($products)) {
            wp_send_json_error(array('message' => __('No products found for testing.', 'ams-wc-amazon')));
            return;
        }

        // Log test start

        
        $processed = 0;
        foreach ($products as $product) {
            $product_url = get_post_meta($product->ID, '_ams_product_url', true);
            if (!empty($product_url)) {
                // Simple availability check (no actual updates)
                $response = wp_remote_get($product_url, array('timeout' => 10));
                if (!is_wp_error($response)) {
                    $processed++;

                } else {

                }
            }
        }



        wp_send_json_success(array(
            'message' => sprintf(__('Test completed! Checked %d products. See logs for details.', 'ams-wc-amazon'), $processed)
        ));
    }

    /**
     * Run a specific cron hook manually (WP Crontrol replacement)
     */
    public function run_cron_hook() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_cron_management')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $hook = sanitize_text_field($_POST['hook']);
        
        // Validate it's one of our AMS hooks
        $allowed_hooks = array(
            'ams_product_availability',
            'ams_review_automation_event', 

        );
        
        if (!in_array($hook, $allowed_hooks)) {
            wp_send_json_error(array('message' => __('Invalid hook name.', 'ams-wc-amazon')));
            return;
        }

        // Execute the hook
        do_action($hook);
        
        wp_send_json_success(array(
            'message' => sprintf(__('Hook %s executed successfully!', 'ams-wc-amazon'), $hook)
        ));
    }

    /**
     * Delete/unschedule a cron hook (WP Crontrol replacement)
     */
    public function delete_cron_hook() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_cron_management')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $hook = sanitize_text_field($_POST['hook']);
        
        // Validate it's one of our AMS hooks
        $allowed_hooks = array(
            'ams_product_availability',
            'ams_review_automation_event', 

        );
        
        if (!in_array($hook, $allowed_hooks)) {
            wp_send_json_error(array('message' => __('Invalid hook name.', 'ams-wc-amazon')));
            return;
        }

        // Clear the scheduled hook
        wp_clear_scheduled_hook($hook);
        
        wp_send_json_success(array(
            'message' => sprintf(__('Hook %s unscheduled successfully!', 'ams-wc-amazon'), $hook)
        ));
    }

    /**
     * Schedule a cron hook with custom frequency and time (WP Crontrol replacement)
     */
    public function schedule_cron_hook() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_cron_management')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $hook = sanitize_text_field($_POST['hook']);
        $frequency = sanitize_text_field($_POST['frequency']);
        $time = sanitize_text_field($_POST['time']);
        
        // Validate hook
        $allowed_hooks = array(
            'ams_product_availability',
            'ams_review_automation_event', 

        );
        
        if (!in_array($hook, $allowed_hooks)) {
            wp_send_json_error(array('message' => __('Invalid hook name.', 'ams-wc-amazon')));
            return;
        }

        // Validate frequency
        $schedules = wp_get_schedules();
        if (!isset($schedules[$frequency])) {
            wp_send_json_error(array('message' => __('Invalid frequency.', 'ams-wc-amazon')));
            return;
        }

        // Clear existing schedule
        wp_clear_scheduled_hook($hook);

        // Calculate next run time
        $next_run = time();
        
        if (in_array($frequency, array('daily', 'ams_every_day', 'weekly'))) {
            // For daily/weekly, use specific time
            $time_parts = explode(':', $time);
            $target_hour = intval($time_parts[0]);
            $target_minute = intval($time_parts[1]);
            
            $next_run = strtotime("today {$target_hour}:{$target_minute}:00");
            
            if ($next_run <= time()) {
                // Time has passed today, schedule for tomorrow/next week
                if ($frequency === 'weekly') {
                    $next_run = strtotime("next week {$target_hour}:{$target_minute}:00");
                } else {
                    $next_run = strtotime("tomorrow {$target_hour}:{$target_minute}:00");
                }
            }
        }

        // Schedule the event
        wp_schedule_event($next_run, $frequency, $hook);
        
        wp_send_json_success(array(
            'message' => sprintf(__('Hook %s scheduled successfully!', 'ams-wc-amazon'), $hook),
            'next_run' => $next_run
        ));
    }

    /**
     * Schedule review automation
     */
    private function schedule_review_automation($settings) {
        // Clear existing schedule
        wp_clear_scheduled_hook('ams_review_automation_event');

        // Calculate next run time with defaults
        $frequency = $settings['frequency'] ?? 'weekly';
        $time = $settings['time'] ?? '03:00';
        
        // Calculate next run based on selected time and frequency
        $target_time = strtotime("today {$time}");
        $current_time = time();
        
        // Determine next run based on frequency type
        if (in_array($frequency, ['every_5_minutes', 'every_15_minutes', 'every_30_minutes', 'hourly', 'every_2_hours', 'every_6_hours', 'every_12_hours'])) {
            // For frequent intervals, start immediately or at next interval
            if ($frequency === 'every_5_minutes') {
                $next_run = $current_time + 300; // 5 minutes from now
            } elseif ($frequency === 'every_15_minutes') {
                $next_run = $current_time + 900; // 15 minutes from now
            } elseif ($frequency === 'every_30_minutes') {
                $next_run = $current_time + 1800; // 30 minutes from now
            } elseif ($frequency === 'hourly') {
                $next_run = $current_time + 3600; // 1 hour from now
            } elseif ($frequency === 'every_2_hours') {
                $next_run = $current_time + 7200; // 2 hours from now
            } elseif ($frequency === 'every_6_hours') {
                $next_run = $current_time + 21600; // 6 hours from now
            } elseif ($frequency === 'every_12_hours') {
                $next_run = $current_time + 43200; // 12 hours from now
            }
        } elseif ($frequency === 'custom') {
            // Handle custom frequency
            $custom_interval = $settings['custom_interval'] ?? 60;
            $custom_unit = $settings['custom_unit'] ?? 'minutes';
            
            if ($custom_unit === 'minutes') {
                $next_run = $current_time + ($custom_interval * 60);
            } else { // hours
                $next_run = $current_time + ($custom_interval * 3600);
            }
            
            // For custom frequency, we need to register a temporary cron schedule
            $custom_schedule_name = "ams_custom_{$custom_interval}_{$custom_unit}";
            add_filter('cron_schedules', function($schedules) use ($custom_interval, $custom_unit, $custom_schedule_name) {
                if ($custom_unit === 'minutes') {
                    $schedules[$custom_schedule_name] = array(
                        'interval' => $custom_interval * 60,
                        'display' => sprintf(__('Every %d minutes', 'ams-wc-amazon'), $custom_interval)
                    );
                } else {
                    $schedules[$custom_schedule_name] = array(
                        'interval' => $custom_interval * 3600,
                        'display' => sprintf(__('Every %d hours', 'ams-wc-amazon'), $custom_interval)
                    );
                }
                return $schedules;
            });
            
            // Use the custom schedule name for wp_schedule_event
            $frequency = $custom_schedule_name;
        } else {
            // For daily/weekly/monthly, use the specific time
            if ($target_time <= $current_time) {
                // Time has passed today, schedule for next occurrence
                if ($frequency === 'daily') {
                    $next_run = strtotime("tomorrow {$time}");
                } elseif ($frequency === 'weekly') {
                    $next_run = strtotime("next monday {$time}");
                } elseif ($frequency === 'monthly') {
                    $next_run = strtotime("first day of next month {$time}");
                } else {
                    $next_run = strtotime("tomorrow {$time}");
                }
            } else {
                // Target time today hasn't passed yet
                $next_run = $target_time;
            }
        }


        
        wp_schedule_event($next_run, $frequency, 'ams_review_automation_event');
        
        // Enhanced logging with more details
        $log_message = sprintf(
            "Scheduled review automation: Frequency=%s, Time=%s, Next Run=%s (in %s)",
            $frequency,
            $time,
            wp_date('Y-m-d H:i:s', $next_run),
            human_time_diff(time(), $next_run)
        );
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'SCHEDULING', $log_message);
    }

    /**
     * Unschedule review automation
     */
    private function unschedule_review_automation() {
        // Clear all instances of the hook (in case there are duplicates)
        while (wp_next_scheduled('ams_review_automation_event')) {
            wp_clear_scheduled_hook('ams_review_automation_event');
        }
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'SCHEDULING', "Unscheduled review automation");
    }

    /**
     * Execute review automation (called by cron or manual test)
     */
    public function execute_review_automation($is_test = false, $force_run = false) {
        // Record the execution time
        update_option('ams_automation_last_run', current_time('mysql'));
        
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'IMPORT', 'Starting review automation' . ($is_test ? ' (TEST MODE)' : ''));
        
        // Get automation settings
        $settings = get_option('ams_review_automation_settings', array(
            'enabled' => false,
            'limit' => 10,
            'min_rating' => 4,
            'notifications' => false
        ));
        
        // Check if automation is enabled (same pattern as daily cron)
        $enable_review_cron = get_option('enable_review_cron');
        
        if ($enable_review_cron !== '1' && !$is_test && !$force_run) {
            AutomationLogs::log('WARNING', 'REVIEW_IMPORT', 'IMPORT', 'Review automation cron is disabled, skipping');
            return array('status' => 'disabled');
        }

        // Get products to process
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => $is_test ? 3 : -1, // Limit to 3 products for testing
            'meta_query' => array(
                array(
                    'key' => '_ams_product_url',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        $products = get_posts($args);
        $total_products = count($products);
        $processed = 0;
        $successful = 0;
        $failed = 0;
        $total_reviews_imported = 0;
        
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'IMPORT', "Found $total_products products to process");
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if (!$product || $product->get_type() === 'variation') {
                continue;
            }
            
            $processed++;
            
            try {
                // Set up $_POST data for the existing import method
                $original_post = $_POST;
                $_POST = array(
                    'ams_product' => $product->get_id(),
                    'review_limit' => $settings['limit'] ?? 10,
                    'min_rating' => $settings['min_rating'] ?? 4,
                    'action' => 'ams_product_review_import'
                );
                
                // Get ProductReviewImport instance
                $product_review_import = new \Amazon\Affiliate\Admin\ProductReviewImport();
                
                // Call the existing import method with return_data=true
                $result = $product_review_import->product_review_import(true);
                
                // Restore original $_POST
                $_POST = $original_post;
                
                if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                    $successful++;
                    $total_reviews_imported += $result['reviews_imported'] ?? 0;
                    AutomationLogs::log('SUCCESS', 'REVIEW_IMPORT', 'IMPORT', "Successfully processed product {$product->get_id()}, imported " . ($result['reviews_imported'] ?? 0) . " reviews" . ($is_test ? " (TEST)" : ""), '', $product->get_id());
                } else {
                    $failed++;
                    $error_message = is_array($result) ? ($result['message'] ?? 'Unknown error') : 'Invalid response';
                    // Strip HTML tags from error message for cleaner logs
                    $clean_error_message = strip_tags($error_message);
                    AutomationLogs::log('ERROR', 'REVIEW_IMPORT', 'IMPORT', "Failed to process product {$product->get_id()}: {$clean_error_message}" . ($is_test ? " (TEST)" : ""), '', $product->get_id());
                }
                
            } catch (Exception $e) {
                $failed++;
                AutomationLogs::log('ERROR', 'REVIEW_IMPORT', 'IMPORT', "Exception processing product {$product->get_id()}: " . $e->getMessage() . ($is_test ? " (TEST)" : ""), '', $product->get_id());
            } catch (Error $e) {
                $failed++;
                AutomationLogs::log('ERROR', 'REVIEW_IMPORT', 'IMPORT', "Fatal error processing product {$product->get_id()}: " . $e->getMessage() . ($is_test ? " (TEST)" : ""), '', $product->get_id());
            }
            
            // Add delay to prevent server overload
            if (!$is_test) {
                usleep(500000); // 0.5 second delay
            }
        }
        
        $result = array(
            'status' => 'completed',
            'processed' => $processed,
            'successful' => $successful,
            'failed' => $failed,
            'total_reviews_imported' => $total_reviews_imported,
            'timestamp' => current_time('mysql')
        );
        
        // Log results
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'IMPORT', "Completed. Processed: $processed, Successful: $successful, Failed: $failed, Reviews imported: $total_reviews_imported");
        
        // Send email notification if enabled and not test
        if (!$is_test && $settings['notifications']) {
            $this->send_automation_notification($result);
        }
        
        // Update last run timestamp
        update_option('ams_automation_last_run', $result['timestamp']);
        
        // Update statistics
        $current_stats = get_option('ams_automation_statistics', array(
            'total_runs' => 0,
            'total_products_processed' => 0,
            'total_reviews_imported' => 0
        ));
        
        $current_stats['total_runs']++;
        $current_stats['total_products_processed'] += $processed;
        $current_stats['total_reviews_imported'] += $total_reviews_imported;
        
        update_option('ams_automation_statistics', $current_stats);
        
        return $result;
    }

    /**
     * Import reviews for a specific product
     */
    private function import_reviews_for_product($product_id, $settings) {
        // This would use the same logic as the manual review import
        // For now, we'll simulate it
        
        $product = wc_get_product($product_id);
        if (!$product) {
            return array('success' => false, 'message' => 'Product not found', 'reviews_imported' => 0);
        }

        $amazon_url = get_post_meta($product_id, '_ams_product_url', true);
        if (empty($amazon_url)) {
            return array('success' => false, 'message' => 'No Amazon URL', 'reviews_imported' => 0);
        }

        // Here you would call the actual review import logic
        // For now, we'll return a simulated result
        $reviews_imported = rand(0, intval($settings['limit'] ?? 10));
        
        return array(
            'success' => true,
            'message' => 'Reviews imported successfully',
            'reviews_imported' => $reviews_imported
        );
    }

    /**
     * Send email notification about automation results
     */
    private function send_automation_notification($result) {
        $admin_email = get_option('admin_email');
        $site_name = get_bloginfo('name');
        
        $subject = sprintf('[%s] Review Automation Report', $site_name);
        
        $message = "Review Automation Report\n\n";
        $message .= "Date: " . $result['timestamp'] . "\n";
        $message .= "Products Processed: {$result['processed']}\n";
        $message .= "Successful: {$result['successful']}\n";
        $message .= "Failed: {$result['failed']}\n";
        $message .= "Total Reviews Imported: {$result['total_reviews_imported']}\n\n";
        
        if ($result['failed'] > 0) {
            $message .= "Some products failed to process. Please check the error logs for details.\n\n";
        }
        
        $message .= "You can view the automation page here:\n";
        $message .= admin_url('admin.php?page=automation') . "\n";
        
        wp_mail($admin_email, $subject, $message);
        
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'EMAIL', "Email notification sent to $admin_email");
    }

    /**
     * Get automation settings (AJAX handler)
     */
    public function get_automation_settings() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $settings = get_option('ams_review_automation_settings', array(
            'enabled' => false,
            'frequency' => 'weekly',
            'time' => '03:00',
            'limit' => '10',
            'notifications' => true
        ));

        wp_send_json_success($settings);
    }

    /**
     * Save automation settings (AJAX handler)
     */
    public function save_automation_settings() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'ams-wc-amazon')));
            return;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'ams_automation_nonce')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'ams-wc-amazon')));
            return;
        }

        $settings = json_decode(stripslashes($_POST['settings']), true);
        
        if (!$settings) {
            wp_send_json_error(array('message' => __('Invalid settings data.', 'ams-wc-amazon')));
            return;
        }

        // Validate settings
        $valid_frequencies = array('daily', 'weekly', 'monthly');
        $valid_times = array('02:00', '03:00', '04:00', '05:00');
        $valid_limits = array('5', '10', '15');

        if (!in_array($settings['frequency'], $valid_frequencies)) {
            $settings['frequency'] = 'weekly';
        }

        if (!in_array($settings['time'], $valid_times)) {
            $settings['time'] = '03:00';
        }

        if (!in_array($settings['limit'], $valid_limits)) {
            $settings['limit'] = '10';
        }

        update_option('ams_review_automation_settings', $settings);

        // Schedule or unschedule the automation
        if ($settings['enabled']) {
            $this->schedule_automation($settings);
        } else {
            $this->unschedule_automation();
        }

        wp_send_json_success(array('message' => __('Settings saved successfully.', 'ams-wc-amazon')));
    }

    /**
     * Schedule automation
     */
    private function schedule_automation($settings) {
        // Clear existing schedule
        wp_clear_scheduled_hook('ams_automated_review_sync');

        // Calculate next run time with defaults
        $frequency = $settings['frequency'] ?? 'weekly';
        $time = $settings['time'] ?? '03:00';
        
        $next_run = strtotime("tomorrow {$time}");
        
        if ($frequency === 'weekly') {
            $next_run = strtotime("next monday {$time}");
        } elseif ($frequency === 'monthly') {
            $next_run = strtotime("first day of next month {$time}");
        }

        wp_schedule_event($next_run, $frequency, 'ams_automated_review_sync', array($settings));
    }

    /**
     * Unschedule automation
     */
    private function unschedule_automation() {
        wp_clear_scheduled_hook('ams_automated_review_sync');
    }

    /**
     * Execute automated review sync (called by WordPress cron)
     */
    public function execute_automated_sync($settings) {
        error_log('AMS Automated Sync: Starting automated review sync');
        
        // Get automation settings
        $automation_settings = get_option('ams_review_automation_settings', array());
        if (!$automation_settings['enabled']) {
            error_log('AMS Automated Sync: Automation is disabled, skipping');
            return;
        }

        // Get all products with Amazon URLs
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_ams_product_url',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        $products = get_posts($args);
        $total_products = count($products);
        $processed = 0;
        $successful = 0;
        $failed = 0;
        
        error_log("AMS Automated Sync: Found $total_products products to process");
        
        // Get ProductReviewImport instance to use its import method
        $product_review_import = new \Amazon\Affiliate\Admin\ProductReviewImport();
        $total_reviews_imported = 0;
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if (!$product || $product->get_type() === 'variation') {
                continue;
            }
            
            $processed++;
            
            try {
                // Set up $_POST data for the existing import method
                $_POST = array(
                    'ams_product' => $product->get_id(),
                    'review_limit' => $automation_settings['limit'] ?? 10,
                    'min_rating' => $automation_settings['min_rating'] ?? 4,
                    'action' => 'ams_product_review_import'
                );
                
                // Get ProductReviewImport instance
                $product_review_import = new \Amazon\Affiliate\Admin\ProductReviewImport();
                
                // Call the existing import method with return_data=true
                $import_result = $product_review_import->product_review_import(true);
                
                if ($import_result['success']) {
                    $successful++;
                    $reviews_imported = $import_result['reviews_imported'];
                    $total_reviews_imported += $reviews_imported;
                    AutomationLogs::log('SUCCESS', 'REVIEW_IMPORT', 'IMPORT', "Successfully processed product {$product->get_id()}, imported {$reviews_imported} reviews", '', $product->get_id());
                } else {
                    $failed++;
                    AutomationLogs::log('ERROR', 'REVIEW_IMPORT', 'IMPORT', "Failed to process product {$product->get_id()}: " . $import_result['message'], '', $product->get_id());
                }
                
            } catch (Exception $e) {
                $failed++;
                AutomationLogs::log('ERROR', 'REVIEW_IMPORT', 'IMPORT', "Failed to process product {$product->get_id()}: " . $e->getMessage(), '', $product->get_id());
            }
            
            // Add small delay to prevent server overload
            usleep(500000); // 0.5 second delay
        }
        
        // Log results
        AutomationLogs::log('INFO', 'REVIEW_IMPORT', 'IMPORT', "Completed. Processed: $processed, Successful: $successful, Failed: $failed, Reviews imported: $total_reviews_imported");
        
        // Send email notification if enabled
        if ($automation_settings['notifications']) {
            $result = array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'processed' => $processed,
                'successful' => $successful,
                'failed' => $failed,
                'total_reviews_imported' => $total_reviews_imported
            );
            $this->send_automation_notification($result);
        }
        
        // Update last run timestamp
        update_option('ams_automation_last_run', current_time('mysql'));
        
        // Reschedule for next run
        $this->schedule_automation($automation_settings);
    }
}
