<?php
namespace Amazon\Affiliate\Admin;

class BestsellersPage {
    
    public function __construct() {
        add_action('wp_ajax_browse_bestsellers', array($this, 'handle_bestsellers_request'));
        add_action('wp_ajax_save_bestsellers_limit', array($this, 'handle_save_bestsellers_limit'));
    }

    /**
     * Get detailed proxy service status for smart badges
     * Shows each service status with import/update capabilities
     */
    public function get_proxy_services_status() {
        // Get all proxy service configurations with import/update settings
        $services = [
            'decodo' => [
                'name' => 'Decodo',
                'active' => get_option('ams_decodo_is_active', '0'),
                'key' => get_option('ams_decodo_auth_token'),
                'import_enabled' => get_option('ams_decodo_is_active', '0'), // Import = is_active
                'update_enabled' => get_option('ams_decodo_on_update', '0')   // Update = on_update
            ],
            'scraperapi' => [
                'name' => 'ScraperAPI',
                'active' => get_option('ams_scraper_api_is_active', '0'),
                'key' => get_option('ams_scraper_api_key'),
                'import_enabled' => get_option('ams_scraper_api_is_active', '0'), // Import = is_active
                'update_enabled' => get_option('ams_scraper_api_on_update', '0')   // Update = on_update
            ],
            'scrapingant' => [
                'name' => 'ScrapingAnt',
                'active' => get_option('ams_scrapingant_is_active', '0'),
                'key' => get_option('ams_scrapingant_api_key'),
                'import_enabled' => get_option('ams_scrapingant_is_active', '0'), // Import = is_active
                'update_enabled' => get_option('ams_scrapingant_on_update', '0')   // Update = on_update
            ]
        ];

        $service_badges = [];
        $has_any_active = false;

        foreach ($services as $service_key => $service) {
            // A service is configured if it has a key AND either import or update is enabled
            $has_key = !empty($service['key']);
            $import_enabled = ($service['import_enabled'] === '1');
            $update_enabled = ($service['update_enabled'] === '1');
            $is_configured = $has_key && ($import_enabled || $update_enabled);
            
            if ($is_configured) {
                $has_any_active = true;
                
                // Determine import/update symbols
                $symbols = '';
                if ($service['import_enabled'] === '1') {
                    $symbols .= '↓'; // Import symbol
                }
                if ($service['update_enabled'] === '1') {
                    $symbols .= '↻'; // Update symbol
                }
                if (empty($symbols)) {
                    $symbols = '○'; // Configured but no operations
                }
                
                $service_badges[] = [
                    'name' => $service['name'],
                    'status' => 'enabled',
                    'badge_class' => 'bg-success',
                    'icon' => 'bi-shield-check',
                    'symbols' => $symbols,
                    'title' => $service['name'] . ' enabled' . 
                              ($service['import_enabled'] === '1' ? ' (Import)' : '') .
                              ($service['update_enabled'] === '1' ? ' (Update)' : '')
                ];
            } else {
                // Service is disabled or not configured
                $service_badges[] = [
                    'name' => $service['name'],
                    'status' => 'disabled',
                    'badge_class' => 'bg-danger',
                    'icon' => 'bi-shield-x',
                    'symbols' => '✕',
                    'title' => $service['name'] . ' disabled or not configured'
                ];
            }
        }

        return [
            'services' => $service_badges,
            'has_any_active' => $has_any_active,
            'show_setup_notification' => !$has_any_active
        ];
    }
    
    public function bestsellers_page() {
        include dirname(__FILE__) . '/views/bestsellers-page.php';
    }
    
    public function handle_bestsellers_request() {
        $__log_file = plugin_dir_path(__FILE__) . '../import_verification.log';
        @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] Start: func=BestsellersPage::handle_bestsellers_request' . "\n", FILE_APPEND);
        try {
            $nonce = isset($_POST['nonce']) ? $_POST['nonce'] : '';
            
            if (!wp_verify_nonce($nonce, 'ams_without_api')) {
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=BestsellersPage::handle_bestsellers_request; status=failure; error=nonce' . "\n", FILE_APPEND);
                wp_send_json_error('Nonce verification failed');
                return;
            }
            
            $category = isset($_POST['category']) ? sanitize_text_field($_POST['category']) : '';
            $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
            $results_limit = isset($_POST['results_limit']) ? intval($_POST['results_limit']) : 20;
            
            $importProduct = new ImportProducts();
            
            $result = $importProduct->get_bestsellers_results([
                'category' => $category,
                'page' => $page,
                'results_limit' => $results_limit
            ]);
            
            if (isset($result['error'])) {
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=BestsellersPage::handle_bestsellers_request; status=failure' . "\n", FILE_APPEND);
                wp_send_json_error($result['error']);
            } else {
                $items = is_array($result) ? count($result) : 0;
                @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=BestsellersPage::handle_bestsellers_request; status=success; items=' . $items . "\n", FILE_APPEND);
                wp_send_json_success($result);
            }
        } catch (\Throwable $e) {
            @file_put_contents($__log_file, '[' . date('Y-m-d H:i:s') . '] End: func=BestsellersPage::handle_bestsellers_request; status=failure; error=' . $e->getMessage() . "\n", FILE_APPEND);
            wp_send_json_error('Unexpected error');
        }
    }
    
    public function handle_save_bestsellers_limit() {
        $nonce = isset($_POST['nonce']) ? $_POST['nonce'] : '';
        if (!wp_verify_nonce($nonce, 'ams_without_api')) {
            wp_send_json_error('Nonce verification failed');
            return;
        }
        
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 20;
        
        // Validate limit
        $allowed_limits = [10, 20, 50, 100];
        if (!in_array($limit, $allowed_limits)) {
            wp_send_json_error('Invalid results limit');
            return;
        }
        
        // Save the setting
        update_option('ams_bestsellers_limit', $limit);
        
        wp_send_json_success(['message' => 'Results limit saved successfully', 'limit' => $limit]);
    }
}
