<?php

namespace Amazon\Affiliate\Admin;
use Amazon\Affiliate\Admin;


/**
 * Class Productsearch
 *
 * @package Amazon\Affiliate\Admin
 */


class Productsearch extends ImportProducts

{

    public function product_page() {
        $this->get_amazon_cat();
        $this->get_wc_terms();
        $template = __DIR__ . '/views/product.php';
        if (file_exists($template)) {
            require_once $template;
        }
    }

    public function get_option($name) {
        $option = get_option($name);
        return $option;
    }

    public function get_wc_terms() {
        $categories = get_terms(array(
            'hide_empty' => false,
        ));

        $cat = array();

        foreach ($categories as $row) {
            if ('product_cat' === $row->taxonomy) {
                $cat[] = array(
                    'term_id' => $row->term_id,
                    'name' => $row->name,
                );
            }
        }
        return array_reverse($cat);

    }

    public function active_site() {
        $ams_activated_site = get_option('ams_activated_site');
        $url = $_SERVER['HTTP_HOST'];

        if (strtolower($ams_activated_site) != strtolower($url)) {
            $license = sprintf("<h4 class='wca-warning'>%s</h4>", esc_html__('You have cloned the website Please activate the license on the website.', 'ams-wc-amazon'));
            echo wp_kses_post($license);
            wp_die();
        }
    }

    public function search_products() {

        

        
        wca_add_products_search_count();

        $results = [];

        if ('keyword' === $_POST['wca_search_by']) {
            // Handle both old and new parameter names
            $keyword = sanitize_text_field($_POST['wca_keyword'] ?? $_POST['keyword'] ?? '');
            $sort_by = sanitize_text_field($_POST['wca_sort_by'] ?? $_POST['sort_by'] ?? 'Relevance');
            $item_page = intval($_POST['wca_item_page'] ?? $_POST['item_page'] ?? 1);
            $amazon_cat = sanitize_text_field($_POST['wca_amazon_cat'] ?? $_POST['ams_amazon_cat'] ?? 'All');
            

            
            // Ensure item_page is at least 1
            if ($item_page < 1) {
                $item_page = 1;
            }
            
            $results = $this->get_keyword_products($keyword, $item_page, $sort_by, $amazon_cat);
            $results = is_array($results) ? $results : [];
        } else {
            $asin_id = sanitize_text_field($_POST['asin_id']);
            $searched_asins = array_values(array_filter(array_map('trim', explode(',', $asin_id)), 'strlen'));
            
            $valid_asins = array_filter($searched_asins, function($asin) {
                return strlen($asin) == 10 && preg_match('/^[0-9A-Z]{10}$/', $asin);
            });

            if (empty($valid_asins)) {
                $this->display_no_results_message("No valid ASINs found. Please check your input and try again.");

                wp_die();
            }

            $valid_asins = array_slice($valid_asins, 0, 10);
            $current_page = isset($_POST['page']) ? intval($_POST['page']) : 1;
            $per_page = 10;
            $offset = ($current_page - 1) * $per_page;
            $current_page_asins = array_slice($valid_asins, $offset, $per_page);

            $results = $this->get_item_id_products(implode(',', $current_page_asins));
        }


        if (empty($results)) {
            $this->display_no_results_message();
        } else {
            foreach ($results as $row) {
                $this->display_product_card($row);
            }

            // Add pagination information if necessary
            if (isset($valid_asins)) {
                $total_pages = ceil(count($valid_asins) / $per_page);
                echo '<div class="pagination-info" data-total-pages="' . $total_pages . '" data-current-page="' . $current_page . '"></div>';

                if (count($valid_asins) < count($searched_asins)) {
                    $skipped_count = count($searched_asins) - count($valid_asins);
                    echo '<div class="alert alert-info" role="alert">' . $skipped_count . ' invalid ASIN(s) were skipped.</div>';
                }
            }
        }

        wp_die();
    }

    private function display_no_results_message($custom_message = null) {
        $message = $custom_message ?? "We couldn't find any products matching your search criteria. Please try adjusting your search terms or filters.";
        ?>
        <div class="col-12">
            <div class="alert alert-info" role="alert">
                <h4 class="alert-heading">No Results Found</h4>
                <p><?php echo esc_html($message); ?></p>
                <hr>
                <p class="mb-0">Tips: 
                    <ul>
                        <li>Check your spelling</li>
                        <li>Try more general keywords</li>
                        <li>Adjust your price range</li>
                        <li>Try a different category</li>
                        <li>Adjust your star rating filter</li>
                    </ul>
                </p>
            </div>
        </div>
        <?php
    }

    private function display_product_card($row) {
        $asin = $row->ASIN;
        $parentASIN = isset($row->ParentASIN) ? $row->ParentASIN : null;
        $detail_page_url = $row->DetailPageURL;
        $image = $row->Images->Primary->Medium->URL ?? '';
        $amount = $row->Offers->Listings[0]->Price->DisplayAmount ?? $row->Offers->Summaries[0]->LowestPrice->DisplayAmount ?? '';
        $saving_amount = $row->Offers->Listings[0]->SavingBasis->DisplayAmount ?? '';
        $title = $row->ItemInfo->Title->DisplayValue ?? '';

        // Use parent ASIN for import if available, otherwise use the product's ASIN
        $import_asin = $parentASIN ?: $asin;
        
        // Calculate discount percentage if available (DYNAMIC)
        $discount_percentage = '';
        if (!empty($amount) && !empty($saving_amount)) {
            $current_price = floatval(preg_replace('/[^\d.]/', '', $amount));
            $original_price = floatval(preg_replace('/[^\d.]/', '', $saving_amount));
            if ($original_price > $current_price) {
                $discount_percentage = round((($original_price - $current_price) / $original_price) * 100);
            }
        }
        
        // Check Prime eligibility (DYNAMIC) - PA-API 5 structure
        $is_prime_eligible = false;
        
        // PA-API 5: Check primary listing for Prime eligibility
        if (isset($row->Offers->Listings[0]->DeliveryInfo->IsPrimeEligible)) {
            $is_prime_eligible = (bool) $row->Offers->Listings[0]->DeliveryInfo->IsPrimeEligible;
        }
        // PA-API 5: Check ProgramEligibility for Prime exclusives
        elseif (isset($row->Offers->Listings[0]->ProgramEligibility->IsPrimeExclusive)) {
            $is_prime_eligible = (bool) $row->Offers->Listings[0]->ProgramEligibility->IsPrimeExclusive;
        }
        // Fallback: Check all listings for Prime eligibility
        elseif (isset($row->Offers->Listings) && is_array($row->Offers->Listings)) {
            foreach ($row->Offers->Listings as $listing) {
                if ((isset($listing->DeliveryInfo->IsPrimeEligible) && $listing->DeliveryInfo->IsPrimeEligible) ||
                    (isset($listing->ProgramEligibility->IsPrimeExclusive) && $listing->ProgramEligibility->IsPrimeExclusive)) {
                    $is_prime_eligible = true;
                    break;
                }
            }
        }
        
        // PA-API 5: Check Amazon fulfillment (often indicates Prime eligibility)
        $is_amazon_fulfilled = isset($row->Offers->Listings[0]->DeliveryInfo->IsAmazonFulfilled) ? 
                              (bool) $row->Offers->Listings[0]->DeliveryInfo->IsAmazonFulfilled : false;
        
        // Debug: Log Prime eligibility data (remove this after testing)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("ASIN: $asin - Prime Eligible: " . ($is_prime_eligible ? 'YES' : 'NO'));
            if (isset($row->Offers->Listings[0]->DeliveryInfo)) {
                error_log("DeliveryInfo available for ASIN: $asin");
            } else {
                error_log("No DeliveryInfo for ASIN: $asin");
            }
        }
        
        // PA-API 5: Get additional product information
        $brand = isset($row->ItemInfo->ByLineInfo->Brand->DisplayValue) ? 
                $row->ItemInfo->ByLineInfo->Brand->DisplayValue : '';
        $manufacturer = isset($row->ItemInfo->ByLineInfo->Manufacturer->DisplayValue) ? 
                       $row->ItemInfo->ByLineInfo->Manufacturer->DisplayValue : '';
        
        // PA-API 5: Check availability status
        $availability_type = isset($row->Offers->Listings[0]->Availability->Type) ? 
                           $row->Offers->Listings[0]->Availability->Type : '';
        $availability_message = isset($row->Offers->Listings[0]->Availability->Message) ? 
                              $row->Offers->Listings[0]->Availability->Message : '';
        
        // PA-API 5: Check if it's a bestseller or has promotions
        $has_promotions = isset($row->Offers->Listings[0]->Promotions) && 
                         !empty($row->Offers->Listings[0]->Promotions);
        
        // PA-API 5: Check condition (New, Used, etc.)
        $condition = isset($row->Offers->Listings[0]->Condition->Value) ? 
                    $row->Offers->Listings[0]->Condition->Value : 'New';
        
        // Check if product is trending/popular based on offer count (DYNAMIC)
        $offer_count = isset($row->Offers->Summaries[0]->OfferCount) ? 
                      intval($row->Offers->Summaries[0]->OfferCount) : 0;
        // Trending logic based on offer count only (since reviews not available in PA-API 5)
        $is_trending = ($offer_count > 15); // High offer count indicates popularity
        ?>
        <div class="col-xl-3 col-lg-4 col-md-6 col-12 mb-4">
            <div class="card h-100 shadow-sm border-0 overflow-hidden product-card">
                <!-- Image Section - Fixed Height -->
                <div class="position-relative bg-white d-flex align-items-center justify-content-center" style="height: 220px; background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);">
                    <!-- PA-API 5: Dynamic badges based on actual data -->
                    <?php if (!empty($discount_percentage)) : ?>
                        <span class="badge bg-danger position-absolute top-0 start-0 m-2 px-2 py-1 rounded-pill shadow-sm">
                            <i class="bi bi-percent me-1"></i><?php echo esc_html($discount_percentage); ?>% OFF
                        </span>
                    <?php endif; ?>
                    
                    <?php if ($has_promotions) : ?>
                        <span class="badge bg-success position-absolute top-0 end-0 m-2 px-2 py-1 rounded-pill shadow-sm">
                            <i class="bi bi-gift me-1"></i>Promotion
                        </span>
                    <?php elseif ($is_trending) : ?>
                        <span class="badge bg-warning text-dark position-absolute top-0 end-0 m-2 px-2 py-1 rounded-pill shadow-sm">
                            <i class="bi bi-graph-up me-1"></i>Trending
                        </span>
                    <?php endif; ?>
                    
                    <?php if ($condition !== 'New' && !empty($condition)) : ?>
                        <span class="badge bg-info position-absolute bottom-0 start-0 m-2 px-2 py-1 rounded-pill shadow-sm">
                            <?php echo esc_html($condition); ?>
                        </span>
                    <?php endif; ?>
                    <?php if (!empty($image)) : ?>
                        <img src="<?php echo esc_attr($image); ?>" alt="<?php echo esc_attr($title); ?>" class="img-fluid" style="max-height: 180px; max-width: 180px; object-fit: contain;">
                    <?php else : ?>
                        <div class="bg-secondary text-white rounded d-flex align-items-center justify-content-center" style="width: 150px; height: 150px;">
                            <i class="bi bi-image fs-1"></i>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Card Body - Flex Column for Equal Heights -->
                <div class="card-body p-3 d-flex flex-column bg-white">
                    <!-- Brand Section - Fixed Height -->
                    <div class="mb-2" style="min-height: 28px;">
                        <?php if (!empty($brand)) : ?>
                        <span class="badge bg-primary bg-opacity-10 text-primary border border-primary border-opacity-25 fw-semibold brand-section" data-brand="<?php echo esc_attr($brand); ?>">
                            <i class="bi bi-tag me-1"></i><?php echo esc_html($brand); ?>
                        </span>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Title Section - Fixed Height -->
                    <h6 class="card-title text-dark mb-2 lh-sm fw-semibold" style="height: 2.8rem; overflow: hidden; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical;">
                        <a href="<?php echo esc_attr($detail_page_url); ?>" target="_blank" title="<?php echo esc_attr($title); ?>" class="text-decoration-none text-dark">
                            <?php echo esc_html(strlen($title) > 55 ? substr($title, 0, 55) . '...' : $title); ?>
                        </a>
                    </h6>
                    
                    <!-- Price Section - Fixed Height -->
                    <div class="mb-3" style="min-height: 32px;">
                        <?php if(!empty($amount)) : ?>
                            <div class="d-flex align-items-center flex-wrap">
                                <span class="fw-bold text-success fs-5 me-2"><?php echo esc_html($amount); ?></span>
                                <?php if (!empty($saving_amount)) : ?>
                                    <small class="text-muted text-decoration-line-through"><?php echo esc_html($saving_amount); ?></small>
                                <?php endif; ?>
                            </div>
                        <?php else : ?>
                            <span class="text-muted small">
                                <i class="bi bi-exclamation-circle me-1"></i>Price Not Available
                            </span>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Availability Section - Fixed Height -->
                    <div class="mb-2" style="min-height: 20px;">
                        <?php if (!empty($availability_message) && $availability_type !== 'Now') : ?>
                        <small class="text-warning">
                            <i class="bi bi-clock me-1"></i><?php echo esc_html($availability_message); ?>
                        </small>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Action Buttons Section - Fixed Height -->
                    <div class="d-flex align-items-center justify-content-between mb-3" style="min-height: 40px;">
                        <a href="<?php echo esc_attr($detail_page_url); ?>" target="_blank" class="btn btn-outline-primary btn-sm rounded-pill">
                            <i class="bi bi-eye me-1"></i>View Product
                        </a>
                        <div class="d-flex gap-1">
                            <?php if ($is_prime_eligible) : ?>
                                <span class="badge bg-primary rounded-pill px-2 py-1">
                                    <i class="bi bi-lightning-charge me-1"></i>Prime
                                </span>
                            <?php elseif ($is_amazon_fulfilled) : ?>
                                <span class="badge bg-info rounded-pill px-2 py-1">
                                    <i class="bi bi-truck me-1"></i>Amazon
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Import Button - Always at Bottom -->
                    <div class="mt-auto">
                    <?php
                        $ams_all_asin = ams_get_all_products_info_with_parent();
                        if (in_array($import_asin, $ams_all_asin['asin'])) {
                        ?>
                            <button disabled type="button" class="btn btn-success btn-sm w-100 rounded-pill shadow-sm wca-add-to-imported">
                                <i class="bi bi-check-circle me-1"></i>
                                <?php echo esc_html__('Already Imported', 'ams-wc-amazon'); ?>
                            </button>
                        <?php
                        } else {
                        ?>
                            <button type="button" class="btn btn-primary btn-sm w-100 rounded-pill shadow-sm wca-add-to-import" data-asin="<?php echo esc_attr($import_asin); ?>">
                                <i class="bi bi-download me-1"></i>
                                <?php echo esc_html__('Import Product', 'ams-wc-amazon'); ?>
                            </button>
                        <?php
                        }
                    ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    private function display_not_found_card($asin) {
        ?>
        <div class="col-xl-3 col-lg-4 col-md-6 col-12">
            <div class="card h-100 shadow-sm">
                <div class="position-relative bg-light d-flex align-items-center justify-content-center" style="height: 200px;">
                    <div class="bg-secondary text-white rounded d-flex align-items-center justify-content-center" style="width: 150px; height: 150px;">
                        <i class="bi bi-exclamation-triangle fs-1"></i>
                    </div>
                </div>
                <div class="card-body p-3 text-center">
                    <h6 class="card-title text-warning mb-2">Product Not Found</h6>
                    <p class="text-muted mb-2">ASIN: <?php echo esc_html($asin); ?></p>
                    <p class="text-muted small">This product may no longer be available or may have incomplete data.</p>
                    <button disabled type="button" class="btn btn-outline-secondary btn-sm w-100">
                        <i class="bi bi-x-circle me-1"></i>
                        Not Available
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    public function get_item_id_products($asin_id) {

        $space_remove_asin_id = str_replace(' ', '', $asin_id);
        $array_asin_id = explode(',', $space_remove_asin_id);
        $locale = get_option('ams_amazon_country');
        $regions = ams_get_amazon_regions();
        $marketplace = ams_get_marketplace_value($locale);
        $serviceName = 'ProductAdvertisingAPI';
        $region = $regions[$locale]['RegionCode'];
        $accessKey = get_option('ams_access_key_id');
        $secretKey = get_option('ams_secret_access_key');
        $payloadArr = array();
        $payloadArr['ItemIds'] = $array_asin_id;
        $payloadArr['Resources'] = array("Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large", "ItemInfo.ByLineInfo", "ItemInfo.ContentInfo", "ItemInfo.ContentRating", "ItemInfo.Classifications", "ItemInfo.ExternalIds", "ItemInfo.Features", "ItemInfo.ManufactureInfo", "ItemInfo.ProductInfo", "ItemInfo.TechnicalInfo", "ItemInfo.Title", "ItemInfo.TradeInInfo", "Offers.Listings.Availability.MaxOrderQuantity", "Offers.Listings.Availability.Message", "Offers.Listings.Availability.MinOrderQuantity", "Offers.Listings.Availability.Type", "Offers.Listings.Condition", "Offers.Listings.Condition.ConditionNote", "Offers.Listings.Condition.SubCondition", "Offers.Listings.DeliveryInfo.IsAmazonFulfilled", "Offers.Listings.DeliveryInfo.IsFreeShippingEligible", "Offers.Listings.DeliveryInfo.IsPrimeEligible", "Offers.Listings.DeliveryInfo.ShippingCharges", "Offers.Listings.IsBuyBoxWinner", "Offers.Listings.LoyaltyPoints.Points", "Offers.Listings.MerchantInfo", "Offers.Listings.Price", "Offers.Listings.ProgramEligibility.IsPrimeExclusive", "Offers.Listings.ProgramEligibility.IsPrimePantry", "Offers.Listings.Promotions", "Offers.Listings.SavingBasis", "Offers.Summaries.HighestPrice", "Offers.Summaries.LowestPrice", "Offers.Summaries.OfferCount");
        $payloadArr['PartnerTag'] = get_option('ams_associate_tag');
        $payloadArr['PartnerType'] = 'Associates';
        $payloadArr['Marketplace'] = $marketplace;
        $payloadArr['Operation'] = 'GetItems';
        $payload = json_encode($payloadArr);
        $host = $regions[$locale]['Host'];
        $uri_path = "/paapi5/getitems";
        $api = new \Amazon\Affiliate\Api\Amazon_Product_Api($accessKey, $secretKey, $region, $serviceName, $uri_path, $payload, $host, 'GetItems');
        $response = $api->do_request();
        $results = @$response->ItemsResult->Items;
        return $results;
    }

    private function is_valid_product($item) {
        $valid = isset($item->ASIN) &&
                 isset($item->ItemInfo->Title->DisplayValue) &&
                 isset($item->Images->Primary->Medium->URL) &&
                 (
                     (isset($item->Offers->Listings) && !empty($item->Offers->Listings)) ||
                     (isset($item->Offers->Summaries) && !empty($item->Offers->Summaries))
                 );

        if (!$valid) {
            $missing = array();
            if (!isset($item->ASIN)) $missing[] = 'ASIN';
            if (!isset($item->ItemInfo->Title->DisplayValue)) $missing[] = 'Title';
            if (!isset($item->Images->Primary->Medium->URL)) $missing[] = 'Image URL';
            if (!(isset($item->Offers->Listings) && !empty($item->Offers->Listings)) &&
                !(isset($item->Offers->Summaries) && !empty($item->Offers->Summaries))) {
                $missing[] = 'Offers';
            }
        }

        return $valid;
    }

    public function get_keyword_products($keyword, $item_page, $sort_by, $amazon_cat) {

        
        $ams_product_per_page = get_option('ams_product_per_page');
        // Ensure ItemCount is never 0 (Amazon API requirement)
        if (empty($ams_product_per_page) || (int) $ams_product_per_page < 1) {
            $ams_product_per_page = 10; // Default fallback
        }
        
        $locale = get_option('ams_amazon_country');
        $regions = ams_get_amazon_regions();
        $marketplace = ams_get_marketplace_value($locale);
        $serviceName = 'ProductAdvertisingAPI';
        $region = $regions[$locale]['RegionCode'];
        $accessKey = get_option('ams_access_key_id');
        $secretKey = get_option('ams_secret_access_key');
        $payloadArr = array();
        $payloadArr['Keywords'] = $keyword;
        $payloadArr['Resources'] = array('Images.Primary.Small', 'Images.Primary.Medium', 'Images.Primary.Large', 'Images.Variants.Small', 'Images.Variants.Medium', 'Images.Variants.Large', 'ItemInfo.ByLineInfo', 'ItemInfo.ContentInfo', 'ItemInfo.ContentRating', 'ItemInfo.Classifications', 'ItemInfo.ExternalIds', 'ItemInfo.Features', 'ItemInfo.ManufactureInfo', 'ItemInfo.ProductInfo', 'ItemInfo.TechnicalInfo', 'ItemInfo.Title', 'ItemInfo.TradeInInfo', 'Offers.Listings.Availability.MaxOrderQuantity', 'Offers.Listings.Availability.Message', 'Offers.Listings.Availability.MinOrderQuantity', 'Offers.Listings.Availability.Type', 'Offers.Listings.Condition', 'Offers.Listings.Condition.SubCondition', 'Offers.Listings.DeliveryInfo.IsAmazonFulfilled', 'Offers.Listings.DeliveryInfo.IsFreeShippingEligible', 'Offers.Listings.DeliveryInfo.IsPrimeEligible', 'Offers.Listings.DeliveryInfo.ShippingCharges', 'Offers.Listings.IsBuyBoxWinner', 'Offers.Listings.LoyaltyPoints.Points', 'Offers.Listings.MerchantInfo', 'Offers.Listings.Price', 'Offers.Listings.ProgramEligibility.IsPrimeExclusive', 'Offers.Listings.ProgramEligibility.IsPrimePantry', 'Offers.Listings.Promotions', 'Offers.Listings.SavingBasis', 'Offers.Summaries.HighestPrice', 'Offers.Summaries.LowestPrice', 'Offers.Summaries.OfferCount', 'ParentASIN', 'SearchRefinements');
        $payloadArr["ItemCount"] = (int) $ams_product_per_page;
        $payloadArr["ItemPage"] = (int) $item_page;
        $payloadArr["SortBy"] = $sort_by;
        $payloadArr["SearchIndex"] = $amazon_cat;
        
        // PA-API 5 Supported Filters
        if (!empty($_POST['api_brand'])) {
            $payloadArr['Brand'] = sanitize_text_field($_POST['api_brand']);
        }
        
        if (!empty($_POST['api_min_price']) || !empty($_POST['api_max_price'])) {
            // Convert prices to cents (PA-API 5 requirement)
            if (!empty($_POST['api_min_price'])) {
                $payloadArr['MinPrice'] = (int) (floatval($_POST['api_min_price']) * 100);
            }
            if (!empty($_POST['api_max_price'])) {
                $payloadArr['MaxPrice'] = (int) (floatval($_POST['api_max_price']) * 100);
            }
        }
        
        if (!empty($_POST['api_condition'])) {
            $payloadArr['Condition'] = sanitize_text_field($_POST['api_condition']);
        }
        
        if (!empty($_POST['api_delivery_flags'])) {
            $payloadArr['DeliveryFlags'] = array(sanitize_text_field($_POST['api_delivery_flags']));
        }
        
        $payloadArr['PartnerTag'] = get_option('ams_associate_tag');
        $payloadArr['PartnerType'] = 'Associates';
        $payloadArr['Availability'] = 'Available';
        $payloadArr['Marketplace'] = $marketplace;
        $payload = json_encode($payloadArr);
        $host = $regions[$locale]['Host'];
        $uri_path = "/paapi5/searchitems";
        $api = new \Amazon\Affiliate\Api\Amazon_Product_Api($accessKey, $secretKey, $region, $serviceName, $uri_path, $payload, $host, 'SearchItems');
        $response = $api->do_request();
        

        
        $results = isset($response->SearchResult) && $response->SearchResult ? $response->SearchResult->Items : [];
        return $results;
    }

    public function get_amazon_cat() {
        $all_country_cat = ams_amazon_departments();
        $country = get_option('ams_amazon_country');
        $cat = $all_country_cat[$country];
        return $cat;
    }

    public function import_products() {

        $nonce = sanitize_text_field($_POST['_wpnonce']);
        
        // Check if nonce is invalid
        if (!wp_verify_nonce($nonce, 'wca_search_product')) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('Invalid nonce, please refresh your screen and try again.', 'ams-wc-amazon')
            );

            wp_send_json_error(new \WP_Error("403", $message));
        }

        if (!isset($_FILES["csv"]) || $_FILES["csv"]["error"] !== UPLOAD_ERR_OK) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('There is no file to upload or an error occurred during upload.', 'ams-wc-amazon')
            );

            wp_send_json_error(new \WP_Error("400", $message));
        }

        $filepath = $_FILES['csv']['tmp_name'];

        if (!file_exists($filepath) || !is_readable($filepath)) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('The uploaded file is not accessible.', 'ams-wc-amazon')
            );
            wp_send_json_error(new \WP_Error("400", $message));
        }

        $fileSize = filesize($filepath);

        if ($fileSize === 0) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('The file is empty.', 'ams-wc-amazon')
            );
            wp_send_json_error(new \WP_Error("400", $message));
        }

        $maxSize = 1024 * 1024 * 1; // 1MB
        if ($fileSize > $maxSize) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('File size must be less than 1MB.', 'ams-wc-amazon')
            );
            wp_send_json_error(new \WP_Error("400", $message));
        }

        $filetype = $this->get_file_type($filepath);
        $allowedTypes = array(
            'text/tsv', 'text/csv', 'text/plain', 'application/vnd.ms-excel'
        );

        if (!in_array($filetype, $allowedTypes)) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('Please select a file with a valid file type.', 'ams-wc-amazon')
            );
            wp_send_json_error(new \WP_Error("400", $message));
        }

        // Declare the delimiter
        $delimiter = ',';
        // Open the file in read mode
        $csv = fopen($filepath, 'r');
        $rows = [];
        $row_number = 0;
        $headers = fgetcsv($csv, 0, $delimiter);
        while ($csv_row = fgetcsv($csv, 0, $delimiter)) {
            // Increment Row Number
            $row_number++;
            if (!empty($csv_row[0])) {
                $rows[] = trim($csv_row[0]); // Add first cell only
            }
            if ($row_number >= 100) break;
        }
        fclose($csv);

        $rows = array_filter($rows);
        // Remove duplicate skus
        $rows = array_unique($rows, SORT_REGULAR);
        
        // Return json with success

        wp_send_json_success(array_chunk($rows, 10), 200);
    }

    private function get_file_type($filepath) {
        if (function_exists('finfo_open')) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            if ($finfo !== false) {
                $filetype = finfo_file($finfo, $filepath);
                finfo_close($finfo);
                return $filetype;
            }
        }

        // Fallback to mime_content_type if finfo is not available
        if (function_exists('mime_content_type')) {
            return mime_content_type($filepath);
        }

        // Last resort: check file extension
        $extension = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
        $mime_types = array(
            'csv' => 'text/csv',
            'tsv' => 'text/tsv',
            'txt' => 'text/plain',
            'xls' => 'application/vnd.ms-excel'
        );

        return isset($mime_types[$extension]) ? $mime_types[$extension] : 'application/octet-stream';
    }


    /**
     * Extract all amazon asin from CSV
     */
    public function wca_import_process() {

        // Response Defaults

        $aResponse = [];
        $aResponse['success'] = [];
        $aResponse['imported'] = [];
        
        // POST Data
        $asin = sanitize_text_field($_POST['asin']);
        $nonce = sanitize_text_field($_POST['nonce']);
        
        // Check if nonce is invalid
        if ( !wp_verify_nonce( $nonce, 'ams_product_import' ) ) {
            $message = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('Invalid nonce, please refresh your screen and try again.', 'ams-wc-amazon')
            );
            $aResponse['message'] = $message;
            $aResponse['cancelled'] = explode(",", $asin);

            wp_send_json_error( $aResponse );
        }

        // Check if plugin license status is active or not
        if (ams_plugin_license_status() === false ) {
            $license = sprintf(
                '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                esc_html__('Please activate your license to enable full plugin functionality.', 'ams-wc-amazon')
            );
            $aResponse['message'] = $license;
            $aResponse['cancelled'] = explode(",", $asin);
            wp_send_json_error( $aResponse );
        }
        
        // AMAZON PAAPI
        $asins          = explode(",", $asin);
        $regions        = ams_get_amazon_regions();
        $locale         = get_option( 'ams_amazon_country' );
        $marketplace    = ams_get_marketplace_value($locale);
        $service_name   = 'ProductAdvertisingAPI';
        $region         = $regions[ $locale ]['RegionCode'] ?? '';

        // AMAZON PAAPI Credentials
        $access_key     = get_option( 'ams_access_key_id' );
        $secret_key     = get_option( 'ams_secret_access_key' );
        
        // AMAZON PAAPI Payload
        $payload_arr                = array();
        $payload_arr['ItemIds']     = $asins;
        $payload_arr['Resources']   = array( "ParentASIN", "Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large", "ItemInfo.ByLineInfo", "ItemInfo.ContentInfo", "ItemInfo.ContentRating", "ItemInfo.Classifications", "ItemInfo.ExternalIds", "ItemInfo.Features", "ItemInfo.ManufactureInfo", "ItemInfo.ProductInfo", "ItemInfo.TechnicalInfo", "ItemInfo.Title", "ItemInfo.TradeInInfo", "Offers.Listings.Availability.MaxOrderQuantity", "Offers.Listings.Availability.Message", "Offers.Listings.Availability.MinOrderQuantity", "Offers.Listings.Availability.Type", "Offers.Listings.Condition", "Offers.Listings.Condition.ConditionNote", "Offers.Listings.Condition.SubCondition", "Offers.Listings.DeliveryInfo.IsAmazonFulfilled", "Offers.Listings.DeliveryInfo.IsFreeShippingEligible", "Offers.Listings.DeliveryInfo.IsPrimeEligible", "Offers.Listings.DeliveryInfo.ShippingCharges", "Offers.Listings.IsBuyBoxWinner", "Offers.Listings.LoyaltyPoints.Points", "Offers.Listings.MerchantInfo", "Offers.Listings.Price", "Offers.Listings.ProgramEligibility.IsPrimeExclusive", "Offers.Listings.ProgramEligibility.IsPrimePantry", "Offers.Listings.Promotions", "Offers.Listings.SavingBasis", "Offers.Summaries.HighestPrice", "Offers.Summaries.LowestPrice", "Offers.Summaries.OfferCount" );
        $payload_arr['PartnerTag']  = get_option( 'ams_associate_tag' );
        $payload_arr['PartnerType'] = 'Associates';
        $payload_arr['Marketplace'] = $marketplace;
        $payload_arr['Operation']   = 'GetItems';
        $payload                    = wp_json_encode( $payload_arr );
        $host = $regions[ $locale ]['Host'];
        $uri_path = "/paapi5/getitems";
        $api = new \Amazon\Affiliate\Api\Amazon_Product_Api( $access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems' );
        $response = $api->do_request();
        
        // Check if ItemsResult is set and not null before accessing Items
        if( isset($response->ItemsResult) && isset($response->ItemsResult->Items) ) {
            $results = $response->ItemsResult->Items;
        } else {
            $results = null;
        }
        
        if( empty($results) ) {
            sleep(5);
            $apiRetried = new \Amazon\Affiliate\Api\Amazon_Product_Api($access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems');
            $response = $apiRetried->do_request();
        
            // Check if ItemsResult is set and not null before accessing Items
            if( isset($response->ItemsResult) && isset($response->ItemsResult->Items) ) {
                $results = $response->ItemsResult->Items;
            } else {
                $results = null;
            }
        
            if( empty($results) ) {            
                // Send failed
                $message = sprintf(
                    '<div class="alert alert-warning mx-3 text-left" role="alert">%s</div>', 
                    esc_html__('API connection error.', 'ams-wc-amazon')
                );
                $aResponse['message'] = $message;
                $aResponse['cancelled'] = explode(",", $asin);
                wp_send_json_error( $aResponse );
                // wp_send_json_error( new \WP_Error( '403', 'API connection error.' ) );
            }
        }

        // $ams_all_asin = ams_get_all_products_info();
        $ams_all_asin = ams_get_all_products_info_with_parent();

        foreach( $results as $row ) {
            $asin            = $row->ASIN;
            $parentASIN = isset($row->ParentASIN) ? $row->ParentASIN : $asin;
            $detail_page_url = $row->DetailPageURL;
            $thumbnail_size  = get_option('ams_product_thumbnail_size');
            
            if( !empty($ams_all_asin) && !empty($ams_all_asin['asin']) && ( in_array($asin, $ams_all_asin['asin']) || in_array($parentASIN, $ams_all_asin['asin']) ) ) {
                // Already imported or not found sku
                array_push($aResponse['imported'], $asin);
                continue;
            }

            // Check if specific thumbnail size exists, otherwise get the first key
            if(isset($row->Images->Primary->{$thumbnail_size})) {
                $image = $row->Images->Primary->{$thumbnail_size}->URL;
            } else {
                $primaryImages = (array) $row->Images->Primary;
                $key = key($primaryImages);
                $image = $row->Images->Primary->{$key}->URL;
            }

            $gallery = isset($row->Images->Variants) ? $row->Images->Variants : [];

            // Safe check for Amount
            if (isset($row->Offers->Listings[0]->Price->Amount)) {
                $amount = $row->Offers->Listings[0]->Price->Amount;
            } else {
                $amount = null; // or some default value
            }

            // Safe check for SavingBasis Amount
            if (isset($row->Offers->Listings[0]->SavingBasis) && isset($row->Offers->Listings[0]->SavingBasis->Amount)) {
                $saving_amount = $row->Offers->Listings[0]->SavingBasis->Amount;
            } else {
                $saving_amount = null; // or some default value
            }

            $title          = $row->ItemInfo->Title->DisplayValue;
            $product_status = isset($row->Offers->Listings[0]->Availability->Message) ? $row->Offers->Listings[0]->Availability->Message : '';
            $product_status = !empty($product_status) ? 'instock' : 'outofstock';
            
            // Safe check for Features DisplayValues
            if (isset($row->ItemInfo->Features) && isset($row->ItemInfo->Features->DisplayValues)) {
                $features = $row->ItemInfo->Features->DisplayValues;
            } else {
                $features = []; // Assign an empty array or some default value if Features or DisplayValues are not set
            }

            // Import Product Faster if setting is Enabled
            if( 'Yes' == get_option('ams_fast_product_importer')  ) {
                $productData = [];
                $productData['asin'] = $asin;
                $productData['title'] = $title;
                $productData['region'] = $region;
                $productData['parentSku'] = $parentASIN;
                $productData['product_url'] = $detail_page_url;
                $productData['import_method'] = '1'; // Set manually
                $default_message = '<span class="dashicons dashicons-saved"></span> ' . esc_html__( 'Success', 'ams-wc-amazon' );
                $productData['default_message'] = $default_message; // Set manually
                $productData['attributes'] = [];
                $productData['sale_price'] = $amount;
                $productData['regular_price'] = !empty($saving_amount) ? $saving_amount : $amount;
                
                // Success
                array_push($aResponse['success'], $asin);
                
                /**
                 * Import Product Faster
                 * 
                 * @param array $productData
                 * 
                 * @return string
                 */
                advancedProductImporter( $productData ); continue;
            }

            $payload_arr2 = array();
            $payload_arr2['ASIN']       = $asin;//'B00T0C9XRK';
            $payload_arr2['Resources']  = array( "ParentASIN", "ItemInfo.Title", "Offers.Listings.Price", "Offers.Listings.ProgramEligibility.IsPrimeExclusive", "Offers.Listings.ProgramEligibility.IsPrimePantry", "Offers.Listings.Promotions", "Offers.Listings.SavingBasis", "Offers.Listings.Availability.Message", "Offers.Summaries.HighestPrice", "Offers.Summaries.LowestPrice", "VariationSummary.Price.HighestPrice", "VariationSummary.Price.LowestPrice","VariationSummary.VariationDimension", "Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large" );
            $payload_arr2['PartnerTag'] = get_option( 'ams_associate_tag' );
            $payload_arr2['PartnerType']= 'Associates';
            $payload_arr2['Marketplace']= $marketplace; //'www.amazon.com';
            $payload_arr2['Operation']  = 'GetVariations';
            $payload2                   = json_encode($payload_arr2);
            $host                       = $regions[ $locale ]['Host'];
            $uri_path                   = "/paapi5/getvariations";
            $api2                       = new  \Amazon\Affiliate\Api\Amazon_Product_Api ( $access_key, $secret_key,$region, $service_name, $uri_path, $payload2, $host, 'getVariation' );
            $response2                  = $api2->do_request();
            
            $variations                 = isset($response2->VariationsResult->VariationSummary) ? $response2->VariationsResult->VariationSummary : null;
            $attributes                 = isset($response2->VariationsResult->Items) ? $response2->VariationsResult->Items : [];

            
            $VariationPage              = 2; 
            $Variationlist              = [];                   
            if( isset($variations->PageCount) && $variations->PageCount >= 1 ) {
                foreach( $response2->VariationsResult->Items as $item ) {
                    $VariationAttribute = [];
                    foreach( $item->VariationAttributes as $ItemVariationAttribute ) {
                        $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                    }
            
                    // Safe check for Amount
                    if (isset($item->Offers->Listings[0]->Price->Amount)) {
                        $amount = $item->Offers->Listings[0]->Price->Amount;
                    } else {
                        $amount = null; // or some default value
                    }
            
                    // Safe check for DisplayAmount
                    if (isset($item->Offers->Listings[0]->Price->DisplayAmount)) {
                        $DisplayAmount = $item->Offers->Listings[0]->Price->DisplayAmount;
                    } else {
                        $DisplayAmount = null; // or some default value
                    }
            
                    // Safe check for SavingBasis Amount
                    if (isset($item->Offers->Listings[0]->SavingBasis) && isset($item->Offers->Listings[0]->SavingBasis->Amount)) {
                        $saving_amount = $item->Offers->Listings[0]->SavingBasis->Amount;
                    } else {
                        $saving_amount = null; // or some default value
                    }
            
                    // Safe check for SavingDisplayAmount
                    if (isset($item->Offers->Listings[0]->SavingBasis) && isset($item->Offers->Listings[0]->SavingBasis->DisplayAmount)) {
                        $SavingDisplayAmount = $item->Offers->Listings[0]->SavingBasis->DisplayAmount;
                    } else {
                        $SavingDisplayAmount = null; // or some default value
                    }
            
                    // Stock status
                    $product_stock = isset($item->Offers->Listings[0]->Availability->Message) ? $item->Offers->Listings[0]->Availability->Message : '';
                    $stock_status  = !empty($product_stock) ? 'instock' : 'outofstock';

                    if (empty($saving_amount)) {
                        $sale_price = $amount;
                        $regular_price = $amount;
                    } else {
                        $sale_price = $amount;
                        $regular_price = $saving_amount;
                    }

                    // Add variation images
                    $v_gallery = [@$item->Images->Primary->Large->URL];
                    $Variationlist[] = array(
                        'post_title' => $item->ItemInfo->Title->DisplayValue,
                        'attributes' => $VariationAttribute,
                        'sku' => $item->ASIN,
                        'regular_price' => floatval($regular_price),
                        'sale_price' => floatval($sale_price),
                        'stock_status' => $stock_status,
                        'product_image_gallery' => $v_gallery,
                    );
                }
                
                while( $VariationPage <= $variations->PageCount ) {
                    $payload_arr2['VariationPage']   = $VariationPage;
                    $payload3                   = json_encode($payload_arr2);
                    $api3 = new  \Amazon\Affiliate\Api\Amazon_Product_Api ( 
                        $access_key, $secret_key,$region, $service_name, $uri_path, $payload3, $host, 'getVariation' 
                    );
                    $response3 = $api3->do_request();

                    foreach( isset($response3->VariationsResult->Items) ? $response3->VariationsResult->Items : [] as $item ) {
                        $VariationAttribute = [];
                        foreach( $item->VariationAttributes as $ItemVariationAttribute ) {
                            $VariationAttribute[$ItemVariationAttribute->Name] = trim($ItemVariationAttribute->Value);
                        }
                        
                        $amount = isset($item->Offers->Listings[0]->Price->Amount) 
                            ? $item->Offers->Listings[0]->Price->Amount 
                            : null;
                        
                        $DisplayAmount = isset($item->Offers->Listings[0]->Price->DisplayAmount) 
                            ? $item->Offers->Listings[0]->Price->DisplayAmount 
                            : null;

                        $saving_amount = isset($item->Offers->Listings[0]->SavingBasis) && isset($item->Offers->Listings[0]->SavingBasis->Amount) 
                            ? $item->Offers->Listings[0]->SavingBasis->Amount 
                            : null;
                        
                        $SavingDisplayAmount = isset($item->Offers->Listings[0]->SavingBasis) && isset($item->Offers->Listings[0]->SavingBasis->DisplayAmount) 
                            ? $item->Offers->Listings[0]->SavingBasis->DisplayAmount 
                            : null;

                        $product_stock = isset($item->Offers->Listings[0]->Availability->Message) 
                            ? $item->Offers->Listings[0]->Availability->Message 
                            : '';
                        
                        $stock_status = !empty($product_stock) ? 'instock' : 'outofstock';

                        if (empty($saving_amount)) {
                            $sale_price = $amount;
                            $regular_price = $amount;
                        } else {
                            $sale_price = $amount;
                            $regular_price = $saving_amount;
                        }

                        // Add variation images
                        $v_gallery = [@$item->Images->Primary->Large->URL];
                        $Variationlist[] = array(
                            'post_title' => $item->ItemInfo->Title->DisplayValue,
                            'attributes' => $VariationAttribute,
                            'sku' => $item->ASIN,
                            'regular_price' => floatval($regular_price),
                            'sale_price' => floatval($sale_price),
                            'stock_status' => $stock_status,
                            'product_image_gallery' => $v_gallery,
                        );
                    }
                    
                    $VariationPage++;
                }
            }
            
            $content = '';
            foreach ( $features as $feature) {
                $content .= '<ul><li>'.$feature.'</li></ul>';
            }

            if(empty($content)) {
                $options_new = array(
                    'http'=>array(
                        'method'=>"GET",
                        'header'=>"Accept-language: en\r\n" .
                                "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/96.0.4664.110 Safari/537.36\r\n" // i.e. An iPad 
                    )
                );

                $url = esc_url_raw($row->DetailPageURL);
                $context_new = stream_context_create( $options_new );
                $headers = @get_headers($url);
                $response_body_new = '';
                if ($headers && strpos($headers[0], '200') !== false) {
                    $response_body_new = @file_get_contents($url, false, $context_new);
                }

                if (!empty($response_body_new)) {
                    if (!class_exists('simple_html_dom')) {
                        require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
                    }
                    $html_new = new \simple_html_dom();
                    $html_new->load($response_body_new);

                    $content = $this->fetchContentFromHtml($html_new);
                }
            }

            $user_id = get_current_user();

            // Get status settings
            $importStatus = get_option( 'ams_product_import_status', true );

            $post_id = wp_insert_post(array(
                'post_author'  => $user_id,
                'post_title'   => stripslashes($title),
                // 'post_name'    => sanitize_title( $title ),
                'post_content' => $content,
                'post_status'  => $importStatus,
                // 'post_status'  => 'publish',
                'post_type'    => "product",
                'post_parent'  => '',
            ));

            if(!isset($variations->VariationDimensions) || empty($variations->VariationDimensions)){
                wp_set_object_terms( $post_id, 'simple', 'product_type');
                $product = wc_get_product( $post_id );
                $product->save(); // Update
                // For simple products, always use the product's own ASIN
                update_post_meta( $post_id, '_sku', $asin );
            } else {
                wp_set_object_terms( $post_id, 'variable', 'product_type');
                $product = wc_get_product( $post_id );
                $product->save(); // Update
                // For variable products, use ParentASIN if available, otherwise use the product's own ASIN
                update_post_meta( $post_id, '_sku', $parentASIN !== $asin ? $parentASIN : $asin );
            }

            $product_category = isset($row->ItemInfo->Classifications->Binding->DisplayValue) ? $row->ItemInfo->Classifications->Binding->DisplayValue : '';
            if( empty( trim($product_category) ) || "unknown binding" == trim( strtolower($product_category) ) ) {
                $product_category = isset($row->ItemInfo->Classifications->ProductGroup->DisplayValue) ? $row->ItemInfo->Classifications->ProductGroup->DisplayValue : '';
            }
            $ams_default_category = get_option('ams_default_category');
            if(!empty($ams_default_category) && $ams_default_category == '_auto_import_amazon') {

                if(!empty($product_category)) {
                    
                    $run_cat = 1;
                    if (isset($_POST['is_cron']) && $_POST['is_cron'] == 1) {
                        if (get_option('product_category_cron') != 1) {
                            $run_cat = 0;
                        }
                    }

                    if(!empty($run_cat)) {
                        // Create if not exists
                        if( ! term_exists( $product_category, 'product_cat', $parent = 0 ) ) {
                            wp_insert_term($product_category, 'product_cat', array(
                                'description' => $product_category,
                                // 'slug' => $slug,
                                'parent' => 0
                            ));
                        }
                        if ($term = get_term_by('name', esc_attr( $product_category ), 'product_cat')) {
                            wp_set_object_terms($post_id, $term->term_id, 'product_cat');
                        } else {
                            wp_set_object_terms($post_id, $product_category, 'product_cat');
                        }
                    }

                } else {
                    // re call amazon code and get category
                    $options_new = array(
                        'http'=>array(
                            'method'=>"GET",
                            'header'=>"Accept-language: en\r\n" .
                                    "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/96.0.4664.110 Safari/537.36\r\n" // i.e. An iPad 
                        )
                    );

                    $context_new = stream_context_create( $options_new );
                    $response_body_new = $page = file_get_contents( esc_url_raw( $row->DetailPageURL ), false, $context_new );
                    if (!empty($response_body_new)) {
                        if (!class_exists('simple_html_dom')) {
                            require_once AMS_PLUGIN_PATH . '/includes/Admin/lib/simplehtmldom/simple_html_dom.php';
                        }
                        $html_new = new \simple_html_dom();
                        $html_new->load($response_body_new);

                        $product_category_new = $this->syncAndGetCategory($html_new);
                        if(!empty($product_category_new)) {
                            wp_set_object_terms($post_id, $product_category_new, 'product_cat');
                        }
                    }

                } //end else
            } else {
                wp_set_object_terms($post_id, $ams_default_category, 'product_cat');
            }
            update_post_meta( $post_id, '_visibility', 'visible' );
            update_post_meta( $post_id, 'total_sales', '0' );
            update_post_meta( $post_id, '_downloadable', 'no' );
            update_post_meta( $post_id, '_virtual', 'yes' );

            if (!isset($variations->VariationDimensions) || empty($variations->VariationDimensions)) {
                if (empty($saving_amount)) {
                    $price = $this->parsePriceStringnew($amount);
                    update_post_meta($post_id, '_regular_price', $price);
                    update_post_meta($post_id, '_sale_price', $price);
                    update_post_meta($post_id, '_price', $price);
                } else {
                    $regular_price = $this->parsePriceStringnew($saving_amount);
                    $sale_price = $this->parsePriceStringnew($amount);
                    update_post_meta($post_id, '_regular_price', $regular_price);
                    update_post_meta($post_id, '_sale_price', $sale_price);
                    update_post_meta($post_id, '_price', $sale_price);
                }
            } else {
                wp_set_object_terms($post_id, 'variable', 'product_type');
            }
            
            update_post_meta( $post_id, '_purchase_note', '' );
            update_post_meta( $post_id, '_featured', 'no' );
            update_post_meta( $post_id, '_weight', '' );
            update_post_meta( $post_id, '_length', '' );
            update_post_meta( $post_id, '_width', '' );
            update_post_meta( $post_id, '_height', '' );
            
            update_post_meta( $post_id, '_sale_price_dates_from', '' );
            update_post_meta( $post_id, '_sale_price_dates_to', '' );
            update_post_meta( $post_id, '_sold_individually', '' );
            update_post_meta( $post_id, '_manage_stock', 'no' );
            update_post_meta( $post_id, '_backorders', 'no' );
            update_post_meta( $post_id, '_stock', '' );
            update_post_meta( $post_id, '_wca_amazon_affiliate_asin', $asin );
            update_post_meta( $post_id, '_wca_amazon_affiliate_parent_asin', $parentASIN );
            
            // Amazon product URL
            update_post_meta( $post_id, '_ams_product_url', $detail_page_url );
            update_post_meta( $post_id, '_detail_page_url', $detail_page_url );
            update_post_meta( $post_id, '_region', $locale );
            update_post_meta( $post_id, '_import_method', '1' );
            
            ############################### Create product Variations ############################################
            if(isset($variations->VariationDimensions) && !empty($variations->VariationDimensions)){
                $attributeChecks = [];
                $attributes_data = [];
                foreach ($variations->VariationDimensions as $attribute => $term_name ) {
                    $attr_label = $term_name->DisplayName;
                    $values = $term_name->Values;
                    $values_array = implode('|', $values);
                    // $attr_slug = sanitize_title($attr_label);
                    $attr_slug = sanitize_title($term_name->Name);

                    // TODO: Step 2 or 3
                    $values = array_map('trim', $values);
                    $attributeChecks[$attr_slug] = sanitize_title($attr_label);

                    // TODO: Step 2
                    $attributes_data[] = array(
                        'name'=>$attr_label, 
                        'slug' => $attr_slug, 
                        'options'=>$values, 
                        'visible' => 1, 
                        'variation' => 1 
                    );
                }
                
                // TODO: Step 2
                wc_update_product_attributes($post_id,$attributes_data);
                
                // TODO: Step 2
                $product = wc_get_product($post_id);
                foreach( $Variationlist as $SingleVariation ) {
                    if (!isset($SingleVariation['sku']) || empty($SingleVariation['attributes']) || "outofstock" == $SingleVariation['stock_status']) {
                        continue;
                    }

                    $variation = array(
                        'post_title'  => $SingleVariation['post_title'],
                        'post_name'   => 'product-'.$post_id.'-variation-'.$SingleVariation['sku'],
                        'post_status' => 'publish',
                        'post_parent' => $post_id,
                        'post_type'   => 'product_variation',
                        'guid'        => $product->get_permalink()
                    );
                    // $existing_variation = get_post($variation);
                    $existing_variation = get_product_by_sku($SingleVariation['sku']);
                    if($existing_variation !== null) {
                        // $variation_id = $existing_variation->ID;
                        // $variation['ID'] = $existing_variation->ID;
                        $variation['ID'] = $variation_id = $existing_variation->get_id();
                        wp_update_post($variation);                        
                    } else {
                        $variation_id = wp_insert_post( $variation );
                    }

                    // Get an instance of the WC_Product_Variation object
                    $variation = new \WC_Product_Variation( $variation_id );
                    if( count($SingleVariation['attributes']) > 0 ) {
                        // Iterating through the variations attributes
                        foreach ($SingleVariation['attributes'] as $attribute => $term_name ) {
                            $taxonomy = 'pa_'.$attribute; // The attribute taxonomy
                            
                            $term_name = esc_attr($term_name);

                            // If taxonomy doesn't exists
                            if( ! taxonomy_exists( $taxonomy ) ) continue;

                            // Check if the Term name exist
                            if( ! term_exists( $term_name, $taxonomy ) ) continue;
                            $term_slug = get_term_by('name', $term_name, $taxonomy )->slug; // Get the term slug

                            // Get the post Terms names from the parent variable product.
                            $post_term_names =  wp_get_post_terms( $post_id, $taxonomy, array('fields' => 'names') );
                            $post_term_names = array_map('strtolower', $post_term_names);

                            // Check if the post term exist
                            if( ! in_array( strtolower($term_name), $post_term_names ) ) continue;
                            // Set/save the attribute data in the product variation
                            update_post_meta( $variation_id, 'attribute_'.$taxonomy, $term_slug );
                        }
                    }
                    
                    // SKU
                    if( ! empty( $SingleVariation['sku'] ) ) {
                        $variation->set_sku( $SingleVariation['sku'] );
                    }

                    // Set defaults
                    if( isset($SingleVariation['regular_price']) && $SingleVariation['regular_price'] == '0' ) {
                        $SingleVariation['regular_price'] = '';
                    }
                    
                    if( isset($SingleVariation['sale_price']) && $SingleVariation['sale_price'] == '0' ) {
                        $SingleVariation['sale_price'] = '';
                    }

                    // Prices
                    if( empty( $SingleVariation['sale_price'] ) ){
                        $variation->set_price( $SingleVariation['regular_price'] ?? '' );
                    } else {
                        $variation->set_price( $SingleVariation['sale_price'] ?? '' );
                        $variation->set_sale_price( $SingleVariation['sale_price'] ?? '' );
                    }
                    
                    $variation->set_regular_price($SingleVariation['regular_price'] ?? '');

                    // Stock
                    if( ! empty($SingleVariation['stock_status']) ){
                        // $variation->set_stock_quantity( $data['stock_qty'] );
                        // $variation->set_manage_stock(true);
                        $variation->set_stock_status($SingleVariation['stock_status']);
                    } else {
                        $variation->set_manage_stock(false);
                    }

                    // Product image gallery
                    if( !empty( $SingleVariation['product_image_gallery'] ) ) {
                        if( count( $SingleVariation['product_image_gallery'] ) > 0 ) {
                            // Create image id
                            $attachment = array_shift($SingleVariation['product_image_gallery']);

                            if ( 'Yes' === get_option( 'ams_remote_amazon_images' ) ) {
                                $this->attach_product_thumbnail_url($variation_id, $attachment, 2);

                                // Remove previous image
                                $image_id = $variation->get_image_id();
                                if( $image_id ) {
                                    // wp_delete_post( $image_id );
                                    // wp_delete_attachment( $image_id, true );
                                }
                            } else {
                                // Reset variation image url 
                                $this->reset_product_thumbnail_url($variation_id, $flag=2);
                                // Attach variation image
                                $attachment_id = $this->attach_product_thumbnail($variation_id, $attachment, 2);

                                if( $attachment_id ) {
                                    // Remove previous image
                                    $image_id = $variation->get_image_id();
                                    if( $image_id ) {
                                        wp_delete_post( $image_id );
                                        wp_delete_attachment( $image_id, true );
                                    }
                                    // Set image id
                                    $variation->set_image_id($attachment_id);
                                }
                            }
                        }
                    }
                    
                    $variation->set_weight(''); // weight (reseting)
                    $variation->save(); // Save the data
                }
            }
            ######################################################################################################
            
            // Check remote amazon images.
            if( $image || $gallery ) {
                // Remove featured image and url.
                $this->delete_product_images($post_id);
                $this->reset_product_thumbnail_url($post_id, $flag=0);
                // Remove product gallery images and url.
                $this->delete_product_gallery_images($post_id);
                $this->reset_product_thumbnail_url($post_id, $flag=1);
                
                $gallery_url = [];
                $gallery = is_array($gallery) ? $gallery : [];
                if ( 'Yes' === get_option( 'ams_remote_amazon_images' ) ) {
                    // Set featured image url
                    $this->attach_product_thumbnail_url( $post_id, $image, 0 );
                    // Set featured image gallary
                    $gallery_url = [];
                    foreach( $gallery as $image ) {
                        // Set gallery image.
                        if(isset($image->{$thumbnail_size})) {
                            $gallery_url[] = $image->{$thumbnail_size}->URL;
                        } else {
                            $imageArray = (array) $image; // Convert object to array
                            $key = key($imageArray);
                            $gallery_url[] = $image->{$key}->URL;
                        }
                    }
                    $this->attach_product_thumbnail_url( $post_id, $gallery_url, 1 );
                } else {
                    // Set featured image url
                    $this->attach_product_thumbnail($post_id, $image, 0);
                    // Set featured image gallary
                    if( count($gallery) > 0 ) {
                        foreach( $gallery as $image ) {
                            // Set gallery image.
                            if(isset($image->{$thumbnail_size})) {
                                $this->attach_product_thumbnail( $post_id, $image->{$thumbnail_size}->URL, 1 );
                            } else {
                                $imageArray = (array) $image; // Convert object to array
                                $key = key($imageArray);
                                $this->attach_product_thumbnail( $post_id, $image->{$key}->URL, 1 );
                            }
                        }
                    }
                }
            }
            
            $Current = get_post_meta( $post_id, '_stock_status',true);
            update_post_meta( $post_id, 'ams_last_cron_update',date('Y-m-d H:i:s'));
            update_post_meta( $post_id, 'ams_last_cron_status',0);
            update_post_meta( $post_id, '_stock_status', $product_status );
            
            // Success
            array_push($aResponse['success'], $asin);
        } // End foreach
        
        /* END HERE */

        // Failed or not found sku
        $aResponse['failed'] = array_diff($asins, array_merge($aResponse['success'], $aResponse['imported']));

        // Return json with success

        wp_send_json_success( $aResponse, 200 );
    }

    public function ams_replace_product() {
        error_log('AMS Replace Product: AJAX handler called');
        

        if (!isset($_POST['nonce'])) {
            error_log('AMS Replace Product: No nonce provided');
            wp_send_json_error(array(
                'message' => __('No security token provided.', 'ams-wc-amazon')
            ));
        }
        
        $nonce = sanitize_text_field($_POST['nonce']);
        
        if (!wp_verify_nonce($nonce, 'ams_replace_product_nonce')) {
            error_log('AMS Replace Product: Nonce verification failed');
            wp_send_json_error(array(
                'message' => __('Security check failed. Please refresh the page and try again.', 'ams-wc-amazon')
            ));
        }

        if (!current_user_can('edit_products')) {
            wp_send_json_error(array(
                'message' => __('You do not have permission to edit products.', 'ams-wc-amazon')
            ));
        }

        $product_id = intval($_POST['product_id']);
        $new_asin = sanitize_text_field($_POST['new_asin']);
        $target_country = isset($_POST['target_country']) ? sanitize_text_field($_POST['target_country']) : null;

        if (!$product_id || !$new_asin) {
            wp_send_json_error(array(
                'message' => __('Invalid product ID or ASIN provided.', 'ams-wc-amazon')
            ));
        }

        if (!preg_match('/^B[0-9A-Z]{9}$/', $new_asin)) {
            wp_send_json_error(array(
                'message' => __('Invalid ASIN format. Must be B followed by 9 alphanumeric characters.', 'ams-wc-amazon')
            ));
        }
        
        // Determine which country/locale to use for API call
        // If target_country is provided, use it; otherwise use 'nl' (Netherlands) 
        // since API credentials are configured for Netherlands marketplace
        $locale = $target_country ? $target_country : 'nl';


        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error(array(
                'message' => __('Product not found.', 'ams-wc-amazon')
            ));
        }

        $current_asin = get_post_meta($product_id, '_wca_amazon_affiliate_asin', true);
        if (!$current_asin) {
            wp_send_json_error(array(
                'message' => __('This product was not imported via AMS Amazon plugin.', 'ams-wc-amazon')
            ));
        }

        if ($current_asin === $new_asin) {
            wp_send_json_error(array(
                'message' => __('New ASIN is the same as current ASIN.', 'ams-wc-amazon')
            ));
        }

        // Get original product's import region
        $original_region = get_post_meta($product_id, '_region', true);
        $import_method = get_post_meta($product_id, '_import_method', true);
        

        
        // If product was imported via API and has a region, validate country consistency
        if ($import_method === '1' && $original_region && $target_country) {
            // Normalize regions for comparison
            $original_country = $original_region;
            $new_country = $target_country;
            
            // Handle special domain mappings
            $domain_mappings = array(
                'co.uk' => 'uk',
                'com.au' => 'au', 
                'com.br' => 'br',
                'com.tr' => 'tr'
            );
            
            // Normalize original region
            if (isset($domain_mappings[$original_country])) {
                $original_country = $domain_mappings[$original_country];
            }
            
            // Normalize target region  
            if (isset($domain_mappings[$new_country])) {
                $new_country = $domain_mappings[$new_country];
            }
            
            // Check if countries are different
            if ($original_country !== $new_country) {
                $regions = ams_get_amazon_regions();
                $original_country_name = isset($regions[$original_region]) ? $regions[$original_region]['RegionName'] : $original_region;
                $new_country_name = isset($regions[$target_country]) ? $regions[$target_country]['RegionName'] : $target_country;
                
                wp_send_json_error(array(
                    'message' => sprintf(
                        __('⚠️ Country Mismatch: This product was imported from Amazon %s, but you\'re trying to replace it with a product from Amazon %s. For API-imported products, the replacement must be from the same Amazon marketplace to ensure compatibility with your API credentials. Please use a product from Amazon %s instead.', 'ams-wc-amazon'),
                        $original_country_name,
                        $new_country_name, 
                        $original_country_name
                    )
                ));
            }
        }

        try {
            $regions = ams_get_amazon_regions();
            $marketplace = ams_get_marketplace_value($locale);
            $service_name = 'ProductAdvertisingAPI';
            $region = $regions[$locale]['RegionCode'] ?? '';

            $access_key = get_option('ams_access_key_id');
            $secret_key = get_option('ams_secret_access_key');

            if (empty($access_key) || empty($secret_key)) {
                wp_send_json_error(array(
                    'message' => __('Amazon API credentials not configured. Please check AMS settings.', 'ams-wc-amazon')
                ));
            }

            $payload_arr = array();
            $payload_arr['ItemIds'] = array($new_asin);
            $payload_arr['Resources'] = array(
                "ParentASIN", "Images.Primary.Small", "Images.Primary.Medium", "Images.Primary.Large", 
                "Images.Variants.Small", "Images.Variants.Medium", "Images.Variants.Large", 
                "ItemInfo.ByLineInfo", "ItemInfo.ContentInfo", "ItemInfo.ContentRating", 
                "ItemInfo.Classifications", "ItemInfo.ExternalIds", "ItemInfo.Features", 
                "ItemInfo.ManufactureInfo", "ItemInfo.ProductInfo", "ItemInfo.TechnicalInfo", 
                "ItemInfo.Title", "ItemInfo.TradeInInfo", "Offers.Listings.Availability.MaxOrderQuantity", 
                "Offers.Listings.Availability.Message", "Offers.Listings.Availability.MinOrderQuantity", 
                "Offers.Listings.Availability.Type", "Offers.Listings.Condition", 
                "Offers.Listings.Condition.ConditionNote", "Offers.Listings.Condition.SubCondition", 
                "Offers.Listings.DeliveryInfo.IsAmazonFulfilled", "Offers.Listings.DeliveryInfo.IsFreeShippingEligible", 
                "Offers.Listings.DeliveryInfo.IsPrimeEligible", "Offers.Listings.DeliveryInfo.ShippingCharges", 
                "Offers.Listings.IsBuyBoxWinner", "Offers.Listings.LoyaltyPoints.Points", 
                "Offers.Listings.MerchantInfo", "Offers.Listings.Price", 
                "Offers.Listings.ProgramEligibility.IsPrimeExclusive", "Offers.Listings.ProgramEligibility.IsPrimePantry", 
                "Offers.Listings.Promotions", "Offers.Listings.SavingBasis", "Offers.Summaries.HighestPrice", 
                "Offers.Summaries.LowestPrice", "Offers.Summaries.OfferCount"
            );
            $payload_arr['PartnerTag'] = get_option('ams_associate_tag');
            $payload_arr['PartnerType'] = 'Associates';
            $payload_arr['Marketplace'] = $marketplace;
            $payload_arr['Operation'] = 'GetItems';

            $payload = json_encode($payload_arr);
            $host = $regions[$locale]['Host'];
            $uri_path = "/paapi5/getitems";

            $api = new \Amazon\Affiliate\Api\Amazon_Product_Api($access_key, $secret_key, $region, $service_name, $uri_path, $payload, $host, 'GetItems');
            $response = $api->do_request();
            $results = @$response->ItemsResult->Items;

            if (!$results || empty($results)) {
                wp_send_json_error(array(
                    'message' => __('Product not found on Amazon or not available in your region.', 'ams-wc-amazon')
                ));
            }

            $item = $results[0];
            $title = isset($item->ItemInfo->Title->DisplayValue) ? $item->ItemInfo->Title->DisplayValue : '';
            $detail_page_url = isset($item->DetailPageURL) ? $item->DetailPageURL : '';
            $parentASIN = isset($item->ParentASIN) ? $item->ParentASIN : $new_asin;

            if (empty($title) || empty($detail_page_url)) {
                wp_send_json_error(array(
                    'message' => __('Incomplete product data received from Amazon API.', 'ams-wc-amazon')
                ));
            }

            // Enhanced duplicate checking - check both main products and variations
            global $wpdb;
            
            // Check if new ASIN exists as SKU in any product or variation (excluding current product and its variations)
            $existing_sku_check = $wpdb->get_results($wpdb->prepare("
                SELECT pm.post_id, p.post_type, p.post_parent 
                FROM {$wpdb->postmeta} pm 
                JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
                WHERE pm.meta_key = '_sku' 
                AND pm.meta_value = %s 
                AND pm.post_id != %d 
                AND (p.post_parent != %d OR p.post_parent = 0)
            ", $new_asin, $product_id, $product_id));
            
            if (!empty($existing_sku_check)) {
                $conflict_info = $existing_sku_check[0];
                $conflict_type = ($conflict_info->post_type === 'product_variation') ? 'variation' : 'main product';
                wp_send_json_error(array(
                    'message' => sprintf(__('ASIN %s is already used by another %s (ID: %d). Please choose a different ASIN.', 'ams-wc-amazon'), $new_asin, $conflict_type, $conflict_info->post_id)
                ));
            }
            
            // Check if new ASIN exists as Amazon ASIN in any other product (excluding current product)
            $existing_asin_check = $wpdb->get_results($wpdb->prepare("
                SELECT post_id 
                FROM {$wpdb->postmeta} 
                WHERE meta_key = '_wca_amazon_affiliate_asin' 
                AND meta_value = %s 
                AND post_id != %d
            ", $new_asin, $product_id));
            
            if (!empty($existing_asin_check)) {
                wp_send_json_error(array(
                    'message' => sprintf(__('ASIN %s is already imported as another product (ID: %d). Please choose a different ASIN.', 'ams-wc-amazon'), $new_asin, $existing_asin_check[0]->post_id)
                ));
            }
            


            // Clean up existing variations to prevent SKU conflicts
            $existing_variations = $wpdb->get_col($wpdb->prepare("
                SELECT ID FROM {$wpdb->posts} 
                WHERE post_parent = %d 
                AND post_type = 'product_variation'
            ", $product_id));
            
            if (!empty($existing_variations)) {
                error_log('AMS Replace: Cleaning up ' . count($existing_variations) . ' existing variations for product ' . $product_id);
                foreach ($existing_variations as $variation_id) {
                    wp_delete_post($variation_id, true); // Force delete
                }
            }
            
            // Update WooCommerce SKU using proper WooCommerce method
            $wc_product = wc_get_product($product_id);
            if ($wc_product) {
                try {
                    // Convert to simple product if it was variable
                    if ($wc_product->is_type('variable')) {
                        wp_set_object_terms($product_id, 'simple', 'product_type');
                        $wc_product = wc_get_product($product_id); // Reload as simple product
                    }
                    
                    $wc_product->set_sku($new_asin);
                    $wc_product->save();
                } catch (Exception $e) {
                    wp_send_json_error(array(
                        'message' => sprintf(__('Failed to update SKU: %s', 'ams-wc-amazon'), $e->getMessage())
                    ));
                }
            }
            
            // Update AMS meta fields
            update_post_meta($product_id, '_wca_amazon_affiliate_asin', $new_asin);
            update_post_meta($product_id, '_wca_amazon_affiliate_parent_asin', $parentASIN);
            update_post_meta($product_id, '_ams_product_url', $detail_page_url);
            update_post_meta($product_id, '_detail_page_url', $detail_page_url);
            update_post_meta($product_id, '_region', $locale);

            $product_update = array(
                'ID' => $product_id,
                'post_title' => sanitize_text_field($title),
                'post_name' => sanitize_title($title)
            );
            wp_update_post($product_update);

            if (isset($item->ItemInfo->Features->DisplayValues)) {
                $features = $item->ItemInfo->Features->DisplayValues;
                $description = '<ul>';
                foreach ($features as $feature) {
                    $description .= '<li>' . esc_html($feature) . '</li>';
                }
                $description .= '</ul>';
                
                wp_update_post(array(
                    'ID' => $product_id,
                    'post_content' => $description
                ));
            }

            if (isset($item->Offers->Listings[0]->Price->DisplayAmount)) {
                $price_string = $item->Offers->Listings[0]->Price->DisplayAmount;
                $price = floatval(preg_replace('/[^\d.]/', '', $price_string));
                if ($price > 0) {
                    update_post_meta($product_id, '_regular_price', $price);
                    update_post_meta($product_id, '_price', $price);
                }
            }

            $thumbnail_size = get_option('ams_product_thumbnail_size', 'Large');
            if (isset($item->Images->Primary->{$thumbnail_size}->URL)) {
                $image_url = $item->Images->Primary->{$thumbnail_size}->URL;
                attach_product_thumbnail_url_api($product_id, $image_url, 0);
            } elseif (isset($item->Images->Primary->Large->URL)) {
                $image_url = $item->Images->Primary->Large->URL;
                attach_product_thumbnail_url_api($product_id, $image_url, 0);
            }

            update_post_meta($product_id, 'ams_last_replacement_date', current_time('mysql'));
            update_post_meta($product_id, 'ams_last_replacement_asin', $current_asin);
            update_post_meta($product_id, 'ams_replaced_by_user', get_current_user_id());

            wp_send_json_success(array(
                'message' => sprintf(
                    __('Product successfully replaced! ASIN updated from %s to %s. Page will reload in 2 seconds.', 'ams-wc-amazon'),
                    $current_asin,
                    $new_asin
                )
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'ams-wc-amazon'), $e->getMessage())
            ));
        }
    }
}