<?php

namespace Amazon\Affiliate;

/**
 * Plugin Assets handler class
 *
 * @package Amazon\Affiliate
 */
class Assets {

    /**
     * Assets constructor.
     */
    function __construct() {
        add_action('admin_enqueue_scripts', array($this, 'backend_register_assets'));
        add_action('wp_enqueue_scripts', array($this, 'frontend_register_assets'));
        add_action('admin_enqueue_scripts', array($this, 'load_form_control'));
        // Load Bootstrap 5 + AMS custom CSS
        add_action('admin_enqueue_scripts', array($this, 'load_bootstrap_and_ams_styles'), 10);
        // Load page-specific AMS CSS
        add_action('admin_enqueue_scripts', array($this, 'load_ams_page_styles'), 1000);
    }

    public function load_form_control() {
        wp_enqueue_script(
            'ams-form-control', 
            AMS_PLUGIN_URL . 'assets/plugin/js/components/custom/form-control.js', 
            array('jquery'), 
            AMS_PLUGIN_VERSION, 
            true
        );
        
        wp_localize_script('ams-form-control', 'amsFormControl', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce_ams_de_activated' => wp_create_nonce('ams_de_activated')
        ));
    }

    /**
     * Load Bootstrap 5 CSS/JS + AMS custom styles
     * Bootstrap provides the foundation, AMS CSS adds custom styling on top
     */
    public function load_bootstrap_and_ams_styles() {
        // Only load on plugin pages
        $screen = get_current_screen();
        
        if (!$screen || (strpos($screen->id, 'wc-amazon-affiliate') === false && 
            strpos($screen->id, 'wc-product') === false && 
            strpos($screen->id, 'automation') === false &&
            strpos($screen->id, 'brand-settings') === false &&
            strpos($screen->id, 'products-search-without-api') === false &&
            strpos($screen->id, 'amazon-bestsellers') === false &&
            strpos($screen->id, 'product-import-by-url') === false &&
            strpos($screen->id, 'product-review-import') === false &&
            strpos($screen->id, 'ams-amazon_page_view-logs') === false)) {
            return;
        }

        // Enqueue Bootstrap 5 CSS - LOCAL FILE (no external dependency)
        wp_enqueue_style(
            'ams-bootstrap-5',
            AMS_PLUGIN_URL . 'assets/vendor/bootstrap/css/bootstrap.min.css',
            array(),
            '5.3.2',
            'all'
        );
        
        // Enqueue Bootstrap Icons - LOCAL FILE (no external dependency)
        wp_enqueue_style(
            'ams-bootstrap-icons',
            AMS_PLUGIN_URL . 'assets/vendor/bootstrap-icons/bootstrap-icons.css',
            array(),
            '1.11.1',
            'all'
        );
        
        // Enqueue Bootstrap 5 JS - LOCAL FILE (no external dependency)
        wp_enqueue_script(
            'ams-bootstrap-5-js',
            AMS_PLUGIN_URL . 'assets/vendor/bootstrap/js/bootstrap.bundle.min.js',
            array('jquery'),
            '5.3.2',
            true
        );
    }

    /**
     * Load AMS page-specific CSS files with unique prefixes
     * Priority 1000 ensures it loads after Bootstrap to override when needed
     */
    public function load_ams_page_styles() {
        // Only load on plugin pages
        $screen = get_current_screen();
        if (!$screen) {
            return;
        }

        // Always load common AMS styles on plugin pages
        if (strpos($screen->id, 'wc-amazon-affiliate') !== false || 
            strpos($screen->id, 'wc-product') !== false || 
            strpos($screen->id, 'automation') !== false ||
            strpos($screen->id, 'brand-settings') !== false ||
            strpos($screen->id, 'products-search-without-api') !== false ||
            strpos($screen->id, 'amazon-bestsellers') !== false ||
            strpos($screen->id, 'product-import-by-url') !== false ||
            strpos($screen->id, 'product-review-import') !== false ||
            strpos($screen->id, 'ams-amazon_page_view-logs') !== false) {
            
            wp_enqueue_style(
                'ams-common-styles',
                AMS_PLUGIN_URL . 'assets/plugin/css/admin-fixes.css',
                array('ams-bootstrap-5'), // Depends on Bootstrap 5
                AMS_PLUGIN_VERSION,
                'all'
            );
        }

        // Load page-specific CSS based on current page
        $page_css_map = array(
            'toplevel_page_wc-amazon-affiliate' => 'ams-dashboard.css',
            'admin_page_automation' => 'ams-automation.css',
        );

        if (isset($page_css_map[$screen->id])) {
            $css_file = $page_css_map[$screen->id];
            $handle = 'ams-page-' . str_replace('.css', '', $css_file);
        }
    }

    public function backend_register_assets($screen) {
        // Dequeue any conflicting styles or scripts
        wp_dequeue_style('inext-wpc-admin-style');
        wp_dequeue_script('inext-wpc-bootstrap-script');

        // Load backend JS on ALL admin pages (original behavior for cron column functionality)
        if (is_admin()) {
            // Load jQuery
            wp_enqueue_script('jquery');

            // Load custom backend JS specifically for your plugin
            wp_enqueue_script('ams-amazon-backend', AMS_PLUGIN_URL . 'assets/plugin/js/backend.js', array('jquery'), AMS_PLUGIN_VERSION, true);

            // Determine if we are on the main dashboard page
            $ams_dashboard = (strtolower($screen) === 'toplevel_page_wc-amazon-affiliate');

            // Localize script with dynamic data
            wp_localize_script('ams-amazon-backend', 'amsbackend', array(
                'ajax_url'                      => admin_url('admin-ajax.php'),
                'check_nonce'                   => wp_create_nonce('ams_product_import'),
                'ams_test_api'                  => wp_create_nonce('ams_test_api'),
                'nonce_ams_dashboard_info'      => wp_create_nonce('ams_dashboard_info'),
                'nonce_ams_de_activated'        => wp_create_nonce('ams_de_activated'),
                'nonce_ams_without_api'         => wp_create_nonce('ams_without_api'),
                'nonce_product_update_request'  => wp_create_nonce('product_update_request'),
                'ams_product_per_page'          => get_option('item_page'),
                'ams_dashboard'                 => $ams_dashboard,
                'ams_assets'                    => AMS_PLUGIN_URL . 'assets/',
                'ams_t_import'                  => esc_html('Importing', 'ams-wc-amazon'),
                'ams_t_loading'                 => esc_html('Loading...', 'ams-wc-amazon'),
                'ams_t_testing_api'             => esc_html('Testing API... Please wait while we process your request.', 'ams-wc-amazon'),
                'ams_t_deactivated'             => esc_html('Deactivated', 'ams-wc-amazon'),
                'ams_mass_product_importing'    => esc_html__('Product import has started.', 'ams-wc-amazon'),
                'nonce_ams_import_product_url'  => wp_create_nonce('ams_import_product_url'),
                'ams_product_availability'      => wp_create_nonce('ams_product_availability'),
            ));
        }
    }


    public function frontend_register_assets() {
        wp_enqueue_script('jquery');
        
        // Enqueue existing frontend CSS
        wp_register_style('ams-amazon-frontend', AMS_PLUGIN_URL . 'assets/plugin/css/frontend.css', false, AMS_PLUGIN_VERSION);
        wp_enqueue_style('ams-amazon-frontend');

        // Enqueue the brand filter CSS
        wp_register_style('ams-amazon-brand-filter', AMS_PLUGIN_URL . 'assets/plugin/css/brand-filter.css', false, AMS_PLUGIN_VERSION);
        wp_enqueue_style('ams-amazon-brand-filter');

        // Enqueue media uploader script
        wp_enqueue_media();
        
        // Enqueue the custom JS for the media uploader
        wp_register_script('brand-logo-upload', AMS_PLUGIN_URL . 'assets/plugin/js/brand-logo-upload.js', array('jquery'), AMS_PLUGIN_VERSION, true);
        wp_enqueue_script('brand-logo-upload');
    }
}