<?php
namespace Amazon\Affiliate\Frontend;
/**
 * Class WooCommerceCart
 *
 * @package Amazon\Affiliate\Frontend
 */
class WooCommerceCart {

    public function __construct() {
        $this->setup_hooks();
    }

    private function setup_hooks() {
        add_action('woocommerce_product_meta_start', [$this, 'buy_now_button_actions']);
        add_action('wp_footer', [$this, 'buy_now_button_actions'], 99);
        
        // Use intelligent auto-detection if custom button is enabled
        $use_custom_button = get_option('ams_use_custom_button', '0');
        if ($use_custom_button === '1') {
            // Universal filter that handles ALL product types (simple, variable, etc.)
            add_filter('woocommerce_loop_add_to_cart_link', [$this, 'buy_now_button_actions_category'], 999, 2);
            
            // Only setup intelligent hooks if auto-detection is enabled
            $auto_detection_enabled = get_option('ams_enable_auto_detection', '1') === '1';
            if ($auto_detection_enabled) {
                $this->setup_intelligent_hooks();
            }
        }
        
        // Also apply to WooCommerce blocks
        add_filter('woocommerce_loop_add_to_cart_args', [$this, 'modify_block_add_to_cart_args'], 999, 2);
        
        add_action('woocommerce_before_single_product', [$this, 'visitor_record']);
        add_action('woocommerce_checkout_init', [$this, 'woocommerce_external_checkout'], 10);
        add_filter('woocommerce_product_additional_information', [$this, 'ams_product_additional_information'], 10, 2);
        add_action('admin_post_cart_redirected_count', 'my_handle_form_submit');
        add_action('admin_post_nopriv_cart_redirected_count', 'my_handle_form_submit');
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        

    }

    public function enqueue_scripts() {
        wp_enqueue_script('cart_custom_js', AMS_PLUGIN_URL . 'assets/plugin/js/components/custom/cart.js', [], null, true);
    }

    // Handle WooCommerce blocks add to cart args
    public function modify_block_add_to_cart_args($args, $product) {
        $buy_action = get_option('ams_buy_action_btn');
        $ams_product_url = get_post_meta($product->get_id(), '_ams_product_url', true);
        
        // Only modify if this is an Amazon product and we're using redirect options
        if (!empty($ams_product_url) && ($buy_action === 'redirect' || $buy_action === 'cart_page')) {
            $btn_text = get_option('ams_buy_now_label', 'Buy Now');
            $ams_associate_tag = get_option('ams_associate_tag');
            $ams_amazon_country = get_option('ams_amazon_country', 'com');
            $enable_no_follow = get_option('ams_enable_no_follow_link') ? 'nofollow' : '';
            
            // Adjust Amazon country for 'mx'
            if ("mx" === strtolower($ams_amazon_country)) {
                $ams_amazon_country = 'com.mx';
            }
            
            if ($buy_action === 'redirect') {
                // Option 1: Direct Amazon product page
                $query = parse_url($ams_product_url, PHP_URL_QUERY);
                $product_associate_tag = (!empty($query) ? '&' : '?') . 'tag=' . $ams_associate_tag;
                $amazon_url = $ams_product_url . $product_associate_tag;
                
                // Generate tracking URL
                $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                    'action' => 'cart_redirected_count',
                    'id' => $product->get_id(),
                    'url' => $amazon_url
                ]);
                
                // Modify the args to point to Amazon
                $args['attributes']['href'] = esc_url($tracking_url);
                $args['attributes']['target'] = '_blank';
                $args['attributes']['rel'] = $enable_no_follow;
                
            } elseif ($buy_action === 'cart_page') {
                // Option 2: Amazon cart page
                $amazon_cart_url = $this->get_amazon_cart_url($product);
                
                // Generate tracking URL
                $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                    'action' => 'cart_redirected_count',
                    'id' => $product->get_id(),
                    'url' => $amazon_cart_url
                ]);
                
                // Modify the args to point to Amazon cart
                $args['attributes']['href'] = esc_url($tracking_url);
                $args['attributes']['target'] = '_blank';
                $args['attributes']['rel'] = $enable_no_follow;
            }
        }
        
        return $args;
    }

    // This function is for product information of front side
    function ams_product_additional_information( $product ) {
        if( $product->is_type('variable') ) {
            echo '<div id="ams-additional-information"></div>';
        } else {
            $ams_additional_information = get_post_meta( $product->get_id(), '_ams_additional_information', true );
            echo '<div id="ams-additional-information">'. $ams_additional_information .'</div>';
        }
    }

    // This function is for option 1+2+3+4 - single page
    public function buy_now_button_actions() {
        global $product;

        // Ensure $product is valid
        if (!$product || !is_object($product) || !method_exists($product, 'get_id')) {
            return;
        }

        $product_id = $product->get_id();
        if (!$product_id || !get_post_meta($product_id, '_ams_product_url', true)) {
            return;
        }

        $btn_text = get_option('ams_buy_now_label', 'Buy Now');
        $ams_associate_tag = get_option('ams_associate_tag');
        $asin_id = get_post_meta($product_id, '_wca_amazon_affiliate_asin', true);
        $ams_amazon_country = get_option('ams_amazon_country', 'com');
        $enable_no_follow = get_option('ams_enable_no_follow_link');
        $buy_action = get_option('ams_buy_action_btn');
        $ams_product_url = get_post_meta($product_id, '_ams_product_url', true);

        if (strtolower($ams_amazon_country) === 'mx') {
            $ams_amazon_country = 'com.mx';
        }

        add_filter('woocommerce_product_single_add_to_cart_text', function($text) use ($btn_text, $buy_action) {
            return ($buy_action === 'multi_cart' || $buy_action === 'dropship') ? __('Add to Cart', 'woocommerce') : (!empty($btn_text) ? $btn_text : __('Buy Now', 'woocommerce'));
        });

        // Add dropship fee indicator for Option 4
        if ($buy_action === 'dropship') {
            $percentage_profit = (float) get_option('ams_percentage_profit', 0);
            if ($percentage_profit > 0) {
                add_action('woocommerce_single_product_summary', function() use ($percentage_profit) {
                    echo '<div class="ams-dropship-notice" style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 12px; margin: 15px 0; font-size: 14px; color: #495057;">
                            <strong>ℹ️ ' . esc_html__('Dropship Notice:', 'ams-wc-amazon') . '</strong> 
                            ' . sprintf(esc_html__('This product includes a %s%% service fee for dropship handling.', 'ams-wc-amazon'), $percentage_profit) . '
                          </div>';
                }, 25);
            }
        }

        add_action('wp_footer', function() use ($btn_text, $buy_action, $ams_product_url, $ams_associate_tag, $product_id, $product) {
            if (!empty($btn_text)) {
                echo '<script type="text/javascript">
                        jQuery(document).ready(function($) {
                            // Try multiple selectors to ensure compatibility with different themes
                            var btn = $("button.single_add_to_cart_button, .single_add_to_cart_button, button[name=\"add-to-cart\"], .cart button[type=\"submit\"]").first();
                            if (btn.length) {
                                var buyAction = "' . esc_js($buy_action) . '";
                                
                                // Set button text based on action (consistent with filter above)
                                if (buyAction === "multi_cart" || buyAction === "dropship") {
                                    btn.text("' . esc_js(__('Add to Cart', 'woocommerce')) . '");
                                } else {
                                    var buttonText = "' . esc_js($btn_text) . '" || "' . esc_js(__('Buy Now', 'woocommerce')) . '";
                                    btn.text(buttonText);
                                }

                                // Handle different button actions
                                if (buyAction === "redirect") {
                                    var amazon_url = "' . esc_js($this->build_amazon_url_with_tag($ams_product_url, $ams_associate_tag)) . '";
                                    var tracking_url = "' . esc_js(admin_url('admin-post.php')) . '?" + $.param({
                                        action: "cart_redirected_count",
                                        id: "' . esc_js($product_id) . '",
                                        url: amazon_url
                                    });
                                    btn.off("click").on("click", function(e) {
                                        e.preventDefault();
                                        window.open(tracking_url, "_blank");
                                    });
                                } else if (buyAction === "cart_page") {
                                    var amazon_cart_url = "' . esc_js($this->get_amazon_cart_url($product)) . '";
                                    var tracking_url = "' . esc_js(admin_url('admin-post.php')) . '?" + $.param({
                                        action: "cart_redirected_count",
                                        id: "' . esc_js($product_id) . '",
                                        url: amazon_cart_url
                                    });
                                    btn.off("click").on("click", function(e) {
                                        e.preventDefault();
                                        window.open(tracking_url, "_blank");
                                    });
                                }
                                // For multi_cart and dropship, keep default WooCommerce behavior
                            }
                        });
                      </script>';
            }
        }, 999);
    }

    // Helper function to generate Amazon cart URL for Option 2
    private function get_amazon_cart_url($product) {
        $ams_associate_tag = get_option('ams_associate_tag');
        $asin_id = get_post_meta($product->get_id(), '_wca_amazon_affiliate_asin', true);
        $ams_amazon_country = get_option('ams_amazon_country', 'com');

        if ("mx" == strtolower($ams_amazon_country)) {
            $ams_amazon_country = 'com.mx';
        }

        $amazon_cart_url = 'https://www.amazon.' . $ams_amazon_country . '/gp/aws/cart/add.html';
        $args = [
            'AssociateTag' => $ams_associate_tag,
            'ASIN.1'       => $asin_id,
            'Quantity.1'   => 1
        ];

        return $amazon_cart_url . '?' . http_build_query($args);
    }

    // This is the redirect process code - option 3
    public function woocommerce_external_checkout() {
        // Check if we're on the checkout page
        if (!is_page('checkout') && !is_checkout()) {
            return;
        }
        // Check if dropship option is enabled
        if (strtolower(get_option('ams_buy_action_btn')) === strtolower('dropship')) {
            return;
        }
        $ams_access_key_id = get_option('ams_access_key_id');
        $ams_associate_tag = get_option('ams_associate_tag');
        $ams_amazon_country = get_option('ams_amazon_country');
        $url = 'https://www.amazon.' . $ams_amazon_country . '/gp/aws/cart/add.html?';
        
        $arg = array(
            'AWSAccessKeyId' => $ams_access_key_id,
            'AssociateTag'   => $ams_associate_tag,
        );
        $count = 1;
        $stay_to_checkout = false;
        foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
            $product_id = $cart_item['product_id'];
            $quantity = $cart_item['quantity'];
            $product = wc_get_product($product_id);
            if ($cart_item['variation_id'] > 0 && in_array($cart_item['variation_id'], $product->get_children())) {
                $asin_id = get_post_meta($cart_item['variation_id'], '_sku', true);
            } else {
                $asin_id = $product->get_sku();
            }
            $arg['ASIN.' . $count] = $asin_id;
            $arg['Quantity.' . $count] = $quantity;
            $this->total_count_products_add_to_cart($product_id);
            if (!get_post_meta($product_id, '_ams_product_url', true)) {
                $stay_to_checkout = true;
            } else {
                WC()->cart->remove_cart_item($cart_item_key);
            }
            $count++;
        }
        if ($stay_to_checkout) {
            return;
        }
        $arg = http_build_query($arg);
        $add_to_cart = $url . $arg;
        
        // Check if redirection delay is enabled for Option 3
        if (get_option('ams_checkout_redirected_seconds', true)) {
            // Use our custom redirect page with countdown
            $redirect_url = add_query_arg(
                array(
                    'ams_redirect' => '1',
                    'ams_redirect_uri' => urlencode($add_to_cart)
                ),
                home_url()
            );
            wp_safe_redirect($redirect_url);
        } else {
            // Direct redirect to Amazon
            wp_redirect(esc_url_raw($add_to_cart));
        }
        exit();
    }

    public function buy_now_button_actions_category($button, $product) {
        // Retrieve settings and metadata
        $product_id         = $product->get_id();
        $product_type       = $product->get_type();
        $btn_text           = get_option('ams_buy_now_label', 'Buy Now');
        $ams_associate_tag  = get_option('ams_associate_tag');
        $ams_amazon_country = get_option('ams_amazon_country', 'com');
        $enable_no_follow   = get_option('ams_enable_no_follow_link') ? 'nofollow' : '';
        $buy_action         = get_option('ams_buy_action_btn');
        $ams_product_url    = get_post_meta($product_id, '_ams_product_url', true);

        // Adjust Amazon country for 'mx'
        if ("mx" === strtolower($ams_amazon_country)) {
            $ams_amazon_country = 'com.mx';
        }

        // Handle each option for `buy_action` - process ALL Amazon products (simple, variable, etc.)
        if ($buy_action === 'redirect' && !empty($ams_product_url)) {
            // Option 1: Tracked Amazon Redirect
            $query = parse_url($ams_product_url, PHP_URL_QUERY);
            $product_associate_tag = (!empty($query) ? '&' : '?') . 'tag=' . $ams_associate_tag;
            $amazon_url = $ams_product_url . $product_associate_tag;
            
            // Generate tracking URL through admin-post.php for proper click counting
            $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                'action' => 'cart_redirected_count',
                'id' => $product_id,
                'url' => $amazon_url
            ]);

            // Clone the entire original button structure with new URL and text
            return $this->clone_button_with_new_url($button, $tracking_url, $btn_text, $enable_no_follow);
        } elseif ($buy_action === 'cart_page' && !empty($ams_product_url)) {
            // Option 2: Tracked Amazon Cart Page Redirect
            $amazon_cart_url = $this->get_amazon_cart_url($product);
            
            // Generate tracking URL through admin-post.php for proper click counting
            $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                'action' => 'cart_redirected_count',
                'id' => $product_id,
                'url' => $amazon_cart_url
            ]);

            // Clone the entire original button structure with new URL and text
            return $this->clone_button_with_new_url($button, $tracking_url, $btn_text, $enable_no_follow);
        } elseif ($buy_action === 'multi_cart' || $buy_action === 'dropship') {
            // Option 3 or 4: Multi-Cart or Dropship
            return $button; // Return the default WooCommerce button
        }

        // Default fallback
        return $button;
    }

    public function fix_button_elements() {
        // Only add on shop/category pages
        if (!is_shop() && !is_product_category() && !is_product_tag()) {
            return;
        }
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Simple fix for buttons that should be links
            function fixButtons() {
                $('button:contains("Buy On Amazon")').each(function() {
                    var $button = $(this);
                    if (!$button.data('ams-fixed')) {
                        // Add href attribute to make it work like a link
                        var href = $button.attr('data-href') || $button.closest('form').attr('action');
                        if (href) {
                            $button.attr('href', href);
                            $button.removeAttr('type');
                            $button.data('ams-fixed', true);
                        }
                    }
                });
            }
            
            // Run the fix
            fixButtons();
            
            // Run after AJAX content loads
            $(document).ajaxComplete(function() {
                setTimeout(fixButtons, 500);
            });
        });
        </script>
        <?php
    }

    private function get_button_classes($button) {
        if (preg_match('/class="([^"]+)"/', $button, $matches)) {
            return $matches[1];
        }
        return 'button'; // Fallback class
    }

    // ========================================
    // THEME AUTO-DETECTION FUNCTIONS
    // ========================================
    // All functions related to automatic theme detection and button styling
    
    public function detect_theme_variable_hooks() {
        $theme = get_template();
        $detected_hooks = [];
        
        // FULLY AUTOMATIC HOOK DETECTION - NO HARDCODED THEMES!
        // The system will automatically detect hooks for any theme
        
        // FULLY AUTOMATIC HOOK DETECTION - NO HARDCODED THEMES!
        
        // Method 2: Scan active hooks for variable product related ones
        global $wp_filter;
        $variable_related_hooks = [];
        
        foreach ($wp_filter as $hook_name => $callbacks) {
            if (strpos($hook_name, 'variation') !== false || 
                strpos($hook_name, 'variable') !== false ||
                strpos($hook_name, 'add_to_cart') !== false ||
                strpos($hook_name, 'shop_loop') !== false) {
                
                $priority = 10; // Default priority
                if (!empty($callbacks->callbacks)) {
                    $priorities = array_keys($callbacks->callbacks);
                    $priority = min($priorities);
                }
                
                $variable_related_hooks[$hook_name] = $priority;
            }
        }
        
        // Combine known hooks with detected ones
        $detected_hooks = array_merge($detected_hooks, $variable_related_hooks);
        
        return $detected_hooks;
    }

    // --- Theme Button Classes Detection ---
    public function detect_theme_button_classes() {
        $theme = get_template();
        $detected_classes = [];
        
        // FULLY AUTOMATIC CSS CLASS DETECTION - NO HARDCODED THEMES!
        // The system will automatically detect button classes for any theme
        
        // Method 2: Scan theme CSS files for button-related classes - FIXED DETECTION
        $theme_css_files = $this->get_theme_css_files();
        $css_classes = [];
        
        foreach ($theme_css_files as $file) {
            if (file_exists($file)) {
                $content = file_get_contents($file);
                
                // FIXED: Find proper CSS class selectors (not random text)
                preg_match_all('/\.([a-zA-Z][a-zA-Z0-9_-]*)/', $content, $all_matches);
                $css_classes = array_merge($css_classes, $all_matches[1]);
                
                // Find button-related CSS classes
                preg_match_all('/\.([a-zA-Z][a-zA-Z0-9_-]*(?:button|btn|cart|add|product)[a-zA-Z0-9_-]*)/', $content, $matches);
                $css_classes = array_merge($css_classes, $matches[1]);
                
                // Find modifier classes (--icon, --text, --full, etc.)
                preg_match_all('/\.([a-zA-Z][a-zA-Z0-9_-]*--[a-zA-Z0-9_-]+)/', $content, $modifier_matches);
                $css_classes = array_merge($css_classes, $modifier_matches[1]);
                
                // Find common button patterns
                preg_match_all('/\.([a-zA-Z][a-zA-Z0-9_-]*(?:button|btn|cart|add|product|woo|shop)[a-zA-Z0-9_-]*)/', $content, $button_matches);
                $css_classes = array_merge($css_classes, $button_matches[1]);
            }
        }
        
        // Filter out invalid classes and keep only meaningful ones
        $css_classes = array_filter($css_classes, function($class) {
            return strlen($class) > 2 && !is_numeric($class) && !preg_match('/^[0-9]/', $class);
        });
        
        // Combine known classes with detected ones
        $detected_classes = array_merge($detected_classes, array_unique($css_classes));
        
        return array_unique($detected_classes);
    }

    public function get_theme_css_files() {
        $theme_dir = get_template_directory();
        $css_files = [];
        
        // Common CSS file locations for automatic detection
        $possible_files = [
            $theme_dir . '/style.css',
            $theme_dir . '/main-style.css',
            $theme_dir . '/assets/css/style.css',
            $theme_dir . '/assets/dist/css/main-style.css',
            $theme_dir . '/css/style.css',
            $theme_dir . '/woocommerce.css',
            $theme_dir . '/assets/woocommerce.css'
        ];
        
        foreach ($possible_files as $file) {
            if (file_exists($file)) {
                $css_files[] = $file;
            }
        }
        
        // Also scan for CSS files in subdirectories
        $css_files = array_merge($css_files, glob($theme_dir . '/**/*.css'));
        
        return array_unique($css_files);
    }

    // --- Auto-Detection Results Summary ---
    public function get_theme_detection_results() {
        $detected_hooks = $this->detect_theme_variable_hooks();
        $detected_classes = $this->detect_theme_button_classes();
        
        return [
            'theme' => get_template(),
            'theme_name' => wp_get_theme()->get('Name'),
            'hooks' => $detected_hooks,
            'classes' => $detected_classes,
            'scan_time' => current_time('mysql'),
            'css_files_found' => count($this->get_theme_css_files()),
            'selected_hook' => $this->get_best_variable_product_hook($detected_hooks),
            'selected_classes' => $this->get_best_button_classes($detected_classes),
            'implementation_status' => $this->get_implementation_status()
        ];
    }

    // ========================================
    // SIMPLE THEME META KEY SCANNER
    // ========================================
    
          public function scan_current_theme_meta_keys($force_refresh = false) {
          $current_theme = get_template();
          $theme_name = wp_get_theme()->get('Name');
          $theme_dir = get_template_directory();
          
          // Check for cached results (unless force refresh)
          if (!$force_refresh) {
              $cached_results = $this->get_cached_scan_results($current_theme);
              if ($cached_results !== false) {
                  return $cached_results;
              }
          }
          
          // 1. SCAN THEME FILES - Based on WordPress/WooCommerce theme structure
          $file_keys = $this->scan_theme_files_for_meta_keys($theme_dir, $current_theme);
          
          // 2. VERIFY IN DATABASE - Check which keys are actually used
          $verified_keys = $this->verify_meta_keys_in_database($file_keys);
          
          $results = [
              'theme_name' => $theme_name,
              'theme_slug' => $current_theme,
              'theme_dir' => $theme_dir,
              'found_keys' => $verified_keys,
              'scan_time' => current_time('mysql'),
              'files_scanned' => $this->get_scanned_files_list($theme_dir),
              'cache_status' => $force_refresh ? 'Fresh Scan' : 'New Scan'
          ];
          
          // Cache the results for 24 hours
          $this->cache_scan_results($current_theme, $results);
          
          return $results;
      }
      
      private function get_cached_scan_results($theme_slug) {
          $cache_key = 'ams_theme_scan_' . $theme_slug;
          $cached = get_transient($cache_key);
          
          if ($cached !== false) {
              // Add cache status to results
              $cached['cache_status'] = 'Cached Results';
              return $cached;
          }
          
          return false;
      }
      
      private function cache_scan_results($theme_slug, $results) {
          $cache_key = 'ams_theme_scan_' . $theme_slug;
          // Cache for 24 hours (86400 seconds)
          set_transient($cache_key, $results, 24 * HOUR_IN_SECONDS);
      }
      
      private function scan_theme_files_for_meta_keys($theme_dir, $theme_slug) {
          $found_keys = [];
          
          if (!is_dir($theme_dir)) {
              return $found_keys;
          }
          
          // WORDPRESS/WOOCOMMERCE THEME FILE STRUCTURE
          $critical_files = [
              // 1. MAIN THEME FILES
              'functions.php',                          // Main theme functions - most likely location
              
              // 2. WOOCOMMERCE INTEGRATION FILES
              'woocommerce.php',                        // WooCommerce theme integration
              'inc/woocommerce.php',                    // WooCommerce functions (common pattern)
              'includes/woocommerce.php',               // Alternative location
              'lib/woocommerce.php',                    // Some themes use lib/
              'framework/woocommerce.php',              // Framework-based themes
              
              // 3. WOOCOMMERCE TEMPLATE OVERRIDES
              'woocommerce/single-product.php',         // Single product page
              'woocommerce/content-single-product.php', // Product content template
              'woocommerce/single-product/product-image.php',        // Product images
              'woocommerce/single-product/product-thumbnails.php',   // Product thumbnails
              'woocommerce/single-product-reviews.php', // Product reviews
              
              // 4. CUSTOM POST TYPE & META HANDLERS
              'inc/custom-fields.php',                  // Custom fields definitions
              'includes/meta-boxes.php',                // Meta box definitions
              'lib/custom-fields.php',                  // Alternative location
              'framework/meta.php',                     // Framework meta handlers
              
              // 5. THEME-SPECIFIC WOOCOMMERCE FILES
              'inc/woo-functions.php',                  // WooCommerce helper functions
              'includes/woo-customizer.php',            // WooCommerce customizations
              'lib/woo-modifications.php',              // WooCommerce modifications
              
              // 6. HOOKS & FILTERS FILES
              'inc/hooks.php',                          // Theme hooks
              'includes/filters.php',                   // Theme filters
              'lib/actions.php',                        // Theme actions
          ];
          
          foreach ($critical_files as $file) {
              $file_path = $theme_dir . '/' . $file;
              if (file_exists($file_path) && is_readable($file_path)) {
                  $keys = $this->extract_meta_keys_from_file($file_path, $file);
                  $found_keys = array_merge($found_keys, $keys);
              }
          }
          
          // SCAN ENTIRE WOOCOMMERCE DIRECTORY
          $woo_dir = $theme_dir . '/woocommerce';
          if (is_dir($woo_dir)) {
              $woo_files = $this->get_all_php_files($woo_dir);
              foreach ($woo_files as $file_path) {
                  $relative_path = str_replace($theme_dir . '/', '', $file_path);
                  $keys = $this->extract_meta_keys_from_file($file_path, $relative_path);
                  $found_keys = array_merge($found_keys, $keys);
              }
          }
          
          return $this->deduplicate_keys($found_keys);
      }
      
      private function extract_meta_keys_from_file($file_path, $relative_path) {
          $content = @file_get_contents($file_path);
          if ($content === false) {
              return [];
          }
          
          $found_keys = [];
          
          // WORDPRESS META KEY PATTERNS - Based on WP/WooCommerce standards
          $patterns = [
              // 1. get_post_meta() calls - Most common
              '/get_post_meta\s*\(\s*[^,]+,\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 2. update_post_meta() calls - Where keys are set
              '/update_post_meta\s*\(\s*[^,]+,\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 3. add_post_meta() calls - Initial meta creation
              '/add_post_meta\s*\(\s*[^,]+,\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 4. Meta box definitions - register_meta(), add_meta_box()
              '/register_meta\s*\([^,]*,\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              '/add_meta_box\s*\([^,]*,\s*[^,]*,\s*[^,]*,\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 5. Array key definitions - 'meta_key' => 'value'
              '/[\'"]meta_key[\'"]\s*=>\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 6. Variable assignments - $meta_key = 'key'
              '/\$[a-zA-Z_][a-zA-Z0-9_]*\s*=\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              
              // 7. WooCommerce specific patterns
              '/wc_get_product_variation_attributes\s*\([^)]*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery)[a-zA-Z0-9_]*)[\'"]/',
              
              // 8. Custom field framework patterns (ACF, Meta Box, etc.)
              '/get_field\s*\(\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
              '/the_field\s*\(\s*[\'"]([a-zA-Z_][a-zA-Z0-9_]*(?:variation|image|gallery|photo|picture)[a-zA-Z0-9_]*)[\'"]/',
          ];
          
          foreach ($patterns as $pattern) {
              if (preg_match_all($pattern, $content, $matches)) {
                  foreach ($matches[1] as $meta_key) {
                      // Filter out generic words and validate key format
                      if ($this->is_valid_meta_key($meta_key)) {
                          $found_keys[] = [
                              'meta_key' => $meta_key,
                              'found_in_file' => $relative_path,
                              'source' => 'Theme File Scan'
                          ];
                      }
                  }
              }
          }
          
          return $found_keys;
      }
      
      private function is_valid_meta_key($key) {
          // Must be valid PHP variable name format
          if (!preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $key)) {
              return false;
          }
          
          // Must be longer than 3 characters
          if (strlen($key) <= 3) {
              return false;
          }
          
          // Exclude generic words
          $excluded = ['image', 'gallery', 'variation', 'photo', 'picture', 'meta', 'key', 'field'];
          if (in_array(strtolower($key), $excluded)) {
              return false;
          }
          
          // Must contain variation/image/gallery related terms
          $required_terms = ['variation', 'image', 'gallery', 'photo', 'picture', 'img', 'pic'];
          $key_lower = strtolower($key);
          $has_required_term = false;
          
          foreach ($required_terms as $term) {
              if (strpos($key_lower, $term) !== false) {
                  $has_required_term = true;
                  break;
              }
          }
          
          return $has_required_term;
      }
      
      private function get_all_php_files($directory) {
          $files = [];
          
          try {
              $iterator = new \RecursiveIteratorIterator(
                  new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
              );
              
              foreach ($iterator as $file) {
                  if ($file->isFile() && $file->getExtension() === 'php') {
                      $files[] = $file->getPathname();
                  }
              }
          } catch (\Exception $e) {
              // If directory scanning fails, return empty array
              return [];
          }
          
          return $files;
      }
      
      private function deduplicate_keys($keys) {
          $unique = [];
          $seen = [];
          
          foreach ($keys as $key_data) {
              $meta_key = $key_data['meta_key'];
              if (!isset($seen[$meta_key])) {
                  $unique[] = $key_data;
                  $seen[$meta_key] = true;
              }
          }
          
          return $unique;
      }
      
      private function verify_meta_keys_in_database($file_keys) {
          global $wpdb;
          
          $verified_keys = [];
          
          foreach ($file_keys as $key_data) {
              $meta_key = $key_data['meta_key'];
              
              // Check usage in database
              $usage_count = $wpdb->get_var($wpdb->prepare("
                  SELECT COUNT(*) 
                  FROM {$wpdb->postmeta} pm
                  INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                  WHERE p.post_type = 'product_variation'
                  AND pm.meta_key = %s
              ", $meta_key));
              
              $verified_keys[] = [
                  'meta_key' => $meta_key,
                  'usage_count' => (int)$usage_count,
                  'found_in_file' => $key_data['found_in_file'],
                  'source' => $key_data['source'],
                  'status' => $usage_count > 0 ? 'In Use' : 'Defined but Unused'
              ];
          }
          
          // Sort by usage count (most used first)
          usort($verified_keys, function($a, $b) {
              return $b['usage_count'] - $a['usage_count'];
          });
          
          return $verified_keys;
      }
      
      private function get_scanned_files_list($theme_dir) {
          // Return list of files that were scanned for debugging
          return [
              'theme_directory' => $theme_dir,
              'critical_files_checked' => 16,
              'woocommerce_templates_scanned' => true
          ];
      }

    // --- Helper: Best Hook Selection ---
    public function get_best_variable_product_hook($detected_hooks) {
        $theme = get_template();
        
        // Generic hook preferences (for unknown themes)
        $generic_preferences = [
            'woocommerce_single_variation',
            'woocommerce_variable_add_to_cart',
            'woocommerce_after_shop_loop_item',
            'woocommerce_loop_add_to_cart_link'
        ];
        
        foreach ($generic_preferences as $preferred_hook) {
            if (isset($detected_hooks[$preferred_hook])) {
                return [
                    'hook' => $preferred_hook,
                    'priority' => $detected_hooks[$preferred_hook],
                    'source' => 'generic_preference',
                    'confidence' => 'medium'
                ];
            }
        }
        
        // Find any variation-related hook
        foreach ($detected_hooks as $hook => $priority) {
            if (strpos($hook, 'variation') !== false || strpos($hook, 'variable') !== false) {
                return [
                    'hook' => $hook,
                    'priority' => $priority,
                    'source' => 'variation_related',
                    'confidence' => 'low'
                ];
            }
        }
        
        return null;
    }

    // --- Helper: Best Button Classes Selection ---
    public function get_best_button_classes($detected_classes) {
        $selected_classes = [];
        
        // Method 1: Analyze existing buttons in the DOM
        $existing_button_structure = $this->analyze_existing_buttons();
        
        if ($existing_button_structure) {
            $selected_classes = $existing_button_structure;
        } else {
            // Method 2: Intelligent class detection from CSS
            $selected_classes = $this->intelligent_class_detection($detected_classes);
        }
        
        return $selected_classes;
    }
    
    private function analyze_existing_buttons() {
        
        // Get the current page content
        ob_start();
        // We'll analyze this in JavaScript since we can't easily get DOM content in PHP
        return null; // Will be handled by JavaScript
    }
    
    // --- Helper: Intelligent Class Detection ---
    private function intelligent_class_detection($detected_classes) {
        // Intelligent detection based on class patterns - ENHANCED FOR UNKNOWN THEMES
        $selected_classes = [];
        
        // Find the most common button class pattern
        $button_patterns = [
            'primary' => ['button', 'btn', 'add-to-cart', 'single_add_to_cart_button', 'add_to_cart_button'],
            'wrapper' => ['button-wrap', 'btn-wrap', 'cart-btn', 'custom-cart-btn', 'product-btn'],
            'icon' => ['icon', 'cart-icon', 'btn-icon', 'button-icon'],
            'text' => ['text', 'btn-text', 'button-text', 'label']
        ];
        
        foreach ($button_patterns as $type => $patterns) {
            foreach ($patterns as $pattern) {
                foreach ($detected_classes as $class) {
                    // Skip screen-reader-text for text type
                    if ($type === 'text' && stripos($class, 'screen-reader') !== false) {
                        continue;
                    }
                    
                    if (strpos($class, $pattern) !== false) {
                        $selected_classes[$type] = [
                            'class' => $class,
                            'source' => 'intelligent_detection',
                            'confidence' => 'high'
                        ];
                        break 2; // Break out of both loops
                    }
                }
            }
        }
        
        // Fallback to most common classes - ALWAYS provide a fallback
        if (empty($selected_classes['primary'])) {
            $selected_classes['primary'] = [
                'class' => 'button', // Always fallback to 'button'
                'source' => 'fallback_detection',
                'confidence' => 'medium'
            ];
        }
        
        // Always provide wrapper class
        if (empty($selected_classes['wrapper'])) {
            $selected_classes['wrapper'] = [
                'class' => 'button', // Always fallback to 'button'
                'source' => 'fallback_detection',
                'confidence' => 'medium'
            ];
        }
        
        return $selected_classes;
    }

    public function normalize_button_size($selected_classes, $theme) {
        return $selected_classes; // Let JavaScript handle the sizing automatically
    }

    // --- Helper: Implementation Status Check ---
    public function get_implementation_status() {
        $theme = get_template();
        $use_custom_button = get_option('ams_use_custom_button', '0');
        
        if ($use_custom_button !== '1') {
            return [
                'status' => 'disabled',
                'message' => 'Custom button functionality is disabled',
                'active_hooks' => [],
                'active_classes' => []
            ];
        }
        
        $detected_hooks = $this->detect_theme_variable_hooks();
        $detected_classes = $this->detect_theme_button_classes();
        $selected_hook = $this->get_best_variable_product_hook($detected_hooks);
        $selected_classes = $this->get_best_button_classes($detected_classes);
        
        $active_hooks = [];
        $active_classes = [];
        
        if ($selected_hook) {
            $active_hooks[] = [
                'hook' => $selected_hook['hook'],
                'priority' => $selected_hook['priority'] - 1, // Our priority (before theme)
                'source' => $selected_hook['source'],
                'confidence' => $selected_hook['confidence']
            ];
        }
        
        foreach ($selected_classes as $type => $class_info) {
            $active_classes[] = [
                'type' => $type,
                'class' => $class_info['class'],
                'source' => $class_info['source'],
                'confidence' => $class_info['confidence']
            ];
        }
        
        return [
            'status' => 'active',
            'message' => 'Plugin is actively implementing theme-specific solution',
            'active_hooks' => $active_hooks,
            'active_classes' => $active_classes,
            'theme' => $theme
        ];
    }

    // --- Hook Setup and Registration ---
    private function setup_intelligent_hooks() {
        $detected_hooks = $this->detect_theme_variable_hooks();
        $selected_hook = $this->get_best_variable_product_hook($detected_hooks);
        
        if ($selected_hook) {
            // Add the hook with our priority (before theme's priority)
            add_action($selected_hook['hook'], [$this, 'universal_variable_button'], $selected_hook['priority'] - 1);
            
            // Also add a filter to intercept theme's button output
            add_filter('woocommerce_loop_add_to_cart_link', [$this, 'intercept_theme_button'], 999, 2);
        }
    }

    // --- Variable Product Button Generation ---
    public function universal_variable_button() {
        global $product;
        
        // Check if auto-detection is enabled
        $auto_detection_enabled = get_option('ams_enable_auto_detection', '1') === '1';
        if (!$auto_detection_enabled) {
            return; // Skip if auto-detection is disabled
        }
        
        // Only on shop/category pages
        if (!is_shop() && !is_product_category() && !is_product_tag()) {
            return;
        }
        
        // Only for variable products
        if (!$product || !$product->is_type('variable')) {
            return;
        }
        
        $product_id = $product->get_id();
        $ams_product_url = get_post_meta($product_id, '_ams_product_url', true);
        
        // Only if product has Amazon URL
        if (empty($ams_product_url)) {
            return;
        }
        
        // Get intelligent class selection
        $detected_classes = $this->detect_theme_button_classes();
        $selected_classes = $this->get_best_button_classes($detected_classes);
        $selected_classes = $this->normalize_button_size($selected_classes, get_template());
        
        // Get plugin settings
        $btn_text = get_option('ams_buy_now_label', 'Buy Now');
        $ams_associate_tag = get_option('ams_associate_tag');
        $ams_amazon_country = get_option('ams_amazon_country', 'com');
        $enable_no_follow = get_option('ams_enable_no_follow_link') ? 'nofollow' : '';
        $buy_action = get_option('ams_buy_action_btn');
        
        // Adjust Amazon country for 'mx'
        if ("mx" === strtolower($ams_amazon_country)) {
            $ams_amazon_country = 'com.mx';
        }
        
        // Handle each option for `buy_action`
        if ($buy_action === 'redirect' && !empty($ams_product_url)) {
            // Build Amazon URL with associate tag
            $query = parse_url($ams_product_url, PHP_URL_QUERY);
            $product_associate_tag = (!empty($query) ? '&' : '?') . 'tag=' . $ams_associate_tag;
            $amazon_url = $ams_product_url . $product_associate_tag;
            
            // Generate tracking URL
            $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                'action' => 'cart_redirected_count',
                'id' => $product_id,
                'url' => $amazon_url
            ]);
            
            // Build classes dynamically based on theme
            $primary_class = isset($selected_classes['primary']) ? $selected_classes['primary']['class'] : 'button';
            $secondary_class = isset($selected_classes['secondary']) ? $selected_classes['secondary']['class'] : '';
            $wrapper_class = isset($selected_classes['wrapper']) ? $selected_classes['wrapper']['class'] : 'button';
            $icon_class = isset($selected_classes['icon']) ? $selected_classes['icon']['class'] : '';
            $text_class = isset($selected_classes['text']) ? $selected_classes['text']['class'] : '';
            $size_class = isset($selected_classes['size']) ? $selected_classes['size']['class'] : '';
            
            // Use the same cloning approach as simple products for 1:1 positioning
            // Create a button structure that matches the theme's original structure
            $original_button = '<div class="c-product-grid__atc-wrap c-product-grid__atc-wrap--icon">';
            $original_button .= '<a href="' . esc_url($product->add_to_cart_url()) . '" class="c-product-grid__atc c-product-grid__atc--icon button" aria-label="' . esc_attr($btn_text) . '">';
            $original_button .= '<i class="ip-cart c-product-grid__atc-icon"></i>';
            $original_button .= '<span class="c-product-grid__atc-text">' . esc_html($btn_text) . '</span>';
            $original_button .= '</a>';
            $original_button .= '</div>';
            
            // Clone the original button with new URL - same as simple products
            echo $this->clone_button_with_new_url($original_button, $tracking_url, $btn_text, $enable_no_follow);
            
            // FULLY AUTOMATIC BUTTON STRUCTURE ANALYSIS
            echo '<script type="text/javascript">
            jQuery(document).ready(function($) {
                // Analyze existing buttons to replicate their exact structure
                function analyzeExistingButtons() {
                    var buttonAnalysis = {
                        hasWrapper: false,
                        wrapperClass: "",
                        buttonClass: "",
                        structure: "single" // "single" or "wrapped"
                    };
                    
                    // Look for existing buttons on the page
                    var existingButtons = $("a.button, button.button, .add-to-cart, .single_add_to_cart_button").not(".ams-amazon-button");
                    
                    if (existingButtons.length > 0) {
                        var firstButton = existingButtons.first();
                        var parent = firstButton.parent();
                        
                        // Check if button is wrapped in a div
                        if (parent.is("div") && parent.hasClass("button")) {
                            buttonAnalysis.hasWrapper = true;
                            buttonAnalysis.wrapperClass = parent.attr("class");
                            buttonAnalysis.structure = "wrapped";
                        } else {
                            buttonAnalysis.structure = "single";
                        }
                        
                        buttonAnalysis.buttonClass = firstButton.attr("class");
                        
                        // Apply the detected structure to our Amazon buttons
                        applyDetectedStructure(buttonAnalysis);
                    }
                }
                
                function applyDetectedStructure(analysis) {
                    var amazonButtons = $(".ams-amazon-button");
                    
                    amazonButtons.each(function() {
                        var $button = $(this);
                        var $parent = $button.parent();
                        
                        if (analysis.structure === "single" && $parent.is("div")) {
                            // Remove wrapper div - make it single element
                            $button.unwrap();
                        } else if (analysis.structure === "wrapped" && !$parent.is("div")) {
                            // Add wrapper div
                            $button.wrap("<div class=\"" + analysis.wrapperClass + "\"></div>");
                        }
                    });
                }
                
                // Run analysis after page load
                setTimeout(analyzeExistingButtons, 500);
                
                // Also run on AJAX complete
                $(document).ajaxComplete(function() {
                    setTimeout(analyzeExistingButtons, 500);
                });
            });
            </script>';
            
            // Add JavaScript to hide theme's button if it still appears
            echo '<script type="text/javascript">
            jQuery(document).ready(function($) {
                // Hide theme\'s original button if it appears after our button
                setTimeout(function() {
                    // Universal selectors that work with most themes
                    $(".woocommerce-variation-add-to-cart button[type=\"submit\"]").hide();
                    $(".woocommerce-variation-add-to-cart .single_add_to_cart_button").hide();
                }, 100);
                
                // Universal AJAX complete handler for dynamic content
                $(document).ajaxComplete(function() {
                    setTimeout(function() {
                        // Universal selectors that work with most themes
                        $(".woocommerce-variation-add-to-cart button[type=\"submit\"]").hide();
                        $(".woocommerce-variation-add-to-cart .single_add_to_cart_button").hide();
                    }, 100);
                });
            });
            </script>';
            
            // Universal theme button handling - no theme-specific code
        }
    }

    // --- Theme Button Interception and Modification ---
    public function intercept_theme_button($button, $product) {
        // Check if auto-detection is enabled
        $auto_detection_enabled = get_option('ams_enable_auto_detection', '1') === '1';
        if (!$auto_detection_enabled) {
            return $button; // Don't intercept if auto-detection is disabled
        }
        
        // Only intercept for variable products on shop/category pages
        if (!is_shop() && !is_product_category() && !is_product_tag()) {
            return $button;
        }
        
        if (!$product || !$product->is_type('variable')) {
            return $button;
        }
        
        $product_id = $product->get_id();
        $ams_product_url = get_post_meta($product_id, '_ams_product_url', true);
        
        // Only if product has Amazon URL
        if (empty($ams_product_url)) {
            return $button;
        }
        
        // Get plugin settings
        $btn_text = get_option('ams_buy_now_label', 'Buy Now');
        $ams_associate_tag = get_option('ams_associate_tag');
        $ams_amazon_country = get_option('ams_amazon_country', 'com');
        $enable_no_follow = get_option('ams_enable_no_follow_link') ? 'nofollow' : '';
        $buy_action = get_option('ams_buy_action_btn');
        
        // Adjust Amazon country for 'mx'
        if ("mx" === strtolower($ams_amazon_country)) {
            $ams_amazon_country = 'com.mx';
        }
        
        // Handle each option for `buy_action`
        if ($buy_action === 'redirect' && !empty($ams_product_url)) {
            // Build Amazon URL with associate tag
            $query = parse_url($ams_product_url, PHP_URL_QUERY);
            $product_associate_tag = (!empty($query) ? '&' : '?') . 'tag=' . $ams_associate_tag;
            $amazon_url = $ams_product_url . $product_associate_tag;
            
            // Generate tracking URL
            $tracking_url = admin_url('admin-post.php') . '?' . http_build_query([
                'action' => 'cart_redirected_count',
                'id' => $product_id,
                'url' => $amazon_url
            ]);
            
            // Get intelligent class selection
            $detected_classes = $this->detect_theme_button_classes();
            $selected_classes = $this->get_best_button_classes($detected_classes);
            $selected_classes = $this->normalize_button_size($selected_classes, get_template());
            
            // Build classes dynamically based on theme
            $primary_class = isset($selected_classes['primary']) ? $selected_classes['primary']['class'] : 'button';
            $secondary_class = isset($selected_classes['secondary']) ? $selected_classes['secondary']['class'] : '';
            $wrapper_class = isset($selected_classes['wrapper']) ? $selected_classes['wrapper']['class'] : 'button';
            $icon_class = isset($selected_classes['icon']) ? $selected_classes['icon']['class'] : '';
            $text_class = isset($selected_classes['text']) ? $selected_classes['text']['class'] : '';
            $size_class = isset($selected_classes['size']) ? $selected_classes['size']['class'] : '';
            
            // FULLY AUTOMATIC CLASS DETECTION - Use detected classes + universal identifier
            $all_button_classes = 'button ams-amazon-button'; // Start with basic button class
            if (!empty($primary_class)) {
                $all_button_classes = $primary_class . ' ams-amazon-button';
            }
            
            // FULLY AUTOMATIC WRAPPER DETECTION
            $all_wrapper_classes = 'button'; // Default wrapper
            if (!empty($wrapper_class)) {
                $all_wrapper_classes = $wrapper_class;
            }
            
            // Return the custom button with intelligent class selection
            $custom_button = '<div class="' . esc_attr($all_wrapper_classes) . '">';
            $custom_button .= '<a href="' . esc_url($tracking_url) . '" class="' . esc_attr($all_button_classes) . '" aria-label="' . esc_attr($btn_text) . '" rel="' . esc_attr($enable_no_follow) . '" target="_blank" role="button">';
            
            if (!empty($icon_class)) {
                $custom_button .= '<i class="' . esc_attr($icon_class) . '"></i>';
            }
            
            if (!empty($text_class)) {
                $custom_button .= '<span class="' . esc_attr($text_class) . '">' . esc_html($btn_text) . '</span>';
            } else {
                $custom_button .= esc_html($btn_text);
            }
            
            $custom_button .= '</a>';
            $custom_button .= '</div>';
            
            return $custom_button;
        }
        
        return $button;
    }
    
    // ========================================
    // END OF THEME AUTO-DETECTION FUNCTIONS
    // ========================================

    private function clone_button_with_new_url($original_button, $new_url, $new_text, $enable_no_follow = '') {
        // Step 1: Replace the href attribute with our tracking URL
        $button = preg_replace('/href="[^"]*"/', 'href="' . esc_url($new_url) . '"', $original_button);
        
        // Step 2: Remove WooCommerce add-to-cart functionality attributes
        $button = preg_replace('/\s*data-quantity="[^"]*"/', '', $button);
        $button = preg_replace('/\s*data-product_id="[^"]*"/', '', $button);
        $button = preg_replace('/\s*data-product_sku="[^"]*"/', '', $button);
        $button = preg_replace('/\s*data-success_message="[^"]*"/', '', $button);
        
        // Step 3: Remove WooCommerce CSS classes that trigger AJAX behavior
        $button = preg_replace('/\s*ajax_add_to_cart/', '', $button);
        $button = preg_replace('/\s*add_to_cart_button/', '', $button);
        $button = preg_replace('/\s*product_type_simple/', '', $button);
        
        // Step 3.5: FULLY AUTOMATIC THEME CSS DETECTION - NO HARDCODED CSS!
        // Only remove classes that break functionality, preserve ALL theme styling
        $button = preg_replace('/\s*button[^"]*disabled[^"]*/', '', $button);
        $button = preg_replace('/\s*button[^"]*inactive[^"]*/', '', $button);
        
        // Add Amazon class only to <a> tags (actual buttons), not wrapper divs
        // This prevents interference with theme's image layout
        $button = preg_replace('/<a([^>]*class="([^"]*)")([^>]*)>/', '<a$1class="$2 ams-amazon-button"$3>', $button);
        
        // Step 4: Add target="_blank" if not present
        if (strpos($button, 'target=') === false) {
            $button = preg_replace('/<a([^>]*)>/', '<a$1 target="_blank">', $button);
        }
        
        // Step 5: Handle rel attribute (nofollow setting)
        if (!empty($enable_no_follow)) {
            if (strpos($button, 'rel=') !== false) {
                // Replace existing rel attribute
                $button = preg_replace('/rel="[^"]*"/', 'rel="' . esc_attr($enable_no_follow) . '"', $button);
            } else {
                // Add rel attribute
                $button = preg_replace('/<a([^>]*)>/', '<a$1 rel="' . esc_attr($enable_no_follow) . '">', $button);
            }
        }
        
        // Step 6: ENHANCED TEXT REPLACEMENT - Handle all button text patterns
        // Replace text in spans with "text" in class name (but not screen-reader-text)
        $button = preg_replace('/(<span[^>]*class="[^"]*text[^"]*"(?!.*screen-reader)[^>]*>)([^<]*)(<\/span>)/i', 
                              '$1' . esc_html($new_text) . '$3', $button);
        
        // Replace "Add to cart" text directly
        $button = preg_replace('/>\s*Add to cart\s*</i', '>' . esc_html($new_text) . '<', $button);
        
        // Replace text in any span that contains "Add to cart"
        $button = preg_replace('/(<span[^>]*>)([^<]*Add[^<]*cart[^<]*?)(<\/span>)/i', 
                              '$1' . esc_html($new_text) . '$3', $button);
        
        // Replace any button text that's not in a span
        $button = preg_replace('/>([^<]*Add[^<]*cart[^<]*?)</i', '>' . esc_html($new_text) . '<', $button);
        
        // Replace text in button elements
        $button = preg_replace('/(<button[^>]*>)([^<]*)(<\/button>)/i', 
                              '$1' . esc_html($new_text) . '$3', $button);
        
        // Handle aria-label attributes
        $button = preg_replace('/aria-label="[^"]*Add[^"]*cart[^"]*"/', 
                              'aria-label="' . esc_attr($new_text) . '"', $button);
        
        // CRITICAL FIX: Remove screen-reader-text class to make text visible
        $button = preg_replace('/\s*class="[^"]*screen-reader-text[^"]*"/', '', $button);
        $button = preg_replace('/\s*screen-reader-text\s*/', ' ', $button);
        
        // Fallback: If no text was replaced, add the text at the end of the button
        if (strpos($button, $new_text) === false) {
            $button = preg_replace('/(<a[^>]*>)([^<]*)(<\/a>)/i', 
                                  '$1' . esc_html($new_text) . '$3', $button);
        }
        
        return $button;
    }

    private function build_amazon_url_with_tag($url, $tag) {
		$parts = parse_url($url);
		$params = array();
		if (isset($parts['query']) && $parts['query'] !== '') {
			parse_str($parts['query'], $params);
		}
		$params['tag'] = $tag;

		$scheme = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
		$host = isset($parts['host']) ? $parts['host'] : '';
		$port = isset($parts['port']) ? ':' . $parts['port'] : '';
		$path = isset($parts['path']) ? $parts['path'] : '';
		$query = http_build_query($params);
		$fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

		return $scheme . $host . $port . $path . ($query ? '?' . $query : '') . $fragment;
	}

    ////// Admin side functions //////
    public function total_count_products_add_to_cart( $product_id ) {
        $count_key     = 'ams_product_added_to_cart';
        $product_count = get_post_meta( $product_id, $count_key, true );
        $product_count++;

        update_post_meta( $product_id, $count_key, $product_count );
    }

    public function visitor_record() {
        global $product;

        // Check if $product is set and is a valid WC_Product object
        if (!isset($product) || !is_a($product, 'WC_Product')) {
            return; // Exit early if not on a product page
        }

        // Safely retrieve the product ID
        $post_id = $product->get_id();

        // Check if $post_id is valid
        if (!$post_id) {
            return; // Exit early if no valid product ID is found
        }

        // Get user's IP address and user agent for better tracking
        $user_ip = $this->get_user_ip();
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '';
        $user_fingerprint = md5($user_ip . $user_agent);
        
        $current_time = time();
        $view_window = 1800; // 30 minutes window to prevent duplicate view counting
        
        // Check if this user (by IP+UA fingerprint) viewed this product recently
        $should_count_view = $this->should_count_interaction('view', $post_id, $user_fingerprint, $view_window);
        
        // Only count view if not viewed recently by this user
        if ($should_count_view) {
            // Define the meta key and safely get the current count
            $count_key = 'ams_product_views_count';
            $count = get_post_meta($post_id, $count_key, true);

            // Ensure $count is numeric before incrementing
            if (!is_numeric($count)) {
                $count = 0; // Initialize count if meta doesn't exist or is invalid
            }

            // Increment and update the post meta
            $count++;
            update_post_meta($post_id, $count_key, $count);
            
            // Record this view in database
            $this->record_interaction('view', $post_id, $user_fingerprint, $current_time);
        }
    }

    /**
     * Get user's real IP address
     */
    private function get_user_ip() {
        $ip_keys = array('HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
    }

    /**
     * Check if we should count this interaction based on recent activity
     */
    private function should_count_interaction($type, $post_id, $user_fingerprint, $time_window) {
        $option_key = 'ams_tracking_' . $type . '_' . $post_id . '_' . $user_fingerprint;
        $last_interaction = get_option($option_key, 0);
        $current_time = time();
        
        return ($current_time - $last_interaction) > $time_window;
    }

    /**
     * Record interaction in database
     */
    private function record_interaction($type, $post_id, $user_fingerprint, $timestamp) {
        $option_key = 'ams_tracking_' . $type . '_' . $post_id . '_' . $user_fingerprint;
        update_option($option_key, $timestamp);
    }
    ////// Admin side functions //////
}