<?php

/**
 * Handles the checkout.
 * 
 * @since 12.4.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}
if (!class_exists('FGF_Checkout_Handler')) {

	/**
	 * Class.
	 * 
	 * @since 12.4.0
	 */
	class FGF_Checkout_Handler {

		/**
		 * Force gifts added.
		 * 
		 * @since 12.4.0
		 * @var boolean
		 */
		public static $force_gifts_added = false;

		/**
		 * Class.
		 * 
		 * @since 12.4.0
		 */
		public static function init() {
			switch (get_option('fgf_settings_force_checkout_type')) {
				case '4':
					// Add remaining manual gifts to the cart automatically.
					add_action('wp', array( __CLASS__, 'maybe_add_force_gifts_to_cart' ));
					// Add remaining manual gifts to the cart automatically via checkout ajax.
					add_action('woocommerce_review_order_before_cart_contents', array( __CLASS__, 'maybe_add_force_gifts_to_cart_ajax' ));          
					break;

				case '3':
				case '2':
					if ('2' === get_option('fgf_settings_force_checkout_type')) {
						// Validate the checkout to ensure that any remaining gifts are added to the cart.
						add_action( 'woocommerce_after_checkout_validation', array( __CLASS__, 'checkout_validation' ), 10, 2 );
					}

					// Render the  in the checkout page.
					add_action('woocommerce_before_checkout_form', array( __CLASS__, 'render_force_gifts_popup_content' ), 20);
					
					break;
			}
		}

		/**
		 * Render the force gifts popup contents.
		 * 
		 * @since 12.4.0
		 */
		public static function render_force_gifts_popup_content() {
			echo '<div id="fgf-checkout-force-gifts-popup-wrapper">';
			self::render_gift_products();
			echo '</div>';
		}

		/**
		 * Render the force gifts products contents.
		 * 
		 * @since 12.4.0
		 */
		public static function render_gift_products() {
			// Restrict the display of the gift products when the maximum gifts count reached.
			if (!FGF_Rule_Handler::manual_product_exists() || !FGF_Rule_Handler::get_unclaimed_gift_products_count()) {
				return;
			}

			// Return if data args does not exists.
			$data_args = FGF_Gift_Products_Handler::get_gift_product_data();
			if (!$data_args) {
				return;
			}

			$data_args['mode'] = 'popup';
			$data_args['popup_location'] = 'forced_checkout_gifts';         
			$data_args['permalink'] =  wc_get_page_permalink('checkout') ;

			if ('2'===get_option('fgf_settings_force_checkout_type')) {
				$data_args['notice'] = str_replace('[remaining_gifts]', FGF_Rule_Handler::get_unclaimed_gift_products_count(), get_option('fgf_settings_force_gifts_popup_warning_message'));
			}

			// Display Gift Products popup layout.
			fgf_get_template('checkout/force-gifts-popup-layout.php', array( 'data_args' => $data_args ));
		}
		
		/**
		 * Validate the checkout to ensure that any remaining gifts are added to the cart.
		 *
		 * @since 12.4.0
		 * @param [array] $data
		 * @param [WP_Error] $errors
		 */
		public static function checkout_validation( $data, $errors ) {
			// Return if cart object is not initialized.
			// Or cart contents count is empty.
			if (!is_object(WC()->cart) || WC()->cart->get_cart_contents_count() == 0) {
				return;
			}

			// Return if the gifts added and reached the per order count.
			if (!FGF_Rule_Handler::get_unclaimed_gift_products_count()) {
				return;
			}

			$error = str_replace('[remaining_gifts]', FGF_Rule_Handler::get_unclaimed_gift_products_count(), get_option('fgf_settings_remaining_gifts_checkout_validation_error_message'));

			// Add checkout validation error notice.
			wc_add_notice($error, 'error');
		}
		
		/**
		 * Add remaining manual gifts to the cart automatically when calling checkout via ajax.
		 * 
		 * @since 12.4.0
		 * */
		public static function maybe_add_force_gifts_to_cart_ajax() {
			if (!isset($_REQUEST['payment_method']) && !isset($_REQUEST['woocommerce-cart-nonce'])) {
				return;
			}

			self::add_force_gifts_to_cart();
		}

		/**
		 * Add remaining manual gifts to the cart automatically when page loaded.
		 * 
		 * @since 12.4.0
		 * */
		public static function maybe_add_force_gifts_to_cart() {
			if (isset($_REQUEST['payment_method']) || isset($_REQUEST['woocommerce-cart-nonce'])) {
				return;
			}

			self::add_force_gifts_to_cart();
		}

		/**
		 * Add remaining manual gifts to the cart automatically.
		 * 
		 * @since 12.4.0
		 * */
		public static function add_force_gifts_to_cart() {
			// Return if the current page is not a checkout page.
			if (!is_checkout()) {
				return;
			}

			// Don't add automatic gift products when it is already executed.
			if (self::$force_gifts_added) {
				return;
			}

			// Return if cart object is not initialized.
			// Or cart contents count is empty.
			if (!is_object(WC()->cart) || WC()->cart->get_cart_contents_count() == 0) {
				return;
			}

			$remaining_gifts_count= FGF_Rule_Handler::get_unclaimed_gift_products_count();
			// Return if the gifts added and reached the per order count.
			if (!$remaining_gifts_count) {
				return;
			}

			// Return if the gifts does not exits.
			$gift_products = FGF_Rule_Handler::get_overall_manual_gift_products();
			if (!fgf_check_is_array($gift_products)) {
				return;
			}

			$gifts_added=0;
			$gifts_added_by_rules=array();
			foreach ($gift_products as $gift_product) {
				// Skip gifts that are either invalid for the cart or already present in it.
				if ($gift_product['hide_add_to_cart']) {
					continue;
				}

				// Exit the loop once the necessary gifts have been added.
				if (!$remaining_gifts_count) {
					break;
				}

				// Skip gifts that are already reached the current rule maximum count.
				$current_rule_maximum_quantity=FGF_Rule_Handler::get_rule_max_quantity($gift_product['rule_id']);
				if (isset($gifts_added_by_rules[$gift_product['rule_id']])&& $gifts_added_by_rules[$gift_product['rule_id']]>=$current_rule_maximum_quantity) {
					continue;
				}

				$rule = fgf_get_rule($gift_product['rule_id']);
				// Skip processing gifts if the rule ID is incorrect.
				if (!$rule->exists()) {
					continue;
				}

				// Skip processing if the gifts of manual and coupon manual are reached maximum per order count. 
				if (in_array($rule->get_rule_type(), array( '1', '6' )) && FGF_Rule_Handler::check_per_order_count_exists()) {
					continue;
				}
				
				$product_ids=!empty($gift_product['variation_ids'])? $gift_product['variation_ids']: array( $gift_product['product_id'] );
				$buy_product_id=isset($gift_product['buy_product_id'])?$gift_product['buy_product_id']:'';
				$coupon_id=isset($gift_product['coupon_id'])?$gift_product['coupon_id']:'';
				foreach ($product_ids as $product_id) {
					// Skip processing gifts if the rule count reached.
					if (isset($gifts_added_by_rules[$gift_product['rule_id']])&& $gifts_added_by_rules[$gift_product['rule_id']]>=$current_rule_maximum_quantity) {
						break;
					}

					// Skip processing gifts if the rule ID is incorrect.
					if (!fgf_rule_product_exists($rule, $product_id, $buy_product_id, $coupon_id)) {
						continue;
					}

					$cart_item_data = array(
						'fgf_gift_product' => array(
							'mode' => $rule->get_rule_mode(),
							'rule_id' => $gift_product['rule_id'],
							'product_id' => $product_id,
							'buy_product_id' => $buy_product_id,
							'coupon_id' => $coupon_id,
							/**
							 * This hook is used to alter the manual gift product price.
							 * 
							 * @since 12.4.0
							 */
							'price' => apply_filters('fgf_manual_gift_product_price', fgf_get_free_gift_product_price(), $gift_product['rule_id'], $product_id),
						),
					);
					
					$remaining_current_rule_quantity=isset($gifts_added_by_rules[$gift_product['rule_id']])?$current_rule_maximum_quantity-$gifts_added_by_rules[$gift_product['rule_id']]:$current_rule_maximum_quantity;
					$gift_product['qty']=$gift_product['qty']>$remaining_current_rule_quantity?$remaining_current_rule_quantity:$gift_product['qty'];
					$quantity=$remaining_gifts_count >= $gift_product['qty'] ? $gift_product['qty'] : $remaining_gifts_count;
					
					// Add gifts to the cart.
					WC()->cart->add_to_cart($product_id, $quantity, 0, array(), $cart_item_data);
					
					$remaining_gifts_count-=$quantity;
					$gifts_added_by_rules[$gift_product['rule_id']]=isset($gifts_added_by_rules[$gift_product['rule_id']])?$gifts_added_by_rules[$gift_product['rule_id']]+$quantity:$quantity;
					++$gifts_added;
				}               
				
			}

			if ($gifts_added) {
				$notice = str_replace('[added_remaining_gifts_count]', $gifts_added, get_option('fgf_settings_force_gifts_success_message'));

				// Gifts added success notice.
				fgf_add_wc_notice($notice);
				FGF_Notices_Handler::add_notice($notice);

				// Reset the rule data after the gifts are added.
				FGF_Rule_Handler::reset();
			}

			self::$force_gifts_added = true;
		}       
	}

	FGF_Checkout_Handler::init();
}
