# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

Gallery Custom Links is a WordPress plugin (Pro version) that allows users to link images from galleries to specified URLs. It works with WordPress Gallery, Gutenberg, Meow Gallery, and other gallery plugins. The plugin processes HTML content to inject custom links into gallery images.

**Plugin Name:** Gallery Custom Links (Pro)
**Version:** 2.2.5
**Package Manager:** pnpm (migrated from Yarn)
**Text Domain:** gallery-custom-links
**Prefix:** mgcl

## Build and Development Commands

### Build Commands
- `pnpm dev` - Run Webpack in watch mode for development
- `pnpm build` - Build for production with Webpack
- `pnpm zip` - Create distribution zip (excludes node_modules, .git, .svn, logs)

### Dependencies
- **PHP Dependencies:** Managed via Composer
  - `imangazaliev/didom` - Fast HTML parser
  - `kub-at/php-simple-html-dom-parser` - Reliable HTML parser
- **JavaScript Dependencies:** Managed via pnpm
  - React 18 + React DOM
  - @tanstack/react-query for state management
  - styled-components for styling
  - Babel for transpilation

### File Structure

**Core PHP Files:**
- `gallery-custom-links-pro.php` - Main plugin entry point
- `classes/init.php` - Autoloader and initialization
- `classes/core.php` - Main plugin logic (linkify, parsing engines, REST API)
- `classes/linker.php` - Link injection logic
- `classes/admin.php` - WordPress admin integration
- `classes/rest.php` - REST API endpoints
- `classes/button/` - Button integration for different gallery types

**Frontend (React):**
- `app/js/index.js` - Main React entry point
- `app/js/components/Settings.js` - Plugin settings UI
- `app/js/components/EditLinkField.js` - Media library link editor
- `app/index.js` and `app/vendor.js` - Webpack build outputs

**Common Library:**
- `common/` - Shared MeowApps framework (admin, REST, licensing, etc.)
- `common/js/` - Shared React components and utilities

**Premium Features:**
- `premium/` - Pro-only features (library extensions, core enhancements)

**External Dependencies:**
- `@neko-ui` - Custom UI library located at `../neko-ui/` (resolved via Webpack alias)

## Architecture

### HTML Parsing Engines

The plugin offers three parsing strategies (controlled by `mgcl_parsing_engine` option):

1. **HtmlDomParser** (default) - Most reliable, handles malformed HTML
2. **DiDom** - Faster but requires valid HTML
3. **Javascript** - Client-side only, no server-side HTML rewriting

### Processing Modes

- **Output Buffering (OB) Mode** (`mgcl_obmode`): Processes entire page via `ob_start()`
- **Content Filter Mode**: Only processes `the_content` filter

### Link Resolution Flow

1. Detect images in HTML (via CSS selectors: `img` tags)
2. Resolve media ID from:
   - `data-attachment-id` attribute
   - `wp-image-{ID}` CSS class
   - Database lookup via URL matching
3. Fetch metadata (`_gallery_link_url`, `_gallery_link_target`, `_gallery_link_rel`, `_gallery_link_aria`)
4. Inject or replace links using `Meow_MGCL_Linker`

### Key Filters and Hooks

- `gallery_custom_links_enabled` - Enable/disable plugin conditionally
- `gallery_custom_links_classes` - Target specific CSS selectors (default: all images)
- `mgcl_linkers` - Custom link injection logic
- `mgcl_button_linker` - Custom button rendering for galleries
- `mgl_link_attributes` - Integration with Meow Gallery

### Autoloader Pattern

Classes are autoloaded based on naming convention:
- `Meow_MGCL_*` → `classes/*.php`
- `MeowCommon_*` → `common/*.php`
- `MeowCommonPro_*` → `common/premium/*.php`
- `MeowPro_MGCL_*` → `premium/*.php`

## Development Workflow

### Working with PHP

1. Main plugin logic is in `classes/core.php` - handles visitor requests, admin initialization, and REST API
2. Link injection happens in `classes/linker.php` - modifies DOM nodes to add/update anchor tags
3. Admin fields are registered in `classes/admin.php` - attachment fields for URL, target, rel, aria-label
4. Always check `$this->parsingEngine` when working with DOM manipulation (HtmlDomParser vs DiDom have different APIs)

### Working with React/JavaScript

1. Entry point: `app/js/index.js`
2. React components use `wp.element` (WordPress-provided React)
3. Wrapped with `<NekoUI>` provider from `@neko-ui`
4. Settings page: Renders into `#mgcl-admin-settings` container
5. Media library fields: Renders into `.mgcl-edit-link-field` elements
6. Use `@tanstack/react-query` for data fetching with QueryClientProvider

### Webpack Configuration

- Entry: `app/js/index.js`
- Output: `app/index.js` + `app/vendor.js` (split chunks)
- Externals: React and ReactDOM (provided by WordPress)
- Aliases:
  - `@app` → `./app/js/`
  - `@common` → `./common/js/`
  - `@neko-ui` → `../neko-ui/`
- Watch mode: Use `pnpm dev`

### Constants and Options

**PHP Constants:**
- `MGCL_VERSION` - Plugin version
- `MGCL_PREFIX` - 'mgcl'
- `MGCL_DOMAIN` - 'gallery-custom-links'
- `MGCL_PATH` - Plugin directory path
- `MGCL_URL` - Plugin URL
- `MGCL_ENTRY` - Main plugin file path

**Database Options:**
- `mgcl_obmode` - Output buffering mode (bool)
- `mgcl_parsing_engine` - HTML parser choice (string)
- `mgcl_log` - Enable debug logging (bool)
- `mgcl_button_enabled` - Show CTA buttons (bool)
- `mgcl_button_label` - Button label text (string)

**Post Meta (per attachment):**
- `_gallery_link_url` - Target URL
- `_gallery_link_target` - `_self` or `_blank`
- `_gallery_link_rel` - Link rel attribute (e.g., 'nofollow', 'noopener noreferrer')
- `_gallery_link_aria` - Aria-label for accessibility

## Common Patterns

### Adding a New Parsing Engine Feature

1. Check `$this->parsingEngine` in `classes/core.php`
2. For HtmlDomParser: Use `$element->{'attribute'}` syntax
3. For DiDom: Use `$element->attr('attribute')` method
4. Both engines support `$element->parent()` and `$html->find(selector)`

### Extending Link Behavior

1. Use `mgcl_linkers` filter to override default linking logic
2. Receives: `$element`, `$parent`, `$mediaId`, `$url`, `$rel`, `$aria`, `$target`
3. Return `true` if handled, `false` to use default linker

### Adding Gallery Support

1. Check `classes/button/` for examples (gutenberg.php, meow_gallery.php, native_gallery.php)
2. Use `mgcl_button_linker` filter to inject custom button HTML
3. Button classes can add CSS selectors to target specific gallery markup

## Distribution

The plugin uses `nekofy.conf` for build configuration:
- Free version: Excludes `premium/` and `common/premium/` directories
- SVN distribution: Excludes source files (JS sources, webpack config, etc.)
- Main repo: https://github.com/meowarts/gallery-custom-links
- SVN repo: https://plugins.svn.wordpress.org/gallery-custom-links

## Pro vs Free

- Pro features are in `premium/` directory
- Check `class_exists( 'MeowPro_MGCL_Core' )` to detect Pro version
- License management handled via `common/premium/` classes

## Notes

- The plugin processes HTML on visitor requests, not at save time
- Image ID resolution uses caching via global `$galleryCustomLinksCache`
- Lightboxes are disabled via `no-lightbox` CSS class and JavaScript event handlers
- The plugin includes vanilla JavaScript to unbind lightbox click handlers in `unlink_lightboxes_script()`
- Custom modifications for aria-label support added by Christoph Letmaier (2020)
