<?php

/**
 * MainWP WP Rocket
 *
 * MainWP WP Rocket Extension handler.
 *
 * @link https://mainwp.com/extension/rocket/
 *
 * @package MainWP\Child
 *
 * Credits
 *
 * Plugin-Name: WP Rocket
 * Plugin URI: https://wp-rocket.me
 * Author:  WP Media
 * Author URI: https://wp-media.me
 * Licence: GPLv2 or later
 */

class MainWP_Rocket { //phpcs:ignore -- NOSONAR - multi methods.

	/**
	 * Public static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	public static $instance = null;

	/**
	 * private variable setting.
	 *
	 * @var array default empty.
	 */
	private $rocket_settings = array();

	/**
	 * Public variable setting.
	 *
	 * @var mixed Default null
	 */
	public $template_path = null;


	/**
	 * Get the only instance of the class.
	 *
	 * @return self The only instance of a class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialize the class and set the path to the template folder.
	 */
	public function __construct() {
		$this->template_path = MAINWP_WP_ROCKET_PATH . 'views/settings';
	}

	/**
	 * Hook action for admin_init
	 *
	 * @return void
	 */
	public function admin_init() {
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_site_override_settings', array( $this, 'ajax_override_settings' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_reload_optimize_info', array( $this, 'ajax_reload_optimize_info' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_purge_cloudflare', array( $this, 'ajax_purge_cloudflare' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_preload_purge_cache_all', array( $this, 'ajax_preload_purge_cache_all' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_clear_priority_elements', array( $this, 'ajax_clear_priority_elements' ) );

		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_generate_critical_css', array( $this, 'ajax_generate_critical_css' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_save_opts_to_child_site', array( $this, 'ajax_save_opts_to_child_site' ) );

		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_optimize_data_on_child_site', array( $this, 'ajax_optimize_database_site' ) );

		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_update_exclusion_list_child_site', array( $this, 'ajax_update_exclusion_list' ) );

		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_site_load_existing_settings', array( $this, 'ajax_load_existing_settings' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_rocket_rightnow_load_sites', array( $this, 'ajax_general_load_sites' ) );
		add_action( 'mainwp_delete_site', array( &$this, 'delete_site_data' ), 10, 1 );
		add_action( 'mainwp_updates_overview_after_update_details', array( &$this, 'mainwp_rocket_hook_buttons' ) );
		add_filter( 'mainwp_managesites_bulk_actions', array( &$this, 'managesites_bulk_actions' ), 10, 1 );
		add_action( 'mainwp_site_synced', array( &$this, 'site_synced' ), 10, 2 );
		add_filter( 'mainwp_sync_others_data', array( $this, 'sync_others_data' ), 10, 2 );
	}

	/**
	 * Site synced config
	 *
	 * @param mixed $p_website
	 * @param array $information
	 * @return void
	 */
	public function site_synced( $p_website, $information = array() ) {
		if ( $p_website && $p_website->plugins != '' ) {
			$this->update_plugin_wprocket_status( $p_website );
		}
		if ( is_array( $information ) && isset( $information['syncWPRocketData'] ) ) {
			$this->update_sync_wprocket_data( $p_website->id, $information['syncWPRocketData'] );
			unset( $information['syncWPRocketData'] );
		}
	}

	/**
	 * Update plugin wp rocket status
	 *
	 * @param mixed $p_website
	 * @return void
	 */
	private function update_plugin_wprocket_status( $p_website ) {
		$plugins   = json_decode( $p_website->plugins, 1 );
		$is_active = $this->check_plugin_wprocket_active( $plugins );

		$update = array(
			'site_id'   => $p_website->id,
			'is_active' => $is_active,
		);
		MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
	}

	/**
	 * Check plugin wp rocket active
	 *
	 * @param mixed $plugins
	 * @return int active or in active
	 */
	private function check_plugin_wprocket_active( $plugins ) {
		if ( is_array( $plugins ) && count( $plugins ) != 0 ) {
			foreach ( $plugins as $plugin ) {
				if ( MAINWP_ROCKET_FILE_PATH == strtolower( $plugin['slug'] ) && $plugin['active'] ) {
					return 1;
				}
			}
		}
		return 0;
	}
	/**
	 * Update sync wp rocket data
	 *
	 * @param mixed $website_id
	 * @param mixed $data
	 */
	private function update_sync_wprocket_data( $website_id, $data ) {
		if ( is_array( $data ) ) {
			$update = array( 'site_id' => $website_id );
			$others = $this->get_wprocket_others( $website_id );

			$fields_to_update = array( 'rocket_boxes', 'lists_delayjs', 'lists_delayjs_full_list' );
			foreach ( $fields_to_update as $field ) {
				if ( isset( $data[ $field ] ) ) {
					$others[ $field ] = $data[ $field ];
				}
			}

			if ( isset( $others['rocket_boxes'] ) && ! isset( $data['rocket_boxes'] ) ) {
				unset( $others['rocket_boxes'] );
			}

			$update['others'] = base64_encode( serialize( $others ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- base64_encode sync update wp-rocket data.
			MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
		}

		// add list One-click exclusions
		$this->wprocket_update_exclusions( $data );
	}
	/**
	 * Get wprocket others.
	 *
	 * @param mixed $website_id
	 * @return mixed
	 */
	private function get_wprocket_others( $website_id ) {
		$wprocket_data = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $website_id, 'others' );
		$others        = unserialize( base64_decode( $wprocket_data->others ) );
		return is_array( $others ) ? $others : array();
	}

	/**
	 * Wp Rocket update exclusions
	 *
	 * @param mixed $data sync website data.
	 */
	private function wprocket_update_exclusions( $data ) {
		if ( ! empty( $data['lists_delayjs_full_list'] ) && ! empty( $data['lists_delayjs_full_list']['scripts']['items'] ) ) {
			update_option( MAINWP_ROCKET_JAVASCRIPT_DELAY_EXCLUSIONS, $data['lists_delayjs_full_list']['scripts']['items'] );
		}
	}

	/**
	 * Sync others data.
	 *
	 * @param mixed $data sync data.
	 * @param mixed $p_website website data.
	 */
	public function sync_others_data( $data, $p_website = null ) { // phpcs:ignore -- NOSONAR - $p_website is currently not used.
		// Note: $p_website is currently not used, but kept for future extensions or consistency with hook definition.

		if ( ! is_array( $data ) ) {
			$data = array();
		}
		$data['syncWPRocketData'] = 'yes';

		return $data;
	}

	/**
	 * get options data
	 *
	 * @return mixed
	 */
	public function get_options() {
		if ( empty( $this->rocket_settings ) ) {
			if ( self::is_manage_sites_page() ) {
				$site_id  = self::get_manage_site_id();
				$wprocket = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $site_id );
				if ( ! empty( $wprocket ) ) {
					$this->rocket_settings = unserialize( base64_decode( $wprocket->settings ) );
				} else {
					$this->rocket_settings = mainwp_get_rocket_default_options();
					$update                = array(
						'site_id'  => $site_id,
						'settings' => base64_encode( serialize( $this->rocket_settings ) ),  // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- base64_encode sync rocket settings.
					);
					MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
				}
			} else {
				$this->rocket_settings = get_option( MAINWP_ROCKET_GENERAL_SETTINGS );
			}

			if ( ! is_array( $this->rocket_settings ) ) {
				$this->rocket_settings = mainwp_get_rocket_default_options();
			}
		}
		return $this->rocket_settings;
	}

	/**
	 * Remove website data
	 *
	 * @param mixed $website
	 * @return void
	 */
	public function delete_site_data( $website ) {
		if ( $website ) {
			MainWP_Rocket_DB::get_instance()->delete_wprocket( 'site_id', $website->id );
		}
	}

	/**
	 * Render get website from request
	 */
	private static function render_get_website_from_request() {
		$website         = null;
		$current_site_id = 0;
		if ( ! empty( $_GET['id'] ) ) {
			$current_site_id = intval( $_GET['id'] );
			global $mainWPRocketExtensionActivator; // phpcs:ignore
			$option      = array(
				'plugin_upgrades' => true,
				'plugins'         => true,
			);
			$db_websites = apply_filters(
				'mainwp_getdbsites',
				$mainWPRocketExtensionActivator->get_child_file(),
				$mainWPRocketExtensionActivator->get_child_key(),
				array( $current_site_id ),
				array(),
				$option
			);
			if ( ! empty( $db_websites ) && is_array( $db_websites ) ) {
				$website = current( $db_websites );
			}
		}
		return $website;
	}

	/**
	 * Render validate website
	 *
	 * @param mixed $website website data.
	 */
	public static function render_validate_website( $website ) {
		if ( empty( $website ) || empty( $website->id ) ) {
			return __( 'Invalid Site ID. Site does not exist.', 'mainwp-rocket-extension' );
		}
		if ( $website->plugins != '' ) {
			$plugins = json_decode( $website->plugins, true );
			if ( ! empty( $plugins ) && is_array( $plugins ) ) {
				foreach ( $plugins as $plugin ) {
					if ( MAINWP_ROCKET_FILE_PATH == strtolower( $plugin['slug'] ) && ! empty( $plugin['active'] ) ) {
						return '';
					}
				}
			}
		}

		return __( 'WP Rocket plugin is not found on the site. Please, try to resync. If the issue persists, ensure that the plugin is installed and activated.', 'mainwp-rocket-extension' );
	}

	/**
	 * Render Error
	 *
	 * @param string $error error msg.
	 */
	public static function render_error( $error ) {
		do_action( 'mainwp_pageheader_sites', 'WPRocket' );
		echo '<div class="ui segment">';
		echo "<div class='ui red message'>$error</div>";
		echo '</div>';
		do_action( 'mainwp_pagefooter_sites', 'WPRocket' );
	}
	/**
	 * Render page html
	 */
	public static function render() {

		$website = self::render_get_website_from_request();
		if ( self::is_manage_sites_page() ) {
			$error = self::render_validate_website( $website );

			if ( ! empty( $error ) ) {
				self::render_error( $error );
				return;
			}
		}

		self::render_tabs( $website );
	}

	/**
	 * Css Display none style.
	 *
	 * @return string
	 */
	public static function display_none_style() {
		return 'style="display:none"';
	}

	/**
	 * Render tabs enqueue script.
	 */
	private static function render_tabs_enqueue_script() {
		?>
		<script type="text/javascript">
			jQuery(document).ready(function() {
				jQuery('#mainwp-rocket-menu a.item').tab();
			});

			{
				function s() {
					var e = this;
					if (this.$links = document.querySelectorAll("#mainwp-rocket-menu a"), this.$menuItem = null, this.pageId = null, window.onhashchange = function() {
							e.detectID()
						}, window.location.hash) this.detectID();
					else {
						var i = localStorage.getItem("mainwp-rocket-hash");
						i ? (window.location.hash = i, this.detectID()) : (localStorage.setItem("mainwp-rocket-hash", "dashboard"), window.location.hash = "#dashboard")
					}

					for (var s = 0; s < this.$links.length; s++) this.$links[s].onclick = function() {
						var t = this.href.split("#")[1];
						if (null != t) {
							window.location.hash = t;
						}
						console.log(t);
					};

					if (this.$menuItem) {
						jQuery(this.$menuItem).trigger("click"); // semantic will show/hide tabs
					}

					// clicking on other links with clear saved status
					var r = document.querySelectorAll("#mainwp-page-navigation-wrapper a, #mainwp-main-menu a, #mainwp-sync-sites");
					for (s = 0; s < r.length; s++) r[s].onclick = function() {
						localStorage.setItem("mainwp-rocket-hash", "");
					}
				}
				document.addEventListener("DOMContentLoaded", function() {
						var t = document.querySelector("#mainwp-rocket-menu");
						t && new s();
					}),
					s.prototype.detectID = function() {
						this.pageId = window.location.hash.split("#")[1], localStorage.setItem("mainwp-rocket-hash", this.pageId), this.$menuItem = document.querySelector('a[data-tab="' + this.pageId + '"]');
					}
			}
		</script>
		<?php
	}

	/**
	 * Render tabs mainwp rocket menu
	 *
	 * @param mixed $is_manage_site status?
	 */
	private static function render_tabs_mainwp_rocket_menu( $is_manage_site ) {
		?>

		<div class="ui labeled icon inverted menu mainwp-sub-submenu" id="mainwp-rocket-menu">
			<?php if ( ! $is_manage_site ) : ?>
				<a href="#dashboard" class="item active" data-tab="dashboard">
					<i class="home icon"></i>
					<?php echo __( 'Dashboard', 'mainwp-rocket-extension' ); ?></a>
			<?php endif; ?>
			<a href="#wp-rocket" class="item" data-tab="wp-rocket">
				<i class="rocket icon"></i>
				<?php echo __( 'Quick Actions', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#file-optimization" class="item" data-tab="file-optimization">
				<i class="layer group icon"></i>
				<?php echo __( 'File Optimization', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#media" class="item" data-tab="media">
				<i class="photo video icon"></i>
				<?php echo __( 'Media', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#preload" class="item" data-tab="preload">
				<i class="sync icon"></i>
				<?php echo __( 'Preload', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#advanced-rules" class="item" data-tab="advanced-rules">
				<i class="list icon"></i>
				<?php echo __( 'Advanced Rules', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#database" class="item" data-tab="database">
				<i class="database icon"></i>
				<?php echo __( 'Database', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#cdn" class="item" data-tab="cdn">
				<i class="globe africa icon"></i>
				<?php echo __( 'CDN', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#heartbeat" class="item" data-tab="heartbeat">
				<i class="heartbeat icon"></i>
				<?php echo __( 'Heartbeat', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#add-on" class="item" data-tab="add-on">
				<i class="puzzle piece icon"></i>
				<?php echo __( 'Add-ons', 'mainwp-rocket-extension' ); ?>
			</a>
			<a href="#tools" class="item" data-tab="tools">
				<i class="tools icon"></i>
				<?php echo __( 'Tools', 'mainwp-rocket-extension' ); ?>
			</a>
		</div>
		<?php
	}

	/**
	 * Render tabs dashboard
	 *
	 * @param mixed $is_manage_site
	 * @param mixed $current_site_id
	 */
	private static function render_tabs_dashboard( $is_manage_site, $current_site_id ) {
		?>
		<div class="ui <?php echo $is_manage_site ? 'active' : ''; ?> tab tab-dashboard" data-tab="wp-rocket">
			<h3 class="ui dividing header"><?php echo __( 'WP Rocket Quick Actions', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for="mainwp_wp_rocket_settings[analytics_enabled]"><?php echo __( 'Rocket analytics', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[analytics_enabled]" <?php checked( mainwp_get_rocket_option( 'analytics_enabled', 0 ), 1 ); ?> id="analytics_enabled">
					<label for=""><em><?php echo __( 'I agree to share anonymous data with the development team to help improve WP Rocket.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Remove all cached files', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">

					<a href="<?php echo $is_manage_site ? '' : wp_nonce_url( admin_url( 'admin.php?page=Extensions-Mainwp-Rocket-Extension&_perform_action=mainwp_rocket_preload_purge_cache_all' ), 'mainwp_rocket_preload_purge_cache_all' ); ?>" class="ui green basic button" onclick="
										<?php
										echo $is_manage_site ? 'mainwp_rocket_individual_purge_all(' . intval( $current_site_id ) . ', this); return false;' : ''; // NOSONAR
										?>
					">
						<?php echo __( 'Clear and Preload Cache', 'mainwp-rocket-extension' ); ?>
					</a>
					<span class="status"></span>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Regenerate critical CSS', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<a href="<?php echo $is_manage_site ? '' : wp_nonce_url( admin_url( 'admin.php?page=Extensions-Mainwp-Rocket-Extension&_perform_action=mainwp_rocket_generate_critical_css' ), 'mainwp_rocket_generate_critical_css' ); ?>" onclick="
										<?php
										echo $is_manage_site ? 'mainwp_rocket_individual_generate_critical_css(' . intval( $current_site_id ) . ', this); return false;' : ''; // NOSONAR
										?>
																																																															" class="ui green basic button"><?php echo __( 'Regenerate Critical CSS', 'mainwp-rocket-extension' ); ?></a>
					<span class="status"></span>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Clear priority elements', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<a href="<?php echo $is_manage_site ? '' : esc_url( wp_nonce_url( admin_url( 'admin.php?page=Extensions-Mainwp-Rocket-Extension&_perform_action=mainwp_rocket_clear_priority_elements' ), 'mainwp_rocket_clear_priority_elements' ) ); ?>" onclick="
										<?php
										echo $is_manage_site ? 'mainwp_rocket_individual_clear_priority_elements(' . intval( $current_site_id ) . ', this); return false;' : ''; // NOSONAR
										?>
																																																																		" class="ui green basic button"><?php esc_html_e( 'Clear Priority Elements', 'mainwp-rocket-extension' ); ?></a>
					<span class="status"></span>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>
		<?php
	}
	/**
	 * Render tabs file optimization
	 */
	private static function render_tabs_file_optimization() {
		$excluded_javascript_files = __( 'Excluded JavaScript Files', 'mainwp-rocket-extension' );
		?>
		<div class="ui tab" id="mainwp-rocket-file-optimization-settings" data-tab="file-optimization">
			<h3 class="ui dividing header"><?php echo __( 'File Optimization Settings', 'mainwp-rocket-extension' ); ?></h3>
			<?php
			$minify_css = mainwp_get_rocket_option( 'minify_css', 0 );
			?>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for="mainwp_wp_rocket_settings[minify_css]"><?php echo __( 'Minify CSS files', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[minify_css]" <?php checked( $minify_css, 1 ); ?> id="minify_css">
					<label for=""><em><?php echo __( 'Minify CSS removes whitespace and comments to reduce the file size.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field" <?php echo $minify_css ? '' : self::display_none_style(); ?>>

				<label class="six wide column middle aligned" for="mainwp_wp_rocket_settings[exclude_css]"><?php echo __( 'Excluded CSS files', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[exclude_css]" placeholder="/wp-content/plugins/some-plugin/(.*).css" id="exclude_css"><?php echo mainwp_rocket_field_value( 'exclude_css', 'textarea' ); ?></textarea><br />
					<p><em><?php echo __( 'Specify URLs of CSS files to be excluded from minification and concatenation.', 'mainwp-rocket-extension' ); ?></em></p>
				</div>
			</div>
			<?php
			$async_css = mainwp_get_rocket_option( 'async_css', 0 );
			?>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for="mainwp_wp_rocket_settings[async_css]"><?php echo __( 'Optimize CSS delivery', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[async_css]" <?php checked( $async_css, 1 ); ?> id="async_css">
					<label for=""><em><?php echo __( 'Optimize CSS delivery eliminates render-blocking CSS on your website for faster perceived load time.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field" <?php echo $async_css ? '' : self::display_none_style(); ?>>
				<div class="six wide column middle aligned"></div>
				<div class="ten wide column ui inverted segment">
					<div class="ui top attached tabular menu optimize-css-tabs">
						<a class="item" data-tab="remove-unsed-css"><?php esc_html_e( 'REMOVE UNUSED CSS', 'mainwp-rocket-extension' ); ?></a>
						<a class="item active" data-tab="load-css-asynchronously"><?php esc_html_e( 'LOAD CSS ASYNCHRONOUSLY', 'mainwp-rocket-extension' ); ?></a>
					</div>
					<div class="ui bottom attached inverted tab segment active" data-tab="remove-unsed-css">
						<div class="column middle aligned">
							<p><em><?php echo esc_html_e( 'Removes unused CSS per page and helps to reduce page size and HTTP requests. Recommended for best performance. Test thoroughly!', 'mainwp-rocket-extension' ); ?></em></p>
							<p><strong><?php echo esc_html_e( 'CSS safelist' ); ?></strong></p>
							<p><em><?php echo esc_html_e( 'Specify CSS filenames, IDs or classes that should not be removed (one per line).', 'mainwp-rocket-extension' ); ?></em></p>
						</div>
						<div class="column">
							<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[remove_unused_css_safelist]" id="remove_unused_css_safelist" placeholder="/wp-content/plugins/some-plugin/(.*).css"><?php echo mainwp_rocket_field_value( 'remove_unused_css_safelist', 'textarea' ); ?></textarea>
						</div>
					</div>
					<div class="ui bottom attached inverted tab segment " data-tab="load-css-asynchronously">
						<p class="column middle aligned" for=""><?php echo esc_html_e( 'Fallback critical CSS', 'mainwp-rocket-extension' ); ?></p>
						<div class="column">
							<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[critical_css]" id="critical_css"><?php echo mainwp_rocket_field_value( 'critical_css', 'textarea' ); ?></textarea><br />
							<p><em><?php echo __( 'Provides a fallback if auto-generated critical path CSS is incomplete.', 'mainwp-rocket-extension' ); ?></em></p>
						</div>
					</div>
				</div>
			</div>

			<?php
			$minify_js = mainwp_get_rocket_option( 'minify_js', 0 );
			?>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Minify JavaScript files', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[minify_js]" <?php checked( $minify_js, 1 ); ?> id="minify_js">
					<label for=""><em><?php echo __( 'Minify JavaScript removes whitespace and comments to reduce the file size.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field mainwp-parent2-field" <?php echo $minify_js ? '' : self::display_none_style(); ?>>
				<?php
				$minify_concatenate_js = mainwp_get_rocket_option( 'minify_concatenate_js', 0 );
				?>
				<label class="six wide column middle aligned" for=""><?php echo __( 'Combine JavaScript files (Enable Minify JavaScript files to select)', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[minify_concatenate_js]" <?php checked( $minify_concatenate_js, 1 ); ?> id="minify_concatenate_js">
					<label for=""><em><?php echo __( "Combine Javascript files combines your site's JS info fewer files, reducing HTTP requests. Not recommended if your site uses HTTP/2.", 'mainwp-rocket-extension' ); ?></em></label>
				</div>
				<label class="six wide column middle aligned mainwp-child2-field " for="" <?php echo $minify_concatenate_js ? '' : self::display_none_style(); ?>><?php echo __( 'Excluded Inline JavaScript', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column mainwp-child2-field" <?php echo $minify_concatenate_js ? '' : self::display_none_style(); ?>>
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[exclude_inline_js]" id="exclude_inline_js"><?php echo mainwp_rocket_field_value( 'exclude_inline_js', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify patterns of inline JavaScript to be excluded from concatenation (one per line).', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
				<label class="six wide column middle aligned" for=""><?php echo $excluded_javascript_files; ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[exclude_js]" placeholder="/wp-content/themes/some-theme/(.*).js" id="exclude_js"><?php echo mainwp_rocket_field_value( 'exclude_js', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify URLs of JavaScript files to be excluded from minification and concatenation.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php
			$defer_all_js = mainwp_get_rocket_option( 'defer_all_js', 0 );
			?>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Load JavaScript deferred', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[defer_all_js]" <?php checked( $defer_all_js, 1 ); ?> id="defer_all_js">
					<label for=""><em><?php echo __( 'Load JavaScript deferred eliminates render-blocking JS on your site and can improve load time.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field" <?php echo $defer_all_js ? '' : self::display_none_style(); ?>>
				<label class="six wide column middle aligned" for=""><?php echo $excluded_javascript_files; ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[exclude_defer_js]" id="exclude_defer_js" placeholder="/wp-content/themes/some-theme/(.*).js"><?php echo mainwp_rocket_field_value( 'exclude_defer_js', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify URLs or keywords of JavaScript files to be excluded from defer (one per line). More info.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<?php
			$delay_js = mainwp_get_rocket_option( 'delay_js', 0 );
			?>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Delay JavaScript execution', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[delay_js]" <?php checked( $delay_js, 1 ); ?> id="delay_js">
					<label for=""><em><?php echo __( 'Improves performance by delaying the loading of JavaScript files until user interaction (e.g. scroll, click).', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field" <?php echo $delay_js ? '' : self::display_none_style();  // phpcs:ignore -- NOSONAR ?>>
				<?php
				$delay_js_exclusions_selected = mainwp_get_rocket_option( 'delay_js_exclusions_selected' ) ?? array();
				$lists_delayjs_full_list      = array();
				$site_id                      = static::get_manage_site_id();
				if ( ! empty( $site_id ) ) {
					$information = array();
					$website_db  = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $site_id );
					if ( ! empty( $website_db ) && ! empty( $website_db->others ) ) {
						$other = unserialize( base64_decode( $website_db->others ) );  // phpcs:ignore -- NOSONAR
						$lists_delayjs_full_list = $other['lists_delayjs_full_list'] ?? array();
					}

                    $trans_name = 'mainwp_rocket_auto_trigger_synced_' . $site_id;
                    $auto_sync = get_transient( $trans_name );
                    // Auto Sync data website.
					if ( empty( $lists_delayjs_full_list ) && empty( $auto_sync ) ) {
                        set_transient( $trans_name , 1, DAY_IN_SECONDS );
						?>
						<script>
							jQuery(document).ready(function($) {
								setTimeout(function() {
									if ($('#mainwp-sync-sites').length > 0) {
										$('#mainwp-sync-sites').trigger('click');
									}
								}, 200);
							});
						</script>
						<?php
					}
				} else {
					// Set default data if it is a general page.
					$lists_delayjs_full_list['scripts']['items'] = get_option( MAINWP_ROCKET_JAVASCRIPT_DELAY_EXCLUSIONS, array() );
				}
				?>
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'One-click exclusions', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
				<?php
				if ( ! empty( $lists_delayjs_full_list ) ) :
					foreach ( $lists_delayjs_full_list as $key_list => $lists ) :

						if ( ! self::is_manage_sites_page() && in_array( $key_list, array( 'plugins', 'themes' ), true ) ) {
							continue;
						}

						if ( ! empty( $lists['items'] ) ) :
							?>
							<div id="delay_js_<?php echo esc_attr( $key_list ); ?>_exclusions_selected" class="one-click-exclusions-field one-click-exclusions-multiple-select one-click-exclusions-field--categorizedmultiselect one-click-exclusions-field--children">
								<?php
								if ( 'scripts' === $key_list ) {
									foreach ( $lists['items'] as $key_item => $val_item ) {
										?>
										<div class="one-click-exclusions-list " id="delay_js_list_<?php echo esc_attr( $key_item ); ?>">
											<?php
												static::lists_delayjs_header( $key_item, $val_item['title'] );
												static::lists_delayjs_body( $key_item, $val_item, $delay_js_exclusions_selected );
											?>
										</div>
										<?php
									}
								} else {
									$title = ( 'plugins' === $key_list ) ? __( 'Plugins', 'mainwp-rocket-extension' ) : __( 'Themes', 'mainwp-rocket-extension' );
									?>
									<div class="one-click-exclusions-list" id="delay_js_list_<?php echo esc_attr( $key_item ); ?>">
										<?php
											static::lists_delayjs_header( $key_list, $title );
											static::lists_delayjs_body( $key_list, $lists, $delay_js_exclusions_selected );
										?>
									</div>
									<?php
								}
								?>
							</div>
							<?php
						endif;
						endforeach;
					endif;
				?>
					<label for=""><em><?php esc_html_e( 'When using the Delay JavaScript Execution, you might experience delay loading elements located in the viewport that need to appear immediately - e.g. slider, header, menu.', 'mainwp-rocket-extension' ); ?></em></label><br /><br />
					<label for=""><em><?php esc_html_e( 'If you need instant visibility, click below on files that should NOT be delayed. This selection will help users interact with the elements straight away.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
				<!-- Excluded JavaScript Files -->
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Excluded JavaScript Files', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[delay_js_exclusions]" id="delay_js_exclusions"><?php echo mainwp_rocket_field_value( 'delay_js_exclusions', 'textarea' );  // phpcs:ignore -- NOSONAR  ?></textarea><br />
					<label for=""><em><?php esc_html_e( 'Specify URLs or keywords that can identify inline or JavaScript files to be excluded from delaying execution (one per line).', 'mainwp-rocket-extension' ); ?></em></label><br /><br />
					<label for=""><em><?php esc_html_e( 'If you have problems after activating this option, copy and paste the default exclusions to quickly resolve issues:', 'mainwp-rocket-extension' ); ?></em></label>
					<pre><code>\/jquery(-migrate)?-?([0-9.]+)?(.min|.slim|.slim.min)?.js(\?(.*))?( |'|"|&gt;)<br>js-(before|after)<br>(?:/wp-content/|/wp-includes/)(.*)</code></pre>
				</div>
				<!-- Safe Mode for Delay JavaScript Execution -->
				<?php $delay_js_safe_mode = mainwp_get_rocket_option( 'delay_js_execution_safe_mode', 0 ); ?>
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Safe Mode for Delay JavaScript Execution', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[delay_js_execution_safe_mode]" <?php checked( $delay_js_safe_mode, 1 ); ?> id="delay_js_execution_safe_mode">
					<label for=""><em><?php esc_html_e( 'The Safe Mode mode prevents all internal scripts from being delayed.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>
		<?php
	}

	/**
	 * Render hearder list delayjs
	 *
	 * @param string $key_list key list.
	 * @param string $title title.
	 */
	private static function lists_delayjs_header( $key_list, $title ) {
		?>
		<div class="one-click-exclusions-list-header">
			<div class="ui checkbox one-click-exclusions-checkbox">
				<input id="checkboxes-<?php echo esc_attr( $key_list ); ?>" class="one-click-exclusions-main-checkbox" type="checkbox" value="<?php echo esc_attr( $key_list ); ?>">
				<label for="">
					<span class="one-click-exclusions-multiple-select-title">
						<?php echo esc_html( $title ); ?>
					</span>
				</label>
			</div>
			<div class="one-click-exclusions-list-header-arrow">
				<!-- SVG arrow icon -->
				<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" height="100%" width="100%">
					<path d="M233.4 105.4c12.5-12.5 32.8-12.5 45.3 0l192 192c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0L256 173.3 86.6 342.6c-12.5 12.5-32.8 12.5-45.3 0s-12.5-32.8 0-45.3l192-192z"></path>
				</svg>
			</div>
		</div>
		<?php
	}

	/**
	 * Render body list delayjs
	 *
	 * @param string $key_list key list.
	 * @param array  $lists lists.
	 * @param array  $delay_js_exclusions_selected delay js exclusions selected.
	 */
	private static function lists_delayjs_body( $key_list, $lists, $delay_js_exclusions_selected ) {
		?>
		<div class="one-click-exclusions-list-body list-<?php echo esc_attr( $key_list ); ?>-body" data-name="<?php echo esc_attr( $key_list ); ?>">
			<ul>
				<?php
				// Check if 'items' key exists and is an array
				if (isset($lists['items']) && is_array($lists['items'])) :
					foreach ( $lists['items'] as $item ) :
				?>
					<li>
						<div class="ui checkbox one-click-exclusions-checkbox">
							<?php
							$item_id = isset($item['id']) ? $item['id'] : '';
							$exclusions_checked = isset( $delay_js_exclusions_selected ) && is_array( $delay_js_exclusions_selected ) && in_array( $item_id, $delay_js_exclusions_selected, true ) ? 1 : 0;
							?>
							<input type="checkbox" name="mainwp_wp_rocket_settings[delay_js_exclusions_selected][]" value="<?php echo esc_attr( $item_id ); ?>" <?php checked( $exclusions_checked ); ?> />
							<label for=""><?php echo esc_html( isset($item['title']) ? $item['title'] : '' ); ?></label>
						</div>
					</li>
				<?php
					endforeach;
				endif;
				?>
			</ul>
		</div>
		<?php
	}

	/**
	 * Render tabs media
	 */
	private static function render_tabs_media() {
		$lazyload_iframes = mainwp_get_rocket_option( 'lazyload_iframes', 0 );
		?>
		<div class="ui tab" id="mainwp-rocket-media-settings" data-tab="media">
			<h3 class="ui dividing header"><?php echo __( 'Media Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column" for=""><?php echo __( 'LazyLoad', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[lazyload]" <?php checked( mainwp_get_rocket_option( 'lazyload', 0 ), 1 ); ?> id="lazyload">
					<label for=""><em><?php echo __( 'Enable for images', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<div class="six wide column" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[lazyload_css_bg_img]" <?php checked( mainwp_get_rocket_option( 'lazyload_css_bg_img', 0 ), 1 ); ?> id="lazyload">
					<label for=""><em><?php esc_html_e( 'Enable for CSS background images', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>


			<div class="ui grid mainwp-parent-field field">
				<div class="six wide column" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[lazyload_iframes]" <?php checked( $lazyload_iframes, 1 ); ?> id="lazyload_iframes">
					<label for=""><em><?php esc_html_e( 'Enable for iframes and videos', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-child-field field" <?php echo $lazyload_iframes ? '' : self::display_none_style(); ?>>
				<div class="six wide column" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[lazyload_youtube]" <?php checked( mainwp_get_rocket_option( 'lazyload_youtube', 0 ), 1 ); ?> id="lazyload_youtube">
					<label for=""><em><?php esc_html_e( 'Replace YouTube iframe with preview image', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-child-field field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Excluded images or iframes', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[exclude_lazyload]" id="exclude_lazyload" placeholder="example-image.jpg&#10;slider-image"><?php echo mainwp_rocket_field_value( 'exclude_lazyload', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php esc_html_e( 'Specify keywords (e.g. image filename, CSS class, domain) from the image or iframe code to be excluded (one per line).', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column" for=""><?php esc_html_e( 'Image dimensions', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[image_dimensions]" <?php checked( mainwp_get_rocket_option( 'image_dimensions', 0 ), 1 ); ?> id="image_dimensions">
					<label for=""><em><?php esc_html_e( 'Add missing width and height attributes to images. Helps prevent layout shifts and improve the reading experience for your visitors.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column" for=""><?php esc_html_e( 'Self-host Google Fonts', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[host_fonts_locally]" <?php checked( mainwp_get_rocket_option( 'host_fonts_locally', 0 ), 1 ); ?> id="host_fonts_locally">
					<label for=""><em><?php esc_html_e( 'Download and serve fonts directly from your server. Reduces connections to external servers and minimizes font shifts.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<?php self::render_save_button(); ?>
		</div>

		<?php
	}
	/**
	 * Render tabs preload
	 *
	 * @return void
	 */
	private static function render_tabs_preload() {
		$manual_preload = mainwp_get_rocket_option( 'manual_preload', 0 );
		$preload_links  = mainwp_get_rocket_option( 'preload_links', 0 );

		?>
		<div class="ui tab" id="mainwp-rocket-preload-settings" data-tab="preload">
			<h3 class="ui dividing header"><?php esc_html_e( 'Preload Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Activate preloading', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[manual_preload]" <?php checked( $manual_preload, 1 ); ?> id="manual_preload">
				</div>
			</div>
			<div class="ui grid mainwp-child-field field mainwp-child-field" <?php echo $manual_preload ? '' : self::display_none_style(); ?>>
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Exclude URLs', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[preload_excluded_uri]" placeholder="/author/(.*)" id="preload_excluded_uri"><?php echo mainwp_rocket_field_value('preload_excluded_uri', 'textarea');  // phpcs:ignore -- NOSONAR
					?>
					</textarea><br />
					<label for=""><em><?php esc_html_e( 'Specify URLs to be excluded from the preload feature (one per line).', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Enable link preloading', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[preload_links]" <?php checked( $preload_links, 1 ); ?> id="preload_links">
					<label for=""><em><?php echo __( 'Link preloading improves the perceived load time by downloading a page when a user hovers over the link.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Prefetch DNS requests', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[dns_prefetch]" placeholder="//example.com" id="dns_prefetch"><?php echo mainwp_rocket_field_value('dns_prefetch', 'textarea');  // phpcs:ignore -- NOSONAR
					?>
					</textarea><br />
					<label for=""><em><?php echo __( 'URLs to prefetch Specify external hosts to be prefetched (no http:, one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Fonts to preload', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[preload_fonts]" placeholder="/wp-content/themes/your-theme/assets/fonts/font-file.woff" id="preload_fonts"><?php echo mainwp_rocket_field_value('preload_fonts', 'textarea');  // phpcs:ignore -- NOSONAR
					?>
					</textarea><br />
					<label for=""><em><?php echo __( 'Specify urls of the font files to be preloaded (one per line). Fonts must be hosted on your own domain, or the domain you have specified on the CDN tab.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>
		<?php
	}
	/**
	 * Render tabs advanced rules
	 */
	private static function render_tabs_advanced_rules() {
		$purge_cron_unit     = mainwp_get_rocket_option( 'purge_cron_unit', 'HOUR_IN_SECONDS' );
		$purge_cron_interval = mainwp_get_rocket_option( 'purge_cron_interval', 10 );
		?>

		<div class="ui tab" id="mainwp-rocket-advanced-rules-settings" data-tab="advanced-rules">
			<h3 class="ui dividing header"><?php echo __( 'Advanced Rules Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Cache lifespan', 'mainwp-rocket-extension' ); ?></label>
				<div class="two wide column">
					<input type="number" id="purge_cron_interval" name="mainwp_wp_rocket_settings[purge_cron_interval]" min="0" value="<?php echo intval( $purge_cron_interval ); ?>">
				</div>
				<div class="eight wide column">
					<select class="ui dropdown" id="purge_cron_unit" name="mainwp_wp_rocket_settings[purge_cron_unit]">
						<option value="HOUR_IN_SECONDS" <?php selected( $purge_cron_unit, 'HOUR_IN_SECONDS' ); ?>><?php echo __( 'hour(s)', 'mainwp-rocket-extension' ); ?></option>
						<option value="DAY_IN_SECONDS" <?php selected( $purge_cron_unit, 'DAY_IN_SECONDS' ); ?>><?php echo __( 'day(s)', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<br />
					<label for=""><em><?php echo __( 'Specify time after which the global cache is cleared (0 = unlimited)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Never cache URL(s)', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cache_reject_uri]" placeholder="/members/(.*)" id="cache_reject_uri"><?php echo mainwp_rocket_field_value( 'cache_reject_uri', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify URLs of pages or posts that should never get cached (one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Never cache cookies', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cache_reject_cookies]" id="cache_reject_cookies"><?php echo mainwp_rocket_field_value( 'cache_reject_cookies', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify the IDs of cookies that, when set in the visitor’s browser, should prevent a page from getting cached (one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Never cache user agent(s)', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cache_reject_ua]" placeholder="(.*)Mobile(.*)Safari(.*)" id="cache_reject_ua"><?php echo mainwp_rocket_field_value( 'cache_reject_ua', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify user agent strings that should never see cached pages (one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Always purge URL(s)', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cache_purge_pages]" id="cache_purge_pages"><?php echo mainwp_rocket_field_value( 'cache_purge_pages', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify URLs you always want purged from cache whenever you update any post or page (one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Cache query string(s)', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cache_query_strings]" id="cache_query_strings"><?php echo mainwp_rocket_field_value( 'cache_query_strings', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify query strings for caching (one per line)', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>

		<?php
	}

	/**
	 * Render tabs database
	 */
	public static function render_tabs_database() {
		$schedule_automatic_cleanup  = mainwp_get_rocket_option( 'schedule_automatic_cleanup', 0 );
		$automatic_cleanup_frequency = mainwp_get_rocket_option( 'automatic_cleanup_frequency', '' )
		?>
		<div class="ui tab" id="mainwp-rocket-database-settings" data-tab="database">
			<h3 class="ui dividing header"><?php echo __( 'Database Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Posts cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_revisions]" <?php checked( mainwp_get_rocket_option( 'database_revisions', 0 ), 1 ); ?> id="database_revisions">
					<label for=""><em><?php echo __( 'Revisions', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_revisions" style="display: none">
						<span class="count-info"></span> <?php echo __( 'revisions in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>

			</div>
			<div class="ui grid field">
				<div class="six wide column middle aligned" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_auto_drafts]" <?php checked( mainwp_get_rocket_option( 'database_auto_drafts', 0 ), 1 ); ?> id="database_auto_drafts">
					<label for=""><em><?php echo __( 'Auto drafts', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_auto_draft" style="display: none">
						<span class="count-info"></span> <?php echo __( 'drafts in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>
			<div class="ui grid field">
				<div class="six wide column middle aligned" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_trashed_posts]" <?php checked( mainwp_get_rocket_option( 'database_trashed_posts', 0 ), 1 ); ?> id="database_trashed_posts">
					<label for=""><em><?php echo __( 'Trashed posts', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_trashed_posts" style="display: none">
						<span class="count-info"></span> <?php echo __( 'trashed posts in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Comments cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_spam_comments]" <?php checked( mainwp_get_rocket_option( 'database_spam_comments', 0 ), 1 ); ?> id="database_spam_comments">
					<label for=""><em><?php echo __( 'Spam comments', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_spam_comments" style="display: none">
						<span class="count-info"></span> <?php echo __( 'spam comments in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>
			<div class="ui grid field">
				<div class="six wide column middle aligned" for=""></div>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_trashed_comments]" <?php checked( mainwp_get_rocket_option( 'database_trashed_comments', 0 ), 1 ); ?> id="database_trashed_comments">
					<label for=""><em><?php echo __( 'Trashed comments', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_trashed_comments" style="display: none">
						<span class="count-info"></span> <?php echo __( 'trashed comments in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Transients cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_all_transients]" <?php checked( mainwp_get_rocket_option( 'database_all_transients', 0 ), 1 ); ?> id="database_all_transients">
					<label for=""><em><?php echo __( 'All transients', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_all_transients" style="display: none">
						<span class="count-info"></span> <?php echo __( 'transients in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Database cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[database_optimize_tables]" <?php checked( mainwp_get_rocket_option( 'database_optimize_tables', 0 ), 1 ); ?> id="database_optimize_tables">
					<label for=""><em><?php echo __( 'Optimize tables', 'mainwp-rocket-extension' ); ?></em></label>
					<span id="opt-info-total_optimize_tables" style="display: none">
						<span class="count-info"></span> <?php echo __( 'tables to optimize in your database.', 'mainwp-rocket-extension' ); ?>
					</span>
				</div>
			</div>

			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Automatic cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[schedule_automatic_cleanup]" <?php checked( $schedule_automatic_cleanup, 1 ); ?> id="schedule_automatic_cleanup">
					<label for=""><em><?php echo __( 'Schedule automatic cleanup', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid mainwp-child-field field" <?php echo $schedule_automatic_cleanup ? '' : self::display_none_style(); ?>>
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Schedule automatic cleanup', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<select id="automatic_cleanup_frequency" name="mainwp_wp_rocket_settings[automatic_cleanup_frequency]" class="ui dropdown">
						<option value="daily" <?php selected( $automatic_cleanup_frequency, 'daily' ); ?>><?php esc_html_e( 'Daily', 'mainwp-rocket-extension' ); ?></option>
						<option value="weekly" <?php selected( $automatic_cleanup_frequency, 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'mainwp-rocket-extension' ); ?></option>
						<option value="monthly" <?php selected( $automatic_cleanup_frequency, 'monthly' ); ?>><?php esc_html_e( 'Monthly', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<label for=""><em><?php esc_html_e( 'frequency', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui clearing divider"></div>
			<input type="submit" name="mainwp_wp_rocket_settings[submit_optimize]" id="rocket_submit_optimize" class="ui big green button" value="<?php esc_attr_e( 'Save Changes and Optimize', 'mainwp-rocket-extension' ); ?>">
		</div>

		<?php
	}

	/**
	 * Render tabs cdn.
	 */
	private static function render_tabs_cdn() {
		$cnames      = mainwp_get_rocket_option( 'cdn_cnames' );
		$cnames_zone = mainwp_get_rocket_option( 'cdn_zone' );
		?>
		<div class="ui tab" id="mainwp-rocket-cdn-settings" data-tab="cdn">
			<h3 class="ui dividing header"><?php echo __( 'CDN Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'CDN', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cdn]" <?php checked( mainwp_get_rocket_option( 'cdn', 0 ), 1 ); ?> id="cdn-chk">
					<label for=""><em><?php echo __( 'Enable content delivery network', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'CDN CNAME', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<?php
					if ( $cnames ) {
						foreach ( $cnames as $k => $_url ) {
							?>
							<div class="cdn-cnames-field">
								<input type="text" placeholder="http://" id="cdn_cnames" name="mainwp_wp_rocket_settings[cdn_cnames][<?php echo $k; ?>]" value="<?php echo esc_attr( $_url ); ?>" /> <?php echo __( ' reserved for ', 'mainwp-rocket-extension' ); ?>
								<select class="ui dropdown" name="mainwp_wp_rocket_settings[cdn_zone][]" id="cdn_zone">
									<option value="all" <?php selected( $cnames_zone[ $k ], 'all' ); ?>><?php echo __( 'All files', 'mainwp-rocket-extension' ); ?></option>
									<option value="images" <?php selected( $cnames_zone[ $k ], 'images' ); ?>><?php echo __( 'Images', 'mainwp-rocket-extension' ); ?></option>
									<option value="css_and_js" <?php selected( $cnames_zone[ $k ], 'css_and_js' ); ?>><?php echo __( 'CSS & JavaScript', 'mainwp-rocket-extension' ); ?></option>
									<option value="js" <?php selected( $cnames_zone[ $k ], 'js' ); ?>><?php echo __( 'JavaScript', 'mainwp-rocket-extension' ); ?></option>
									<option value="css" <?php selected( $cnames_zone[ $k ], 'css' ); ?>><?php echo __( 'CSS', 'mainwp-rocket-extension' ); ?></option>
								</select>
								<a href="javascript:void(0)" id="mainwp-rocket-cname-remove" class="ui button basic red">Remove</a>
							</div>
							<?php
						}
					}
					ob_start();
					?>
					<input type="text" placeholder="cdn.example.com" id="cdn_cnames" name="mainwp_wp_rocket_settings[cdn_cnames][]" value="" /> <?php echo __( ' reserved for ', 'mainwp-rocket-extension' ); ?>
					<select class="ui dropdown" name="mainwp_wp_rocket_settings[cdn_zone][]" id="cdn_zone">
						<option value="all"><?php echo __( 'All files', 'mainwp-rocket-extension' ); ?></option>
						<option value="images"><?php echo __( 'Images', 'mainwp-rocket-extension' ); ?></option>
						<option value="css_and_js"><?php echo __( 'CSS & JavaScript', 'mainwp-rocket-extension' ); ?></option>
						<option value="js"><?php echo __( 'JavaScript', 'mainwp-rocket-extension' ); ?></option>
						<option value="css"><?php echo __( 'CSS', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<?php
					$field_creator = ob_get_clean();
					if ( empty( $cnames ) ) {
						?>
						<div class="cdn-cnames-field">
							<?php
							echo $field_creator;
							?>
						</div>
						<?php
					}
					?>
					<a href="#" id="mainwp-rocket-cname-add" class="ui button basic green"><?php _e( 'Add CNAME', 'mainwp' ); ?></a>
					<div id="cdn-cnames-field-creator" style="display: none">
						<div class="cdn-cnames-field">
							<?php echo $field_creator; ?>
							<a href="javascript:void(0)" id="mainwp-rocket-cname-remove" class="ui button basic red">Remove</a>
						</div>
					</div>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Exclude files from CDN', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column">
					<textarea rows="8" cols="48" name="mainwp_wp_rocket_settings[cdn_reject_files]" placeholder="/wp-content/plugins/some-plugins/(.*).css" id="cdn_reject_files"><?php echo mainwp_rocket_field_value( 'cdn_reject_files', 'textarea' ); ?></textarea><br />
					<label for=""><em><?php echo __( 'Specify URL(s) of files that should not get served via CDN. The domain part of the URL will be stripped automatically. Use (.*) wildcards to exclude all files of a given file type located at a specific path. ', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>
		<?php
	}

	/**
	 * Render tabs heartbe
	 */
	private static function render_tabs_heartbeat() {
		$heartbeat_admin_behavior  = mainwp_get_rocket_option( 'heartbeat_admin_behavior', 'reduce_periodicity' );
		$heartbeat_editor_behavior = mainwp_get_rocket_option( 'heartbeat_editor_behavior', 'reduce_periodicity' );
		$heartbeat_site_behavior   = mainwp_get_rocket_option( 'heartbeat_site_behavior', 'reduce_periodicity' );
		$empty                     = ' ';

		$str_not_limit       = __( 'Do not limit', 'mainwp-rocket-extension' );
		$str_reduce_activity = __( 'Reduce activity', 'mainwp-rocket-extension' )
		?>
		<div class="ui tab" id="mainwp-rocket-heartbeat-settngs" data-tab="heartbeat">
			<h3 class="ui dividing header"><?php echo __( 'Heartbeat Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Control heartbeat', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[control_heartbeat]" <?php checked( mainwp_get_rocket_option( 'control_heartbeat', 0 ), 1 ); ?> id="control_heartbeat">
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Reduce or disable heartbeat activity', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<select id="automatic_cleanup_frequency" name="mainwp_wp_rocket_settings[heartbeat_admin_behavior]" class="ui dropdown">
						<option value="<?php echo $empty; ?>" <?php selected( $heartbeat_admin_behavior, '' ); ?>>
							<?php echo $str_not_limit; ?></option>
						<option value="reduce_periodicity" <?php selected( $heartbeat_admin_behavior, 'reduce_periodicity' ); ?>>
							<?php echo $str_reduce_activity; ?>
						</option>
						<option value="disable" <?php selected( $heartbeat_admin_behavior, 'disable' ); ?>><?php echo __( 'Disable', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<label for=""><em><?php echo __( 'Behavior in backend', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<div class="six wide column middle aligned"> </div>
				<div class="ten wide column ui">
					<select id="automatic_cleanup_frequency" name="mainwp_wp_rocket_settings[heartbeat_editor_behavior]" class="ui dropdown">
						<option value="<?php echo $empty; ?>" <?php selected( $heartbeat_editor_behavior, '' ); ?>>
							<?php echo $str_not_limit; ?></option>
						<option value="reduce_periodicity" <?php selected( $heartbeat_editor_behavior, 'reduce_periodicity' ); ?>><?php echo $str_reduce_activity; ?></option>
						<option value="disable" <?php selected( $heartbeat_editor_behavior, 'disable' ); ?>><?php echo __( 'Disable', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<label for=""><em><?php echo __( 'Behavior in post editor', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<div class="ui grid field">
				<div class="six wide column middle aligned"></div>
				<div class="ten wide column ui">
					<select id="automatic_cleanup_frequency" name="mainwp_wp_rocket_settings[heartbeat_site_behavior]" class="ui dropdown">
						<option value="<?php echo $empty; ?>" <?php selected( $heartbeat_site_behavior, '' ); ?>><?php echo $str_not_limit; ?></option>
						<option value="reduce_periodicity" <?php selected( $heartbeat_site_behavior, 'reduce_periodicity' ); ?>><?php echo $str_reduce_activity; ?></option>
						<option value="disable" <?php selected( $heartbeat_site_behavior, 'disable' ); ?>><?php echo __( 'Disable', 'mainwp-rocket-extension' ); ?></option>
					</select>
					<label for=""><em><?php echo __( 'Behavior in frontend', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>
			<?php self::render_save_button(); ?>
		</div>
		<?php
	}

	/**
	 * Render tabs add on
	 */
	private static function render_tabs_add_on() {
		$do_cloudflare         = mainwp_get_rocket_option( 'do_cloudflare', 0 );
		$sucury_waf_cache_sync = mainwp_get_rocket_option( 'sucury_waf_cache_sync', 0 );
		?>
		<div class="ui tab" id="mainwp-rocket-add-on-settngs" data-tab="add-on">
			<h3 class="ui dividing header"><?php esc_html_e( 'Add-ons', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'User cache', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cache_logged_user]" <?php checked( mainwp_get_rocket_option( 'cache_logged_user', 0 ), 1 ); ?> id="cache_logged_user">
					<label for=""><em><?php esc_html_e( 'If you need to create a dedicated set of cache files for each logged-in WordPress user, you must activate this add-on. User cache is great when you have user-specific or restricted content on your website.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'Varnish', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[varnish_auto_purge]" <?php checked( mainwp_get_rocket_option( 'varnish_auto_purge', 0 ), 1 ); ?> id="varnish_auto_purge">
					<label for=""><em><?php esc_html_e( 'Varnish cache will be purged each time WP Rocket clears its cache to ensure content is always up-to-date.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php esc_html_e( 'WebP compatibility', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cache_webp]" <?php checked( mainwp_get_rocket_option( 'cache_webp', 0 ), 1 ); ?> id="cache_webp">
					<label for=""><em><?php esc_html_e( 'Improve browser compatibility for WebP images. We have not detected any compatible WebP plugin! . If you don’t already have WebP images on your site consider using Imagify or another supported plugin. If you are not using WebP do not enable this option.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<h3 class="ui dividing header"><?php echo __( 'Rocket Add-ons', 'mainwp-rocket-extension' ); ?></h3>


			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Cloudflare', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[do_cloudflare]" <?php checked( $do_cloudflare, 1 ); ?> id="do_cloudflare">
					<label for=""><em><?php echo __( 'Provide your account email, global API key, and domain to use options such as clearing the Cloudflare cache and enabling optimal settings with WP Rocket.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-child-field field" <?php echo $do_cloudflare ? '' : self::display_none_style(); ?>>
				<label class="six wide column middle aligned" for=""><?php echo __( 'Global API key:', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<input type="text" placeholder="" id="cloudflare_api_key" name="mainwp_wp_rocket_settings[cloudflare_api_key]" value="<?php echo mainwp_get_rocket_option( 'cloudflare_api_key', '' ); ?>" />
					<a href="//docs.wp-rocket.me/article/18-using-wp-rocket-with-cloudflare/?utm_source=wp_plugin&utm_medium=wp_rocket#add-on" target="_blank" rel="noopener noreferrer"><?php echo __( 'Find your API key.', 'mainwp-rocket-extension' ); ?></a>
				</div>
				<label class="six wide column middle aligned" for=""><?php echo __( 'Account email', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<input type="text" placeholder="" id="cloudflare_email" name="mainwp_wp_rocket_settings[cloudflare_email]" value="<?php echo mainwp_get_rocket_option( 'cloudflare_email', '' ); ?>" />
				</div>
				<label class="six wide column middle aligned" for=""><?php echo __( 'Zone ID', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<input type="text" placeholder="" id="cloudflare_zone_id" name="mainwp_wp_rocket_settings[cloudflare_zone_id]" value="<?php echo mainwp_get_rocket_option( 'cloudflare_zone_id', '' ); ?>" />
				</div>

				<label class="six wide column middle aligned" for=""><?php echo __( 'Development mode', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cloudflare_devmode]" <?php checked( mainwp_get_rocket_option( 'cloudflare_devmode', 0 ), 1 ); ?> id="cloudflare_devmode">
					<label for=""><em><?php echo __( 'Temporarily activate development mode on your website. This setting will automatically turn off after 3 hours.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>

				<label class="six wide column middle aligned" for=""><?php echo __( 'Optimal settings', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cloudflare_auto_settings]" <?php checked( mainwp_get_rocket_option( 'cloudflare_auto_settings', 0 ), 1 ); ?> id="cloudflare_auto_settings">
					<label for=""><em><?php echo __( 'Automatically enhances your Cloudflare configuration for speed, performance grade and compatibility.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>

				<label class="six wide column middle aligned" for=""><?php echo __( 'Relative protocol', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[cloudflare_protocol_rewrite]" <?php checked( mainwp_get_rocket_option( 'cloudflare_protocol_rewrite', 0 ), 1 ); ?> id="cloudflare_protocol_rewrite">
					<label for=""><em><?php echo __( 'Should only be used with Cloudflare\'s flexible SSL feature. URLs of static files (CSS, JS, images) will be rewritten to use // instead of http:// or https://. ', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-parent-field field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Sucuri', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_wp_rocket_settings[sucury_waf_cache_sync]" <?php checked( $sucury_waf_cache_sync, 1 ); ?> id="sucury_waf_cache_sync">
					<label for=""><em><?php echo __( 'Provide your API key to clear the Sucuri cache when WP Rocket’s cache is cleared.', 'mainwp-rocket-extension' ); ?></em></label>
				</div>
			</div>

			<div class="ui grid mainwp-child-field field" <?php echo $sucury_waf_cache_sync ? '' : self::display_none_style(); ?>>
				<label class="six wide column middle aligned" for=""><?php echo __( 'Sucuri credentials:', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<input type="text" placeholder="" id="sucury_waf_api_key" name="mainwp_wp_rocket_settings[sucury_waf_api_key]" value="<?php echo mainwp_get_rocket_option( 'sucury_waf_api_key', '' ); ?>" />
					<a href="//kb.sucuri.net/firewall/Performance/clearing-cache" target="_blank" rel="noopener noreferrer"><?php echo __( 'Find your API key.', 'mainwp-rocket-extension' ); ?></a>
					<p><em><?php echo __( 'Firewall API key (for plugin), must be in format <code>{32 characters}/{32 characters}</code>:', 'mainwp-rocket-extension' ); ?></em></p>
				</div>
			</div>

			<?php self::render_save_button(); ?>
		</div>
		<?php
	}
	/**
	 * Render tabs tools
	 *
	 * @param  mixed $current_site_id
	 */
	private static function render_tabs_tools( $current_site_id ) {
		$_extra = '';
		if ( $current_site_id ) {
			$_extra = '&id=' . $current_site_id;
		}

		$download_url = wp_nonce_url( admin_url( 'admin-post.php?action=mainwp_rocket_export' . $_extra ), 'mainwp_rocket_export' );
		$bytes        = apply_filters( 'import_upload_size_limit', wp_max_upload_size() ); // Filter from WP Core
		$size         = size_format( $bytes );
		$desc         = __( 'Choose a file from your computer', 'mainwp-rocket-extension' ) . ' (' . sprintf( __( 'maximum size: %s', 'mainwp-rocket-extension' ), $size ) . ')';

		?>
		<div class="ui tab" id="mainwp-rocket-tools-settngs" data-tab="tools">
			<h3 class="ui dividing header"><?php echo __( 'Tools', 'mainwp-rocket-extension' ); ?></h3>

			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Export settings', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<a href="<?php echo $download_url; ?>" id="export" class="ui button"><?php echo __( 'Download settings' ); ?></a>
				</div>
			</div>
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Import settings', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<div class="ui file input">
						<input type="file" accept=".txt,.json" id="upload" name="import" size="25" />
					</div>
					<input type="hidden" name="max_file_size" value="<?php echo $bytes; ?>" />
					<label for=""><em><?php echo $desc; ?></em></label>
					<div class="ui hidden divider"></div>
					<input type="submit" name="import" id="import" class="ui button" value="<?php echo __( 'Upload file and import settings' ); ?>">
				</div>
			</div>
			<!-- Update Inclusion and Exclusion Lists -->
			<div class="ui grid field">
				<label class="six wide column middle aligned" for=""><?php echo __( 'Update inclusion and exclusion lists', 'mainwp-rocket-extension' ); ?></label>
				<div class="ten wide column ui">
					<a href="<?php echo wp_nonce_url( admin_url( 'admin-post.php?action=mainwp_rocket_update_exclusion_list' ), 'mainwp_rocket_nonce_update_exclusion_list' ); ?>" id="export" class="ui button"><?php echo __( 'Update lists' ); ?></a>
					<p><em><?php echo __( 'Compatibility lists are downloaded automatically every week. Click the button if you want to update them manually.', 'mainwp-rocket-extension' ); ?></em></p>
				</div>
			</div>
			<!-- End update Inclusion and Exclusion Lists -->
			<div class="ui clearing divider"></div>
		</div>
		<?php
	}

	/**
	 * Render tabs get message from query
	 *
	 * @return string message
	 */
	private static function render_tabs_get_message_from_query() {
		$str_string = __( 'Settings saved successfully!', 'mainwp-rocket-extension' );
		$message    = '';
		if ( ! empty( $_GET['message'] ) ) {
			switch ( $_GET['message'] ) {
				case 1:
					$message = $str_string;
					break;
				case 2:
					$import_to_sites_lnk = add_query_arg(
						array(
							'_perform_action' => 'mainwp_rocket_save_opts_child_sites',
							'_wpnonce'        => wp_create_nonce( 'mainwp_rocket_save_opts_child_sites' ),
						),
						remove_query_arg( 's', wp_get_referer() )
					);
					$message             = sprintf(
						__( 'Settings imported and saved successfully! %1$s%2$sSave to Child sites%3$s', 'mainwp-rocket-extension' ),
						'<br/>',
						'<a href="' . $import_to_sites_lnk . '">',
						'</a>'
					);
					break;
				default:
					$message = $str_string;
					break;
			}
		}
		return $message;
	}

	/**
	 * Render tabs get db websites wprocket
	 *
	 * @param mixed $rocket_extension_activator
	 * @return mixed data
	 */
	private static function render_tabs_get_db_websites_wprocket( $rocket_extension_activator ) {
		if ( self::is_manage_sites_page() ) {
			return array();
		}

		$websites = apply_filters(
			'mainwp_getsites',
			$rocket_extension_activator->get_child_file(),
			$rocket_extension_activator->get_child_key(),
			null,
			false,
			array( 'plugins_slug' => MAINWP_ROCKET_FILE_PATH )
		);
		if ( is_array( $websites ) ) {
			foreach ( $websites as $site ) {
				$sites_ids[] = $site['id'];
			}
		}

		// Check if we have any sites with the plugin installed.
		if ( empty( $sites_ids ) ) {
			return array();
		}

		$option      = array(
			'plugin_upgrades' => true,
			'version'         => true,
			'plugins'         => true,
		);
		$db_websites = apply_filters(
			'mainwp_getdbsites',
			$rocket_extension_activator->get_child_file(),
			$rocket_extension_activator->get_child_key(),
			$sites_ids,
			array(),
			$option
		);

		$selected_group = ! empty( $_POST['mainwp_rocket_plugin_groups_select'] ) ? intval( $_POST['mainwp_rocket_plugin_groups_select'] ) : 0;

		$plugin_data_sites = ! empty( $sites_ids ) ? MainWP_Rocket_DB::get_instance()->get_wprockets_data( $sites_ids ) : array();

		return MainWP_Rocket_Plugin::get_instance()->get_websites_with_the_plugin( $db_websites, $selected_group, $plugin_data_sites );
	}

	/**
	 * Render tabs get perform action from query
	 */
	private static function render_tabs_get_perform_action_from_query() {
		if ( ! empty( $_GET['_perform_action'] ) ) {
			$perform_action = $_GET['_perform_action'];
			if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], $perform_action ) ) {
				esc_html_e( 'WP Nonce not verfied.', 'mainwp-rocket-extension' );
				return '';
			}
			return str_replace( 'mainwp_rocket_', '', $perform_action );
		}
		return '';
	}
	/**
	 * Render tabs modal
	 *
	 * @param string $action
	 */
	private static function render_tabs_modal( $action ) {

		?>

		<div class="ui modal" id="mainwp-rocket-sync-data-modal">
			<i class="close reload icon"></i>
			<div class="header"><?php echo __( 'MainWP Rocket', 'mainwp-rocket-extension' ); ?></div>
			<div class="scrolling content"><?php self::ajax_general_load_sites( $action ); ?></div>
		</div>
		<script type="text/javascript">
			jQuery('#mainwp-rocket-sync-data-modal').modal('show');
		</script>

		<?php
	}

	/**
	 * Render tabs hidden fields
	 *
	 * @param  string $current_site_id
	 */
	private static function render_tabs_hidden_fields( $current_site_id ) {

		?>
		<input type="hidden" name="mainwp_rocket_current_site_id" value="<?php echo $current_site_id; ?>">
		<input type="hidden" name="mainwp_rocket_post_nonce" value="<?php echo wp_create_nonce( 'mainwp_rocket_post_nonce' ); ?>" />
		<?php
	}

	/**
	 * Render tabs message
	 *
	 * @param string $message
	 */
	private static function render_tabs_message( $message ) {
		?>
		<!-- Show message on save data -->
		<div class="ui message" id="mainwp-rocket-message-zone" <?php echo self::display_none_style(); ?>></div>
		<!-- End show message on save data -->

		<?php if ( ! empty( $message ) ) : ?>
			<div class="ui green message"><i class="close icon"></i> <?php echo $message; ?></div>
		<?php endif; ?>
		<?php
	}

	/**
	 * Render tabs content function
	 *
	 * @param mixed $is_manage_site
	 * @param mixed $current_site_id
	 * @param mixed $db_websites_wprocket
	 */
	private static function render_tabs_content( $is_manage_site, $current_site_id, $db_websites_wprocket ) {

		?>
		<?php if ( ! $is_manage_site ) : ?>
			<div class="ui active tab" id="mainwp-rocket-dashboard" data-tab="dashboard">
				<?php MainWP_Rocket_Plugin::render_mainwp_rocket_bulk_actions(); ?>
				<?php MainWP_Rocket_Plugin::gen_dashboard_tab( $db_websites_wprocket ); ?>
			</div>
		<?php endif; ?>

		<!-- tab dashboard -->
		<?php self::render_tabs_dashboard( $is_manage_site, $current_site_id ); ?>
		<!-- tab file optimization -->
		<?php self::render_tabs_file_optimization(); ?>
		<!-- tab media -->
		<?php self::render_tabs_media(); ?>
		<!-- tab preload -->
		<?php self::render_tabs_preload(); ?>
		<!-- tab advanced rules -->
		<?php self::render_tabs_advanced_rules(); ?>
		<!-- tab database -->
		<?php self::render_tabs_database(); ?>
		<!-- tab cdn -->
		<?php self::render_tabs_cdn(); ?>
		<!-- tab heartbeat -->
		<?php self::render_tabs_heartbeat(); ?>
		<!-- tab add-on -->
		<?php self::render_tabs_add_on(); ?>
		<!-- tab add-on -->
		<?php self::render_tabs_tools( $current_site_id ); ?>
		<div class="ui clearing hidden divider"></div>
		<?php
	}

	private static function render_tabs_enqueue_action_script( $action, $current_site_id ) {

		?>
		<script type="text/javascript">
			jQuery(document).ready(function($) {
				<?php if ( $action == 'save_opts_child_sites' && isset( $_GET['optimize_db'] ) && $_GET['optimize_db'] ) : ?>
					$('<input>').attr({
						type: 'hidden',
						name: 'rocket_do_optimize_db',
						value: '1'
					}).appendTo('#mainwp-rocket-settings-form');
				<?php endif; ?>
				mainwp_rocket_individual_perform_action(<?php echo $current_site_id; ?>, '<?php echo esc_js( $action ); ?>');
			});
		</script>

		<?php
	}

	/**
	 * Render tab content page
	 *
	 * @param mixed $p_website
	 */
	public static function render_tabs( $p_website = null ) {
		global $mainWPRocketExtensionActivator; // phpcs:ignore

		$message = self::render_tabs_get_message_from_query();

		$db_websites_wprocket = self::render_tabs_get_db_websites_wprocket( $mainWPRocketExtensionActivator );

		$perform_action = self::render_tabs_get_perform_action_from_query();
		$action         = esc_html( $perform_action );

		$is_manage_site  = self::is_manage_sites_page();
		$current_site_id = $is_manage_site ? intval( $p_website->id ?? 0 ) : 0;

		// render tab is overview
		if ( $is_manage_site ) {
			do_action( 'mainwp_pageheader_sites', 'WPRocket' );
		}

		// render modal
		if ( ! $is_manage_site && ! empty( $action ) ) :
			self::render_tabs_modal( $action );
		endif;

		?>
		<?php self::render_tabs_mainwp_rocket_menu( $is_manage_site ); ?>
		<div class="ui segment">
			<?php self::render_tabs_message( $message ); ?>
			<form action="options.php" class="ui form" id="mainwp-rocket-settings-form" method="post" enctype="multipart/form-data">
				<?php self::render_tabs_hidden_fields( $current_site_id ); ?>
				<?php settings_fields( 'mainwp_wp_rocket' ); ?>
				<?php
				if ( $is_manage_site ) {
					self::site_settings_top_box( $current_site_id );
				}
				?>
				<div id="<?php $is_manage_site ? 'mainwp-individual-rocket-settings' : 'mainwp-rocket-wp-rocket-settings'; ?>">
					<?php self::render_tabs_content( $is_manage_site, $current_site_id, $db_websites_wprocket ); ?>
				</div>
			</form>
		</div>
		<?php self::render_tabs_enqueue_script(); ?>

		<?php if ( $is_manage_site && ! empty( $action ) ) : ?>
			<?php self::render_tabs_enqueue_action_script( $action, $current_site_id ); ?>
			<?php do_action( 'mainwp_pagefooter_sites', 'WPRocket' ); ?>
		<?php endif; ?>
		<?php
	}

	/**
	 * Render html save button.
	 */
	public static function render_save_button() {
		?>
		<div class="ui clearing divider"></div>
		<input type="submit" name="submit" id="submit" class="ui green big button" value="<?php esc_attr_e( 'Save Changes', 'mainwp-rocket-extension' ); ?>" />
		<?php
	}

	/**
	 * WP Rocket Site Settings on top.
	 *
	 * @param mixed $site_id
	 */
	public static function site_settings_top_box( $site_id ) {
		$override = 0;
		if ( $site_id ) {
			$site_wprocket = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $site_id, 'override' );
			if ( $site_wprocket ) {
				$override = $site_wprocket->override;
			}
		}
		?>
		<div class="ui message" id="mainwp-rocket-top-site-settings">
			<div class="ui message status" style="display:none"></div>
			<h3 class="ui header"><?php esc_html_e( 'Site-Specific Settings', 'mainwp-rocket-extension' ); ?></h3>
			<div class="ui hidden divider"></div>
			<div class="ui grid field">
				<label class="six wide middle aligned column" for="mainwp_rocket_override_general_settings">
					<?php echo __( 'Override general settings', 'mainwp-rocket-extension' ); ?>
				</label>
				<div class="ui one wide middle aligned column toggle checkbox">
					<input type="checkbox" value="1" name="mainwp_rocket_override_general_settings" <?php checked( $override, 1 ); ?> id="mainwp_rocket_override_general_settings">
				</div>
				<div class="nine wide middle aligned column">
					<input class="ui mini green button rocket_individual_settings_save_btn" type="button" value="<?php echo __( 'Save', 'mainwp-rocket-extension' ); ?>" />
				</div>
			</div>
		</div>
		<div class="ui hidden divider"></div>
		<?php
		if ( get_option( 'mainwp_rocket_perform_individual_settings_update' ) == 1 ) {
			delete_option( 'mainwp_rocket_perform_individual_settings_update' );
			?>
			<script type="text/javascript">
				mainwp_rocket_individual_save_settings(<?php echo $site_id; ?>);
			</script>
			<?php
		}
	}

	/**
	 * Is manage sites page.
	 *
	 * @return bool
	 */
	public static function is_manage_sites_page() {
		if (
			( ! empty( $_POST['individual'] ) && ! empty( $_POST['wprocketRequestSiteID'] ) ) ||
			( ! empty( $_GET['page'] ) && 'ManageSitesWPRocket' == $_GET['page'] ) ||
			( ! empty( $_REQUEST['mainwp_rocket_current_site_id'] ) )
		) {
			return true;
		}

		return false;
	}

	/**
	 * Get manage site id
	 *
	 * @return int
	 */
	public static function get_manage_site_id() {
		$site_id = 0;
		if ( self::is_manage_sites_page() ) {
			if ( isset( $_POST['individual'] ) && ! empty( $_POST['individual'] ) && isset( $_REQUEST['wprocketRequestSiteID'] ) && ! empty( $_REQUEST['wprocketRequestSiteID'] ) ) {
				$site_id = intval( $_REQUEST['wprocketRequestSiteID'] );
			} elseif ( isset( $_GET['id'] ) && ! empty( $_GET['id'] ) ) {
				$site_id = intval( $_GET['id'] );
			} elseif ( isset( $_REQUEST['mainwp_rocket_current_site_id'] ) && ! empty( $_REQUEST['mainwp_rocket_current_site_id'] ) ) {
				$site_id = intval( $_REQUEST['mainwp_rocket_current_site_id'] );
			}
		}
		return $site_id;
	}

	/**
	 * Ajax general load sites get action from post
	 *
	 * @param string $default_action string action
	 */
	private static function ajax_general_load_sites_get_action_from_post( $default_action ) {
		return ! empty( $_POST['rightnow_action'] ) ? $_POST['rightnow_action'] : $default_action;
	}

	/**
	 * Ajax general load sites get websites with plugin
	 */
	public static function ajax_general_load_sites_get_websites_with_plugin() {
		global $mainWPRocketExtensionActivator; // phpcs:ignore
		return apply_filters(
			'mainwp_getsites',
			$mainWPRocketExtensionActivator->get_child_file(),
			$mainWPRocketExtensionActivator->get_child_key(),
			null,
			false,
			array( 'plugins_slug' => MAINWP_ROCKET_FILE_PATH )
		);
	}

	/**
	 * Ajax general load sites extract site ids
	 *
	 * @param mixed $websites websites data.
	 */
	private static function ajax_general_load_sites_extract_site_ids( $websites ) {
		$sites_ids = array();
		if ( is_array( $websites ) ) {
			foreach ( $websites as $website ) {
				$sites_ids[] = $website['id'];
			}
		}
		return $sites_ids;
	}

	/**
	 * Ajax general load sites get db websites
	 *
	 * @param array $sites_ids website ids.
	 */
	private static function ajax_general_load_sites_get_db_websites( $sites_ids ) {
		global $mainWPRocketExtensionActivator; // phpcs:ignore
		return apply_filters(
			'mainwp_getdbsites',
			$mainWPRocketExtensionActivator->get_child_file(),
			$mainWPRocketExtensionActivator->get_child_key(),
			$sites_ids,
			array(),
			array(
				'plugin_upgrades' => true,
				'plugins'         => true,
			)
		);
	}
	/**
	 * Ajax general load sites has active site
	 *
	 * @param mixed $websites website data.
	 */
	private static function ajax_general_load_sites_has_active_sites( $websites ) {
		if ( ! empty( $websites ) && is_array( $websites ) ) {
			foreach ( $websites as $website ) {
				if ( ! empty( $website['wprocket_active'] ) ) {
					return true;
				}
			}
		}
		return false;
	}
	private static function ajax_general_load_sites_display_active_sites( $websites ) {
		?>

		<div class="ui relaxed divided list">
			<?php
			foreach ( $websites as $website ) :
				if ( empty( $website['wprocket_active'] ) ) :
					continue;
				endif;
				$site_id   = $website['id'];
				$site_name = stripslashes( $website['name'] );
				?>
				<div class='item processing-item' site-id='<?php echo $site_id; ?>' status='queue'>
					<?php echo $site_name; ?>
					<span class='status right floated'><i class='clock outline icon'></i></span>
				</div>

			<?php endforeach; ?>
		</div>
		<?php
	}
	/**
	 * Ajax general load display no active sites message
	 */
	private static function ajax_general_load_display_no_active_sites_message() {
		?>
		<div class="ui yellow message">
			<?php echo __( 'WP Rocket plugin not detected on the child sites. Please, install and activate the plugin first.', 'mainwp-rocket-extension' ); ?>
		</div>;
		<?php
	}
	/**
	 * Ajax general load enqueue action scrip
	 *
	 * @param mixed $action action
	 */
	private static function ajax_general_load_enqueue_action_script( $action ) {
		?>
		<script type="text/javascript">
			jQuery(document).ready(function($) {
				rocket_bulkTotalThreads = jQuery('.processing-item[status=queue]').length;
				if (rocket_bulkTotalThreads > 0) {
					<?php if ( ! empty( $action ) ) { ?>
						mainwp_rocket_perform_action_start_next('<?php echo esc_js( $action ); ?>');
					<?php } ?>
				}
			});
		</script>

		<?php
	}
	/**
	 * Ajax general load sites.
	 *
	 * @param mixed $action
	 */
	public static function ajax_general_load_sites( $action = null ) {

		$action = self::ajax_general_load_sites_get_action_from_post( $action );
		// get sites with the wp-rocket plugin installed only
		$websites  = self::ajax_general_load_sites_get_websites_with_plugin();
		$sites_ids = self::ajax_general_load_sites_extract_site_ids( $websites );

		$db_websites         = self::ajax_general_load_sites_get_db_websites( $sites_ids );
		$dbwebsites_wprocket = MainWP_Rocket_Plugin::get_instance()->get_websites_with_the_plugin( $db_websites );

		if ( self::ajax_general_load_sites_has_active_sites( $dbwebsites_wprocket ) ) {
			self::ajax_general_load_sites_display_active_sites( $dbwebsites_wprocket );
		} else {
			self::ajax_general_load_display_no_active_sites_message();
		}

		if ( ! empty( $action ) ) {
			self::ajax_general_load_enqueue_action_script( $action );
		}

		if ( ! empty( $_POST['_wprocketNonce'] ) ) {
			die();
		}
	}

	/**
	 * Ajax check data
	 */
	public function ajax_check_data() {
		$action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : '';
		do_action( 'mainwp_secure_request', $action );

		if ( ! isset( $_REQUEST['_wprocketNonce'] ) || ! wp_verify_nonce( $_REQUEST['_wprocketNonce'], 'mainwp_rocket_nonce' ) ) {
			die( wp_json_encode( array( 'error' => __( 'Nonce authentication failed. Please, try again.', 'mainwp-rocket-extension' ) ) ) );
		}
		if ( empty( $_POST['wprocketRequestSiteID'] ) ) {
			die( wp_json_encode( array( 'error' => __( 'Site ID not found. Please, try again.', 'mainwp-rocket-extension' ) ) ) );
		}
	}
	/**
	 * Ajax check individual setting
	 *
	 * @param mixed $pSiteId
	 */
	public function ajax_check_individual_setting( $pSiteId ) {
		$site_wprocket = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $pSiteId, 'override, is_active' );
		$msg           = __( 'To manage WP Rocket for the individual site, first enable the "Overwrite general settings" option.', 'mainwp-rocket-extension' );
		if ( isset( $_POST['individual'] ) ) {
			$individual = ! empty( $_POST['individual'] ) ? true : false;
			if ( $individual ) {
				if ( $site_wprocket && ! $site_wprocket->override ) {
					die( wp_json_encode( array( 'error' => esc_html( $msg ) ) ) );
				}
			} elseif ( $site_wprocket && $site_wprocket->override ) {
				die( wp_json_encode( array( 'error' => esc_html( $msg ) ) ) );
			}
		}
	}

	/**
	 * Ajax override settings.
	 */
	public function ajax_override_settings() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		global $mainWPRocketExtensionActivator;
		$website = apply_filters( 'mainwp_getsites', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId );

		if ( $website && is_array( $website ) ) {
			$website = current( $website );
		}

		if ( ! $website ) {
			die( wp_json_encode( array( 'error' => __( 'Site data not found.', 'mainwp-rocket-extension' ) ) ) );
		}

		$update = array(
			'site_id'  => $website['id'],
			'override' => ! empty( $_POST['override'] ) && $_POST['override'] === '1' ? 1 : 0,
		);

		MainWP_Rocket_DB::get_instance()->update_wprocket( $update );

		die( wp_json_encode( array( 'result' => 'SUCCESS' ) ) );
	}

	/**
	 * Ajax reload optimize info
	 */
	public function ajax_reload_optimize_info() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		global $mainWPRocketExtensionActivator;
		$post_data   = array( 'mwp_action' => 'get_optimize_info' );
		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
		die( wp_json_encode( $information ) );
	}

	/**
	 * Ajax purge cloudflare
	 */
	public function ajax_purge_cloudflare() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		$this->ajax_check_individual_setting( $websiteId );

		global $mainWPRocketExtensionActivator;
		$post_data = array( 'mwp_action' => 'purge_cloudflare' );

		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
		die( wp_json_encode( $information ) );
	}

	/**
	 * Ajax preload purge cache all
	 */
	public function ajax_preload_purge_cache_all() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];

		$individual = ! empty( $_POST['individual'] ) ? true : false;
		if ( ! $individual ) {
			$this->ajax_check_individual_setting( $websiteId );
		}

		global $mainWPRocketExtensionActivator;
		$post_data   = array( 'mwp_action' => 'preload_purge_all' );
		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );

		if (
			isset( $_POST['where'] ) && 'dashboard_tab' == $_POST['where'] &&
			is_array( $information ) && isset( $information['result'] ) && 'SUCCESS' == $information['result']
		) {
			$update        = array( 'site_id' => $websiteId );
			$wprocket_data = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $websiteId, 'others' );
			$others        = unserialize( base64_decode( $wprocket_data->others ) );
			if ( ! is_array( $others ) ) {
				$others = array();
			}
			if ( isset( $data ) && ! empty( $data['rocket_boxes'] ) ) {
				if ( ! is_array( $data['rocket_boxes'] ) ) {
					$data['rocket_boxes'] = array();
				}
				$others['rocket_boxes'][] = 'rocket_warning_plugin_modification';
			} else {
				$others['rocket_boxes'] = array( 'rocket_warning_plugin_modification' );
			}
			$update['others'] = base64_encode( serialize( $others ) );
			MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
		}

		die( wp_json_encode( $information ) );
	}

	/**
	 * Ajax Clear priority elements
	 */
	public function ajax_clear_priority_elements() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];

		$individual = ! empty( $_POST['individual'] ) ? true : false;
		if ( ! $individual ) {
			$this->ajax_check_individual_setting( $websiteId );
		}

		global $mainWPRocketExtensionActivator;
		$post_data   = array( 'mwp_action' => 'clear_priority_elements' );
		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );

		die( wp_json_encode( $information ) );
	}

	/**
	 * Ajax generate critical css.
	 */
	public function ajax_generate_critical_css() {
		$this->ajax_check_data();
		$websiteId = ! empty( $_POST['wprocketRequestSiteID'] ) ? wp_unslash( $_POST['wprocketRequestSiteID'] ) : '';

		$individual = ! empty( $_POST['individual'] ) ? true : false; // phpcs:disable WordPress.Security.NonceVerification
		if ( ! $individual ) {
			$this->ajax_check_individual_setting( $websiteId );
		}

		global $mainWPRocketExtensionActivator;  //phpcs:ignore
		$post_data = array( 'mwp_action' => 'generate_critical_css' );

		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );

		error_log(print_r($_POST, true)); //phpcs:ignore -- NOSONAR - for dev.
		error_log(print_r($information, true)); //phpcs:ignore -- NOSONAR - for dev.

		die( wp_json_encode( $information ) );
	}

	/**
	 * Check override setting
	 *
	 * @param mixed $override string or int
	 * @return void Returns false if the site has been overwritten.
	 */
	private function check_override_settings( $override ) {
		$msg = __( 'To manage WP Rocket for the individual site, first enable the "Overwrite general settings" option.', 'mainwp-rocket-extension' );
		if ( 1 == $override ) {
			die( wp_json_encode( array( 'error' => esc_html( $msg ) ) ) );
		}
	}

	private function ajax_save_opts_to_child_site_retrieve_settings( $website_id, $individual_update ) {
		$settings_site = MainWP_Rocket_DB::get_instance()->get_wprocket_by( 'site_id', $website_id );
		$msg           = __( 'To manage WP Rocket for the individual site, first enable the "Overwrite general settings" option.', 'mainwp-rocket-extension' );

		if ( $individual_update ) {
			if ( $settings_site ) {
				if ( $settings_site->override ) {
					return unserialize( base64_decode( $settings_site->settings ) );
				} else {
					die( wp_json_encode( array( 'error' => esc_html( $msg ) ) ) );
				}
			}
		} else {
			if ( $settings_site ) {
				$this->check_override_settings( $settings_site->override );
			}
			return get_option( MAINWP_ROCKET_GENERAL_SETTINGS );
		}
		return array();
	}

	private function ajax_save_opts_to_child_site_validate_settings( $settings, $individual_update ) {
		if ( empty( $settings ) || ! is_array( $settings ) ) {
			die( wp_json_encode( array( 'error' => $individual_update ? __( 'Invalid individual settings data.', 'mainwp-rocket-extension' ) : __( 'Invalid general settings data.', 'mainwp-rocket-extension' ) ) ) );
		}
	}

	private function ajax_save_opts_to_child_site_prepare_send_fields( $settings ) {
		$send_fields = array();
		$defaults    = mainwp_get_rocket_default_options();
		foreach ( $settings as $field => $value ) {
			if ( isset( $defaults[ $field ] ) ) {
				$send_fields[ $field ] = $value;
			}
		}
		return $send_fields;
	}
	private function ajax_save_opts_to_child_site_get_website_data( $website_id ) {
		global $mainWPRocketExtensionActivator;  //phpcs:ignore
		$website = apply_filters( 'mainwp_getsites', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $website_id );
		return current( $website );
	}

	private function ajax_save_opts_to_child_site_handle_sitemaps( $send_fields, $url ) {
		if ( ! empty( $send_fields['sitemaps'] ) ) {
			$search                  = array( '%url%' );
			$replace                 = array( $url );
			$send_fields['sitemaps'] = str_replace( $search, $replace, $send_fields['sitemaps'] );
		}
		return $send_fields;
	}

	private function ajax_save_opts_to_child_handle_checkbox_defaults( $send_fields ) {
		$defaults = mainwp_get_rocket_default_options();
		foreach ( $defaults as $idx => $val ) {
			if ( ( 0 === $val || 1 === $val ) && ! isset( $send_fields[ $idx ] ) ) {
				$send_fields[ $idx ] = 0;
			}
		}
		return $send_fields;
	}

	private function ajax_save_opts_to_child_database_optimization_status() {
		$database_optimization_status = ! empty( $_POST['optimize_db'] ) && $_POST['optimize_db'] === 'true';
		if ( ! $database_optimization_status && ! empty( $_SERVER['HTTP_REFERER'] ) ) {
			$url_referer = wp_parse_url( esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) );
			$url_query   = array();
			wp_parse_str( $url_referer['query'], $url_query );
			$database_optimization_status = ! empty( $url_query['optimize_db'] );
		}
		return $database_optimization_status;
	}

	private function ajax_save_opts_to_child_request_to_child_site( $websiteId, $post_data ) {
		global $mainWPRocketExtensionActivator;
		return apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
	}

	public function ajax_save_opts_to_child_site() {
		$this->ajax_check_data();
		$website_id        = $_POST['wprocketRequestSiteID'];
		$individual_update = ! empty( $_POST['individual'] ) ? true : false;
		$settings          = $this->ajax_save_opts_to_child_site_retrieve_settings( $website_id, $individual_update );
		$this->ajax_save_opts_to_child_site_validate_settings( $settings, $individual_update );
		$send_fields = $this->ajax_save_opts_to_child_site_prepare_send_fields( $settings );

		$defaults = mainwp_get_rocket_default_options();
		foreach ( $settings as $field => $value ) {
			if ( isset( $defaults[ $field ] ) ) {
				$send_fields[ $field ] = $value;
			}
		}
		$website     = $this->ajax_save_opts_to_child_site_get_website_data( $website_id );
		$send_fields = $this->sanitize_fields( $send_fields, $website['url'], ! $individual_update );

		$send_fields = $this->ajax_save_opts_to_child_site_handle_sitemaps( $send_fields, $website['url'] );
		$send_fields = $this->ajax_save_opts_to_child_handle_checkbox_defaults( $send_fields );

		$database_optimization_status = $this->ajax_save_opts_to_child_database_optimization_status();
		$post_data                    = array(
			'mwp_action'               => 'save_settings',
			'settings'                 => base64_encode( wp_json_encode( $send_fields ) ),
			'do_database_optimization' => $database_optimization_status,
		);
		$information                  = $this->ajax_save_opts_to_child_request_to_child_site( $website_id, $post_data );

		die( wp_json_encode( $information ) );
	}

	/**
	 * [Description for ajax_optimize_database_site]
	 */
	public function ajax_optimize_database_site() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		$this->ajax_check_individual_setting( $websiteId );

		global $mainWPRocketExtensionActivator;
		$post_data = array( 'mwp_action' => 'optimize_database' );

		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
		die( wp_json_encode( $information ) );
	}

	/**
	 * [Update Inclusion and Exclusion Lists for ajax_update_exclusion_list]
	 */
	public function ajax_update_exclusion_list() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		$this->ajax_check_individual_setting( $websiteId );

		global $mainWPRocketExtensionActivator;
		$post_data = array( 'mwp_action' => 'update_exclusion_list' );

		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
		die( wp_json_encode( $information ) );
	}

	/**
	 * [Ajax load existing settings  for ajax_load_existing_settings]
	 */
	public function ajax_load_existing_settings() {
		$this->ajax_check_data();
		$websiteId = $_POST['wprocketRequestSiteID'];
		global $mainWPRocketExtensionActivator;
		$post_data   = array( 'mwp_action' => 'load_existing_settings' );
		$information = apply_filters( 'mainwp_fetchurlauthed', $mainWPRocketExtensionActivator->get_child_file(), $mainWPRocketExtensionActivator->get_child_key(), $websiteId, 'wp_rocket', $post_data );
		if ( ( is_array( $information ) && isset( $information['result'] ) && 'SUCCESS' == $information['result'] ) && ( isset( $information['options'] ) && is_array( $information['options'] ) ) ) {
			$options      = $information['options'];
			$save_fields  = mainwp_get_rocket_default_options();
			$save_options = array();
			foreach ( $options as $field => $value ) {
				if ( isset( $save_fields[ $field ] ) ) {
					$save_options[ $field ] = $value;
				}
			}
			$update = array(
				'site_id'  => $websiteId,
				'settings' => base64_encode( serialize( $save_options ) ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- base64_encode sync setting.
			);
			MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
			unset( $information['options'] );
		}
		die( wp_json_encode( $information ) );
	}

	/**
	 * Sanitize fields
	 *
	 * @param mixed $input
	 * @param mixed $site_url
	 * @param mixed $isGeneral
	 * @return mixed
	 */
	public function sanitize_fields( $input, $site_url, $isGeneral ) {
		if ( $isGeneral && isset( $input['cloudflare_domain'] ) && 1 == $input['cloudflare_domain'] ) {
			$input['cloudflare_domain'] = $site_url;
		}

		return $input;
	}

	/**
	 * [show action WP Rocket to Updates Overview for mainwp_rocket_hook_buttons]
	 */
	public function mainwp_rocket_hook_buttons() {
		$site_id = 0;

		if ( isset( $_GET['page'] ) && 'managesites' == $_GET['page'] ) {
			$site_id = isset( $_GET['dashboard'] ) ? intval( $_GET['dashboard'] ) : 0;
			if ( empty( $site_id ) ) {
				return;
			}
		}
		?>
		<div class="ui light card">
			<div class="content">
				<div class="header">
					<span class="ui large text">
						<i class="rocket icon"></i>
						<?php esc_html_e( 'WP Rocket', 'mainwp' ); ?>
					</span>
				</div>
				<div class="description"><?php esc_html_e( 'Clear & Preload Cache', 'mainwp-rocket-cache' ); ?></div>
			</div>
			<div class="extra content">
				<button class="ui mini fluid green button btn-preload-purge-cache" onclick="return <?php echo ! $site_id ? "mainwp_rocket_rightnow_loadsites( 'preload_purge_cache_all' );" : 'mainwp_rocket_rightnow_clearcache_individual( ' . esc_html( $site_id ) . ' );'; ?>"><?php esc_html_e( 'Clear & Preload WP Rocket Cache', 'mainwp-rocket-extension' ); ?></button>
			</div>
		</div>


		<div class="ui modal" id="mainwp-rocket-overview-page-clear-cache-modal">
			<i class="close reload icon"></i>
			<div class="header"><?php esc_html_e( 'MainWP Rocket', 'mainwp-rocket-extension' ); ?></div>
			<div class="scrolling content"></div>
		</div>
		<?php
	}

	/**
	 * [Hook add item to dropdown id mainwp-sites-bulk-actions-menu for managesites_bulk_actions]
	 *
	 * @param mixed $actions
	 */
	public function managesites_bulk_actions( $actions ) {
		$actions['clear_wprocket_preload_cache'] = __( 'Clear and Preload Cache', 'mainwp-rocket-extension' );
		return $actions;
	}

	/**
	 * [Description for get_template_path]
	 *
	 * @param mixed $path
	 */
	private function get_template_path( $path ) {
		return $this->template_path . '/' . $path . '.php';
	}

	/**
	 * [Description for checkbox]
	 *
	 * @param mixed $args data.
	 */
	public function checkbox( $args ) {
		echo $this->generate( 'fields/checkbox', $args );
	}

	/**
	 * [generate for get_template_path]
	 *
	 * @param mixed $template template name.
	 * @param array $data data use template generate.
	 */
	public function generate( $template, $data = array() ) { //phpcs:ignore -- NOSONAR - data current not use
		// Note: $data current not use.
		$template_path = $this->get_template_path( $template );

		if ( ! file_exists( $template_path ) ) {
			return;
		}

		ob_start();

		include_once $template_path; //phpcs:ignore -- NOSONAR -- include file

		return trim( ob_get_clean() );
	}
}
