<?php

/*
 * Formatting
 *
 */
defined( 'ABSPATH' ) || exit;

/**
 * Get relative url
 * Clean URL file to get only the equivalent of REQUEST_URI
 * ex: mainwp_rocket_clean_exclude_file( 'http://www.geekpress.fr/referencement-wordpress/') return /referencement-wordpress/
 *
 * @since 1.3.5 Redo the function
 * @since 1.0
 *
 * @param  string $file URL we want to parse.
 * @return bool\string false if $file is empty or false, relative path otherwise
 */
function mainwp_rocket_clean_exclude_file( $file ) {
    if ( ! $file ) {
        return false;
    }

    return wp_parse_url( $file, PHP_URL_PATH );
}

/**
 * Clean Never Cache URL(s) bad wildcards
 *
 * @since  3.4.2
 * @author Soponar Cristina
 *
 * @param  string $path URL which needs to be cleaned.
 * @return bool\string  false if $path is empty or cleaned URL
 */
function mainwp_rocket_clean_wildcards( $path ) {
    if ( ! $path ) {
        return false;
    }

    $path_components = explode( '/', $path );
    $arr             = array(
        '.*'   => '(.*)',
        '*'    => '(.*)',
        '(*)'  => '(.*)',
        '(.*)' => '(.*)',
    );

    foreach ( $path_components as &$path_component ) {
        $path_component = strtr( $path_component, $arr );
    }

    return implode( '/', $path_components );
}


/**
 * Used with array_filter to remove files without .css extension
 *
 * @since 1.0
 *
 * @param  string $file filepath to sanitize.
 * @return bool\string false if not a css file, filepath otherwise
 */
function mainwp_rocket_sanitize_css( $file ) {
    $file = preg_replace( MAINWP_ROCKET_PREG_SANITIZE, '', $file );
    $ext  = strtolower( pathinfo( $file, PATHINFO_EXTENSION ) );
    return ( 'css' === $ext || 'php' === $ext ) ? trim( $file ) : false;
}

/**
 * Used with array_filter to remove files without .js extension
 *
 * @since 1.0
 *
 * @param  string $file filepath to sanitize.
 * @return bool\string false if not a js file, filepath otherwise
 */
function mainwp_rocket_sanitize_js( $file ) {
    $file = preg_replace( MAINWP_ROCKET_PREG_SANITIZE, '', $file );
    $ext  = strtolower( pathinfo( $file, PATHINFO_EXTENSION ) );
    return ( 'js' === $ext || 'php' === $ext ) ? trim( $file ) : false;
}

/**
 * Sanitize and validate JS files to exclude from the minification.
 *
 * @since  3.3.7
 * @author Remy Perona
 * @author Grégory Viguier
 *
 * @param  string $file filepath to sanitize.
 * @return string
 */
function mainwp_rocket_validate_js( $file ) {
    if ( mainwp_rocket_is_internal_file( $file ) ) {
        $file = trim( $file );
        $file = mainwp_rocket_clean_exclude_file( $file );
        $file = mainwp_rocket_sanitize_js( $file );

        return $file;
    }

    return mainwp_rocket_remove_url_protocol( esc_url_raw( strtok( $file, '?' ) ) );
}

/**
 * Sanitize and validate CSS files to exclude from the minification.
 *
 * @since 3.7
 *
 * @param  string $file filepath to sanitize.
 * @return string
 */
function mainwp_rocket_validate_css( $file ) {
    if ( mainwp_rocket_is_internal_file( $file ) ) {
        return mainwp_rocket_sanitize_css( mainwp_rocket_clean_exclude_file( trim( $file ) ) );
    }

    return mainwp_rocket_remove_url_protocol( esc_url_raw( strtok( $file, '?' ) ) );
}

/**
 * Check if the passed value is an internal URL (default domain or CDN/Multilingual).
 *
 * @since 3.3.7
 *
 * @param  string $file string to test.
 * @return bool
 */
function mainwp_rocket_is_internal_file( $file ) {
    $file_host = wp_parse_url( mainwp_rocket_add_url_protocol( $file ), PHP_URL_HOST );

    if ( empty( $file_host ) ) {
        return false;
    }

    /**
     * Filters the allowed hosts for optimization
     *
     * @since 3.4
     *
     * @param array $hosts Allowed hosts.
     * @param array $zones Zones to check available hosts.
     */
    $hosts   = apply_filters( 'rocket_cdn_hosts', array(), array( 'all', 'css_and_js', 'css', 'js' ) );
    $hosts[] = wp_parse_url( content_url(), PHP_URL_HOST );
    if ( function_exists( 'mainwp_rocket_is_stream' ) ) {
        $langs = mainwp_get_rocket_i18n_uri();

        // Get host for all langs.
        if ( ! empty( $langs ) ) {
            foreach ( $langs as $lang ) {
                $hosts[] = wp_parse_url( $lang, PHP_URL_HOST );
            }
        }
    }

    $hosts = array_unique( $hosts );

    if ( empty( $hosts ) ) {
        return false;
    }

    return in_array( $file_host, $hosts, true );
}

/**
 * Sanitize a setting value meant for a textarea.
 *
 * @since 3.3.7
 *
 * @param  string       $field The field’s name. Can be one of the following:
 *                             'exclude_css', 'exclude_inline_js', 'exclude_js', 'cache_reject_uri',
 *                             'cache_reject_ua', 'cache_purge_pages', 'cdn_reject_files'.
 * @param  array|string $value The value to sanitize.
 * @return array|null
 */
function mainwp_rocket_sanitize_textarea_field( $field, $value ) {
    $fields = array(
        'cache_purge_pages'    => array( 'esc_url', 'mainwp_rocket_clean_exclude_file', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'cache_reject_cookies' => array( 'mainwp_rocket_sanitize_key' ),
        'cache_reject_ua'      => array( 'mainwp_rocket_sanitize_ua', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'cache_reject_uri'     => array( 'esc_url', 'mainwp_rocket_clean_exclude_file', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'cache_query_strings'  => array( 'mainwp_rocket_sanitize_key' ),
        'cdn_reject_files'     => array( 'mainwp_rocket_clean_exclude_file', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'exclude_css'          => array( 'mainwp_rocket_validate_css', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'exclude_inline_js'    => array( 'sanitize_text_field' ),
        'exclude_defer_js'     => array( 'sanitize_text_field' ),
        'exclude_js'           => array( 'mainwp_rocket_validate_js', 'mainwp_rocket_clean_wildcards' ), // Pattern.
        'exclude_lazyload'     => array( 'sanitize_text_field' ),
        'delay_js_scripts'     => array( 'sanitize_text_field' ),
        'preload_excluded_uri' => array( 'esc_url', 'mainwp_rocket_clean_exclude_file', 'mainwp_rocket_clean_wildcards' ), // Pattern.
    );

    if ( ! isset( $fields[ $field ] ) ) {
        return null;
    }

    $sanitizations = $fields[ $field ];

    if ( ! is_array( $value ) ) {
        $value = explode( "\n", $value );
    }

    $value = array_map( 'trim', $value );
    $value = array_filter( $value );

    if ( ! $value ) {
        return array();
    }

    // Sanitize.
    foreach ( $sanitizations as $sanitization ) {
        $value = array_map( $sanitization, $value );
    }

    return array_unique( $value );
}

/**
 * Used with array_filter to remove files without .xml extension
 *
 * @since  2.8
 * @author Remy Perona
 *
 * @param  string $file filepath to sanitize.
 * @return string|boolean filename or false if not xml
 */
function mainwp_rocket_sanitize_xml( $file ) {
    $file = preg_replace( MAINWP_ROCKET_PREG_SANITIZE, '', $file );
    $ext  = strtolower( pathinfo( $file, PATHINFO_EXTENSION ) );
    return ( 'xml' === $ext ) ? trim( $file ) : false;
}

/**
 * Sanitizes a string key like the sanitize_key() WordPress function without forcing lowercase.
 *
 * @since 2.7
 *
 * @param  string $key Key string to sanitize.
 * @return string
 */
function mainwp_rocket_sanitize_key( $key ) {
    $key = preg_replace( '/[^a-z0-9_\-]/i', '', $key );
    return $key;
}

/**
 * Used to sanitize values of the "Never send cache pages for these user agents" option.
 *
 * @since 2.6.4
 *
 * @param  string $user_agent User Agent string.
 * @return string
 */
function mainwp_rocket_sanitize_ua( $user_agent ) {
    $user_agent = preg_replace( '/[^a-z0-9._\(\)\*\-\/\s\x5c]/i', '', $user_agent );
    return $user_agent;
}

/**
 * Get an url without HTTP protocol
 *
 * @since 1.3.0
 *
 * @param  string $url     The URL to parse.
 * @param  bool   $no_dots (default: false).
 * @return string $url The URL without protocol
 */
function mainwp_rocket_remove_url_protocol( $url, $no_dots = false ) {
    $url = preg_replace( '#^(https?:)?\/\/#im', '', $url );

    /**
    * This filter is documented in inc/front/htaccess.php
*/
    if ( apply_filters( 'rocket_url_no_dots', $no_dots ) ) {
        $url = str_replace( '.', '_', $url );
    }
    return $url;
}

/**
 * Add HTTP protocol to an url that does not have it.
 *
 * @since 2.2.1
 *
 * @param string $url The URL to parse.
 *
 * @return string $url The URL with protocol.
 */
function mainwp_rocket_add_url_protocol( $url ) {
    // Bail out if the URL starts with http:// or https://.
    if (
    strpos( $url, 'http://' ) !== false
    || strpos( $url, 'https://' ) !== false
    ) {
        return $url;
    }

    if ( substr( $url, 0, 2 ) !== '//' ) {
        $url = '//' . $url;
    }

    return set_url_scheme( $url );
}

/**
 * Set the scheme for a internal URL
 *
 * @since 2.6
 *
 * @param  string $url Absolute url that includes a scheme.
 * @return string $url URL with a scheme.
 */
function mainwp_rocket_set_internal_url_scheme( $url ) {
    $tmp_url = set_url_scheme( $url );

    if ( mainwp_rocket_extract_url_component( $tmp_url, PHP_URL_HOST ) === mainwp_rocket_extract_url_component( home_url(), PHP_URL_HOST ) ) {
        $url = $tmp_url;
    }

    return $url;
}

/**
 * Get the domain of an URL without subdomain
 * (ex: mainwp_rocket_get_domain( 'http://www.geekpress.fr' ) return geekpress.fr
 *
 * @source : http://stackoverflow.com/a/15498686
 * @since  2.7.3 undeprecated & updated
 * @since  1.0
 *
 * @param  string $url URL to parse.
 * @return string|bool Domain or false
 */
function mainwp_rocket_get_domain( $url ) {
    // Add URL protocol if the $url doesn't have one to prevent issue with parse_url.
    $url = mainwp_rocket_add_url_protocol( trim( $url ) );

    $url_array = wp_parse_url( $url );
    $host      = $url_array['host'];
    /**
     * Filters the tld max range for edge cases
     *
     * @since 2.7.3
     *
     * @param string Max range number
     */
    $match = '/(?P<domain>[a-z0-9][a-z0-9\-]{1,63}\.[a-z\.]{2,' . apply_filters( 'rocket_get_domain_preg', '6' ) . '})$/i';

    if ( preg_match( $match, $host, $regs ) ) {
        return $regs['domain'];
    }

    return false;
}

/**
 * Extract a component from an URL.
 *
 * @since  2.11
 * @author Remy Perona
 *
 * @param  string $url       URL to parse and extract component of.
 * @param  string $component URL component to extract using constant as in parse_url().
 * @return string extracted component
 */
function mainwp_rocket_extract_url_component( $url, $component ) {
    return _get_component_from_parsed_url_array( wp_parse_url( $url ), $component );
}


/**
 * Test if a given path is a stream URL.
 *
 * @since 3.5.3
 *
 * @source wp_is_stream() in /wp-includes/functions.php
 *
 * @param string $path The resource path or URL.
 *
 * @return bool true if the path is a stream URL; else false.
 */
function mainwp_rocket_is_stream( $path ) {
    $scheme_separator = strpos( $path, '://' );

    if ( false === $scheme_separator ) {
        // $path isn't a stream.
        return false;
    }

    $stream = substr( $path, 0, $scheme_separator );

    return in_array( $stream, stream_get_wrappers(), true );
}

/**
 * Returns realpath to file (used for relative path with /../ in it or not-yet existing file)
 *
 * @since  2.11
 * @author Remy Perona
 *
 * @param  string $file File to determine realpath for.
 * @return string Resolved file path
 */
function mainwp_rocket_realpath( $file ) {
    $wrapper = null;

    // Strip the protocol.
    if ( function_exists( 'mainwp_rocket_is_stream' ) && mainwp_rocket_is_stream( $file ) ) {
        list($wrapper, $file) = explode( '://', $file, 2 );
    }

    $path = array();

    foreach ( explode( '/', $file ) as $part ) {
        if ( '' === $part || '.' === $part ) {
            continue;
        }

        if ( '..' !== $part ) {
            array_push( $path, $part );
        } elseif ( count( $path ) > 0 ) {
            array_pop( $path );
        }
    }

    $file = join( '/', $path );

    if ( null !== $wrapper ) {
        return $wrapper . '://' . $file;
    }

    $prefix = 'WIN' === strtoupper( substr( PHP_OS, 0, 3 ) ) ? '' : '/';

    return $prefix . $file;
}

/**
 * Checks if the constant is defined.
 *
 * NOTE: This function allows mocking constants when testing.
 *
 * @since 3.5
 *
 * @param string $constant_name Name of the constant to check.
 *
 * @return bool true when constant is defined; else, false.
 */
function mainwp_rocket_has_constant( $constant_name ) {
    return defined( $constant_name );
}

/**
 * Gets the constant is defined.
 *
 * NOTE: This function allows mocking constants when testing.
 *
 * @since 3.5
 *
 * @param string     $constant_name Name of the constant to check.
 * @param mixed|null $default Optional. Default value to return if constant is not defined.
 *
 * @return mixed
 */
function mainwp_rocket_get_constant( $constant_name, $default = null ) {
    if ( ! mainwp_rocket_has_constant( $constant_name ) ) {
        return $default;
    }

    return constant( $constant_name );
}

/**
 * Converts an URL to an absolute path.
 *
 * @since  2.11.7
 * @author Remy Perona
 *
 * @param  string $url   URL to convert.
 * @param  array  $zones Zones to check available hosts.
 * @return string|bool
 */
function mainwp_rocket_url_to_path( $url, array $zones = array( 'all' ) ) {
    $wp_content_dir = mainwp_rocket_get_constant( 'WP_CONTENT_DIR', '' );
    $root_dir       = trailingslashit( dirname( $wp_content_dir ) );
    $root_url       = str_replace( wp_basename( $wp_content_dir ), '', content_url() );
    $url_host       = wp_parse_url( $url, PHP_URL_HOST );

    // relative path.
    if ( null === $url_host ) {
        $subdir_levels = substr_count( preg_replace( '/https?:\/\//', '', site_url() ), '/' );
        $url           = trailingslashit( site_url() . str_repeat( '/..', $subdir_levels ) ) . ltrim( $url, '/' );
    }

    /**
     * Filters the URL before converting it to a path
     *
     * @since  3.5.3
     * @author Remy Perona
     *
     * @param string $url   URL of the asset.
     * @param array  $zones CDN zones corresponding to the current assets type.
     */
    $url = apply_filters( 'rocket_asset_url', $url, $zones );

    $url      = rawurldecode( $url );
    $root_url = preg_replace( '/^https?:/', '', $root_url );
    $url      = preg_replace( '/^https?:/', '', $url );
    $file     = str_replace( $root_url, $root_dir, $url );
    $file     = mainwp_rocket_realpath( $file );

    /**
     * Filters the absolute path to the asset file
     *
     * @since  3.3
     * @author Remy Perona
     *
     * @param string $file Absolute path to the file.
     * @param string $url  URL of the asset.
     */
    $file = apply_filters( 'rocket_url_to_path', $file, $url );

    if ( ! mainwp_rocket_direct_filesystem()->is_readable( $file ) ) {
        return false;
    }

    return $file;
}

/**
 * Simple helper to get some external URLs.
 *
 * @since  2.10.10
 * @author Grégory Viguier
 *
 * @param  string $target     What we want.
 * @param  array  $query_args An array of query arguments.
 * @return string The URL.
 */
function mainwp_rocket_get_external_url( $target, $query_args = array() ) {

    if ( defined( 'WP_ROCKET_WEB_MAIN' ) ) {
        $site_url = WP_ROCKET_WEB_MAIN;
    } else {
        $site_url = 'https://wp-rocket.me/';
    }

    switch ( $target ) {
        case 'support':
            $locale = function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
            $paths  = array(
                'default' => 'support',
                'fr_FR'   => 'fr/support',
                'fr_CA'   => 'fr/support',
                'it_IT'   => 'it/supporto',
                'de_DE'   => 'de/support',
                'es_ES'   => 'es/soporte',
            );

            $url = isset( $paths[ $locale ] ) ? $paths[ $locale ] : $paths['default'];
            $url = $site_url . $url . '/';
            break;
        case 'account':
            $locale = function_exists( 'get_user_locale' ) ? get_user_locale() : get_locale();
            $paths  = array(
                'default' => 'account',
                'fr_FR'   => 'fr/compte',
                'fr_CA'   => 'fr/compte',
                'it_IT'   => 'it/account/',
                'de_DE'   => 'de/konto/',
                'es_ES'   => 'es/cuenta/',
            );

            $url = isset( $paths[ $locale ] ) ? $paths[ $locale ] : $paths['default'];
            $url = $site_url . $url . '/';
            break;
        default:
            $url = $site_url;
    }

    if ( $query_args ) {
        $url = add_query_arg( $query_args, $url );
    }

    return $url;
}

/**
 * Get all active languages URI.
 *
 * @since 2.0
 *
 * @return array $urls List of all active languages URI.
 */
function mainwp_get_rocket_i18n_uri() { // phpcs:ignore -- NOSONAR - Current complexity is the only way to achieve desired results, pull request solutions appreciated.
    $i18n_plugin = mainwp_rocket_has_i18n();
    $urls        = array();

    if ( 'wpml' === $i18n_plugin ) {
        // WPML.
        foreach ( mainwp_get_rocket_i18n_code() as $lang ) {
            $urls[] = $GLOBALS['sitepress']->language_url( $lang );
        }
    } elseif ( 'qtranslate' === $i18n_plugin || 'qtranslate-x' === $i18n_plugin ) {
        // qTranslate, qTranslate-x.
        foreach ( mainwp_get_rocket_i18n_code() as $lang ) {
            if ( 'qtranslate' === $i18n_plugin ) {
                $urls[] = qtrans_convertURL( home_url(), $lang, true );
            } else {
                $urls[] = qtranxf_convertURL( home_url(), $lang, true );
            }
        }
    } elseif ( 'polylang' === $i18n_plugin ) {
        // Polylang, Polylang Pro.
        $pll = function_exists( 'PLL' ) ? PLL() : $GLOBALS['polylang'];

        if ( ! empty( $pll ) && is_object( $pll ) ) {
            if ( ! defined( 'POLYLANG_VERSION' ) || version_compare( POLYLANG_VERSION, '3.4', '<' ) ) {
                $urls = wp_list_pluck( $pll->model->get_languages_list(), 'search_url' );
            } else {
                $languages = $pll->model->get_languages_list();
                foreach ( $languages as $language ) {
                    $urls[] = $language->get_home_url();
                }
            }
        }
    }

    /**
     * Filters the value of all active languages URI
     *
     * @param array Array of active languages URI.
     */
    $urls = apply_filters( 'rocket_get_i18n_uri', $urls );

    if (
    ! is_array( $urls )
    ||
    empty( $urls )
    ) {
        $urls[] = home_url();
    }

    return $urls;
}
/**
 * Get infos of all active languages.
 *
 * @since 2.0
 *
 * @return array A list of language codes.
 */
function mainwp_get_rocket_i18n_code() { // phpcs:ignore -- NOSONAR - Current complexity is the only way to achieve desired results, pull request solutions appreciated.
    $i18n_plugin = mainwp_rocket_has_i18n();

    if ( ! $i18n_plugin ) {
        return array();
    }

    if ( 'wpml' === $i18n_plugin ) {
        // WPML.
        return array_keys( $GLOBALS['sitepress']->get_active_languages() );
    }

    if ( 'qtranslate' === $i18n_plugin || 'qtranslate-x' === $i18n_plugin ) {
        // qTranslate, qTranslate-x.
        return ! empty( $GLOBALS['q_config']['enabled_languages'] ) ? $GLOBALS['q_config']['enabled_languages'] : array();
    }

    if ( 'polylang' === $i18n_plugin ) {
        // Polylang, Polylang Pro.
        return pll_languages_list();
    }

    $codes   = array();
    $default = $codes;

    /**
     * Filters the active languages codes list
     *
     * @param array $codes Array of languages codes.
     */
    $codes = apply_filters( 'rocket_get_i18n_code', $codes );

    if ( ! is_array( $codes ) ) {
        $codes = $default;
    }

    return $codes;
}

/**
 * Tell if a translation plugin is activated.
 *
 * @since 2.0
 * @since 3.2.1 Return an identifier on success instead of true.
 *
 * @return string An identifier corresponding to the active plugin.
 */
function mainwp_rocket_has_i18n() { // phpcs:ignore -- NOSONAR - Current complexity is the only way to achieve desired results, pull request solutions appreciated.
    global $sitepress, $q_config, $polylang;

    if ( ! empty( $sitepress ) && is_object( $sitepress ) && method_exists( $sitepress, 'get_active_languages' ) ) {
        // WPML.
        return 'wpml';
    }

    if ( ! empty( $polylang ) && function_exists( 'pll_languages_list' ) ) {
        $languages = pll_languages_list();

        if ( empty( $languages ) ) {
            return '';
        }

        // Polylang, Polylang Pro.
        return 'polylang';
    }

    if ( ! empty( $q_config ) && is_array( $q_config ) ) {
        if ( function_exists( 'qtranxf_convertURL' ) ) {
            // qTranslate-x.
            return 'qtranslate-x';
        }

        if ( function_exists( 'qtrans_convertURL' ) ) {
            // qTranslate.
            return 'qtranslate';
        }
    }

    $identifier = '';
    $default    = $identifier;

    /**
     * Filters the value of i18n plugin detection
     *
     * @param string $identifier An identifier value.
     */
    $identifier = apply_filters( 'rocket_has_i18n', $identifier );

    if ( ! is_string( $identifier ) ) {
        $identifier = $default;
    }

    return $identifier;
}
