<?php

namespace Salesloo\Abstracts;

use WP_Error;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

abstract class Email
{
    public $target;

    public $event;

    public $target_title;

    public $event_title;

    protected $_to;
    protected $_subject;
    protected $_body;

    protected function shortcodes()
    {
        return array();
    }

    protected function field_before()
    {
        return;
    }

    protected function field_after()
    {
        return;
    }

    /**
     * fields
     *
     * @return void
     */
    public function fields()
    {
        $shortcodes = [
            'site_name'        => get_bloginfo('name'),
        ];

        $shortcodes = wp_parse_args($this->shortcodes(), $shortcodes);
        $shortcodes = apply_filters('salesloo/email/' . $this->target . '/' . $this->event . '/shortcodes', $shortcodes);

        ob_start();
        foreach ($shortcodes as $tag => $note) {
            echo '<strong>[' . $tag . ']</strong> => ' . $note . '<br/>';
        }
        $shortcodes = ob_get_clean();

        \salesloo_field_heading([
            'label' => __('Shortcodes', 'salesloo'),
            'description' => $shortcodes,
        ]);

        $this->field_before();

        \salesloo_field_textarea([
            'label'       => __('Subject', 'salesloo'),
            'name'        => 'salesloo_notification_email_' . $this->target . '_' . $this->event . '_subject',
            'value'       => salesloo_get_option('notification_email_' . $this->target . '_' . $this->event . '_subject', $this->default_subject()),
            'direction'   => 'column',
            'style'       => 'width: 100%'
        ]);

        salesloo_field_editor([
            'label'       => __('Body', 'salesloo'),
            'name'        => 'salesloo_notification_email_' . $this->target . '_' . $this->event . '_body',
            'description' => '',
            'value'       => salesloo_get_option('notification_email_' . $this->target . '_' . $this->event . '_body', $this->default_body()),
            'direction'   => 'column'
        ]);

        $this->field_after();
    }

    /**
     * default_subject
     *
     * @return void
     */
    abstract protected function default_subject();

    /**
     * default_body
     *
     * @return void
     */
    abstract protected function default_body();

    /**
     * admins preview
     *
     * @return void
     */
    abstract function preview();

    /**
     * set recipient
     *
     * @param  mixed $to
     * @return void
     */
    public function to($to)
    {
        if (is_array($to)) {
            $this->_to = $to;
        } else {
            $this->_to = array($to);
        }
        return $this;
    }

    protected function build_subject($args = [])
    {
        /**
         * filter argument
         */
        $args = apply_filters('salesloo/email/' . $this->target . '/' . $this->event . '/shortcodes/args', $args);

        /**
         * get email subject
         */
        $subject = salesloo_get_option('notification_email_' . $this->target . '_' . $this->event . '_subject', $this->default_subject());

        /**
         * replace shortcode on subject
         */
        $subject = salesloo_email_replace_shortcode($subject, $args);

        /**
         * filter subject
         */
        $this->_subject = apply_filters('salesloo/email/' . $this->target . '/' . $this->event . '/subject', $subject);
    }

    protected function build_body($args)
    {
        /**
         * filter argument
         */
        $args = apply_filters('salesloo/email/' . $this->target . '/' . $this->event . '/shortcodes/args', $args);

        /**
         * get body content
         */
        $body = salesloo_get_option('notification_email_' . $this->target . '_' . $this->event . '_body', $this->default_body());

        /**
         * replace shortcode from body
         */
        $body = salesloo_email_replace_shortcode($body, $args);

        /**
         * filter body
         */
        $body = apply_filters('salesloo/email/' . $this->target . '/' . $this->event . '/body', $body);

        /**
         * apply shortcode from wp
         */
        $body = wp_kses_post(wpautop(do_shortcode($body)));

        $this->_body = salesloo_email_layout($body);
    }

    /**
     * send email
     *
     * @return void
     */
    public function send()
    {
        if (count($this->_to) === 0) {
            return new WP_Error('error', 'You must set at least 1 recipient');
        }

        if (empty($this->_subject)) {
            return new WP_Error('error', 'Subject must defined');
        }

        if (empty($this->_body)) {
            return new WP_Error('error', 'body must defined');
        }

        $headers = [
            sprintf("From: %s <%s>", \salesloo_get_option('notification_email_from_name'), \salesloo_get_option('notification_email_from_email'))
        ];

        if (\salesloo_get_option('notification_email_reply_to_name') && \salesloo_get_option('notification_email_reply_to_email')) {
            $headers[] = sprintf("Reply-To: %s <%s>", \salesloo_get_option('notification_email_reply_to_name'), \salesloo_get_option('notification_email_reply_to_email'));
        }

        add_filter('wp_mail_content_type', function ($content_type) {
            return 'text/html';
        });

        return wp_mail($this->_to, $this->_subject, $this->_body, $headers, array());
    }
}
