<?php

namespace Salesloo;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Salesloo plugin.
 *
 * The main plugin handler class is responsible for initializing Salesloo. The
 * class registers and all the components required to run the plugin.
 *
 * @since 1.0.0
 */
class Plugin
{

    /**
     * Instance.
     *
     * Holds the plugin instance.
     *
     * @since 1.0.0
     * @access public
     */
    public static $instance = null;

    /**
     * License
     * 
     * Holds the plugin license
     * 
     * @since 1.0.0
     * @access public
     */
    public $license;

    public $admin_ajax;

    public $affiliate;

    public $coupon;

    public $payment_method;

    public $system_info;

    /**
     * Clone.
     *
     * Disable class cloning and throw an error on object clone.
     *
     * @access public
     * @since 1.0.0
     */
    public function __clone()
    {
        // Cloning instances of the class is forbidden.
        _doing_it_wrong(__FUNCTION__, esc_html__('Something went wrong.', 'salesloo'), '1.0.0');
    }

    /**
     * Wakeup.
     *
     * Disable unserializing of the class.
     *
     * @access public
     * @since 1.0.0
     */
    public function __wakeup()
    {
        // Unserializing instances of the class is forbidden.
        _doing_it_wrong(__FUNCTION__, esc_html__('Something went wrong.', 'salesloo'), '1.0.0');
    }

    /**
     * Instance.
     *
     * Ensures only one instance of the plugin class is loaded or can be loaded.
     *
     * @since 1.0.0
     * @access public
     */
    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();

            /**
             * Salesloo loaded 
             * 
             * Fires when Salesloo was fully loaded and instantiated.
             *
             * @since 1.0.0
             */
            $timezone = get_option('timezone_string');
            if (!empty($timezone)) {
                date_default_timezone_set($timezone);
            }
            do_action('salesloo/loaded');
        }

        return self::$instance;
    }

    /**
     * Init.
     *
     * Initialize Salesloo Plugin. 
     *
     * @since 1.0.0
     */
    public function init()
    {

        $this->init_components();

        /**
         * Salesloo init.
         *
         * @since 1.0.0
         */
        do_action('salesloo/init');
    }

    /**
     * Init components.
     *
     * Initialize Salesloo components.
     *
     * @since 1.0.0
     * @access private
     */
    private function init_components()
    {
        Files::init();
        Upgrade::init();
        //Webhooks::init();
        Page::init();
        Products::init();
        Rest_Api::init();
        Orders::init();
        Addon::init();
        Compatibility::init();
        $this->system_info = new System_Info();
        new Notifications;
        $this->license        = new License;
        $this->admin_ajax     = new Admin_Ajax();
        $this->affiliate      = new Affiliate();
        $this->coupon         = new Coupon();
        $this->payment_method = new Payment_Method();
        new Events;

        if (is_admin()) {
            Admin::init();
            new Settings();
        }

        Frontend::init();

        $this->license->periodic_check();
    }

    /**
     * Register autoloader.
     *
     * Salesloo autoloader loads all the classes needed to run the plugin.
     *
     * @since 1.0.0
     * @access private
     */
    private function register_autoloader()
    {
        require_once SALESLOO_PATH . '/includes/autoloader.php';

        Autoloader::run();
    }

    /**
     * Plugin constructor.
     *
     * @since 1.0.0
     * @access private
     */
    private function __construct()
    {
        $this->register_autoloader();

        add_action('init', [$this, 'init'], 0);
    }
}

// Run the instance.
Plugin::instance();
