<?php

namespace Salesloo;

/**
 * system info
 */
class System_Info
{
    /**
     * Register Menu
     * 
     * register system info menu
     */
    public function register_menu($submenu)
    {
        $submenu[] = [
            'page_title' => __('System Info', 'salesloo'),
            'menu_title' => __('System Info', 'salesloo'),
            'capability' => 'manage_options',
            'slug'       => 'salesloo-system-info',
            'callback'   => [$this, 'show_page'],
            'position'   => 99
        ];

        return $submenu;
    }

    /**
     * Get web server name
     * 
     * Get of current web server name
     * 
     * @since 1.0.0
     * @access public
     */
    public function get_web_server_name()
    {
        return [
            'value' => $_SERVER['SERVER_SOFTWARE'],
        ];
    }

    /**
     * Get php version
     * 
     * @since 1.0.0
     * @access public
     */
    public function get_php_version()
    {
        $result = [
            'value' => PHP_VERSION,
        ];

        if (version_compare($result['value'], '7.3', '<')) {
            $result['message'] = __('We recommend to use php 7.3 or higher', 'salesloo');

            $result['notice'] = 'warning';
        }

        return $result;
    }

    /**
     * Get PHP memory limit.
     *
     * Retrieve the value of php memory limit from `php.ini` configuration file.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_php_memory_limit()
    {
        $phpmemory = @ini_get('memory_limit');
        $phpmemory_num = str_replace('M', '', $phpmemory);

        $result = [
            'value' => $phpmemory_num,
        ];

        if (version_compare($result['value'], '64', '<')) {
            $result['message'] = __('We recommend to increase php memory limit to 64 or higher', 'salesloo');

            $result['notice'] = 'danger';
        }

        return $result;
    }

    /**
     * Get PHP `max_input_vars`.
     *
     * Retrieve the value of `max_input_vars` from `php.ini` configuration file.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_php_max_input_vars()
    {
        return [
            'value' => ini_get('max_input_vars'),
        ];
    }

    /**
     * Get PHP `post_max_size`.
     *
     * Retrieve the value of `post_max_size` from `php.ini` configuration file.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_php_max_post_size()
    {
        return [
            'value' => ini_get('post_max_size'),
        ];
    }

    /**
     * Get GD installed.
     *
     * Whether the GD extension is installed.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_gd_installed()
    {
        $gd_installed = extension_loaded('gd');

        $result = [
            'value' => $gd_installed ? 'Yes' : 'No'
        ];

        if (!$gd_installed) {
            $result['message'] = __('Please Install or Enable GD extension !', 'salesloo');

            $result['notice'] = 'danger';
        }

        return $result;
    }

    /**
     * Get ZIP installed.
     *
     * Whether the ZIP extension is installed.
     * 
     * @since 1.0.0
     * @access public
     */
    public function get_zip_installed()
    {
        $zip_installed = extension_loaded('zip');

        $result = [
            'value' => $zip_installed ? 'Yes' : 'No',
        ];

        if (!$zip_installed) {
            $result['message'] = __('Please Install or Enable ZIP extension !', 'salesloo');

            $result['notice'] = 'danger';
        }

        return $result;
    }

    /**
     * Get Curl installed.
     *
     * Whether the ZIP extension is installed.
     * 
     * @since 1.0.0
     * @access public
     */
    public function get_curl_installed()
    {
        $result = [
            'value' => function_exists("curl_init") ? 'Yes' : 'No',
        ];

        if (!function_exists("curl_init")) {
            $result['message'] = __('Please Install or Enable CURL extension !', 'salesloo');

            $result['notice'] = 'danger';
        }

        return $result;
    }

    /**
     * Get WordPress version.
     *
     * Retrieve the WordPress version.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_wp_version()
    {
        return [
            'value' => get_bloginfo('version'),
        ];
    }

    /**
     * Get PHP `max_upload_size`.
     *
     * Retrieve the value of maximum upload file size defined in `php.ini` configuration file.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_wp_max_upload_size()
    {
        return [
            'value' => size_format(wp_max_upload_size()),
        ];
    }

    /**
     * Get WordPress timezone.
     *
     * Retrieve WordPress timezone.
     *
     * @since 1.0.0
     * @access public
     */
    public function get_timezone()
    {
        $timezone = get_option('timezone_string');
        if (!$timezone) {
            $timezone = get_option('gmt_offset');
        }

        return [
            'value' => $timezone,
        ];
    }

    /**
     * Get debug mode.
     *
     * Whether WordPress debug mode is enabled or not.
     *
     * @since 1.0.0
     * @access public
     *
     * @return array {
     *    Report data.
     *
     *    @type string $value Active if debug mode is enabled, Inactive otherwise.
     * }
     */
    public function get_wp_debug_mode()
    {
        return [
            'value' => WP_DEBUG ? 'Active' : 'Inactive',
        ];
    }

    /**
     * Menu Page
     * 
     * Output system info page
     */
    public function show_page()
    {
        $server_environment = [
            [
                'label' => 'Web Server',
                'value' => $this->get_web_server_name(),
            ],
            [
                'label' => 'PHP Version',
                'value' => $this->get_php_version(),
            ],
            [
                'label' => 'PHP Memory Limit',
                'value' => $this->get_php_memory_limit(),
            ],
            [
                'label' => 'PHP Max Input Vars',
                'value' => $this->get_php_max_input_vars(),
            ],
            [
                'label' => 'PHP Max Post Size',
                'value' => $this->get_php_max_post_size(),
            ],
            [
                'label' => 'GD Extension Installed',
                'value' => $this->get_gd_installed(),
            ],
            [
                'label' => 'ZIP Extension Installed',
                'value' => $this->get_zip_installed(),
            ],
            [
                'label' => 'CURL Extension Installed',
                'value' => $this->get_curl_installed(),
            ],
            [
                'label' => 'WordPress Versin',
                'value' => $this->get_wp_version(),
            ],
            [
                'label' => 'WordPress max Upload size',
                'value' => $this->get_wp_max_upload_size(),
            ],
            [
                'label' => 'TimeZone',
                'value' => $this->get_timezone(),
            ],
            [
                'label' => 'Debug Mode',
                'value' => $this->get_wp_debug_mode(),
            ],
        ];

        echo '<div class="wrap">';
        echo '<h2>' . __('System Info', 'salesloo') . '</h2>';
        echo '<div>';
        echo '<table class="widefat">';
        echo '<thead>
                    <tr>
                        <th>Website Environment</th>
                        <th></th>
                        <th style="min-width: 200px"></th>
                    </tr>
                </thead>';
        echo '<tbody>';
        foreach ($server_environment as $s) {
            $message = isset($s['value']['message']) ? $s['value']['message'] : '';
            $style = '';

            if (isset($s['value']['notice'])) {
                if ($s['value']['notice'] == 'warning') {
                    $style = ' style="color:yellow"';
                } else if ($s['value']['notice'] == 'danger') {
                    $style = ' style="color:red"';
                }
            }
            echo '<tr>
                        <td>' . $s['label'] . '</td>
                        <td' . $style . '>' . $s['value']['value'] . '</td>
                        <td' . $style . '>' . $message . '</td>
                    </tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * System Info Constructor
     */
    public function __construct()
    {
        add_filter('salesloo/admin/submenu', [$this, 'register_menu'], 1, 10);
    }
}
