<?php
/**
 * Password Markup Class file.
 *
 * @package sureforms-pro.
 * @since 1.8.0
 */

namespace SRFM_Pro\Inc\Business\User_Registration\Password;

use SRFM\Inc\Helper;
use SRFM_Pro\Inc\Fields\Base;
use SRFM_Pro\Inc\Helper as Pro_Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Password_Markup Class.
 *
 * @since 1.8.0
 */
class Password_Markup extends Base {
	/**
	 * Flag indicating whether confirmation password is required.
	 *
	 * @var bool
	 * @since 1.8.0
	 */
	protected $is_confirm_password;

	/**
	 * Label for the input field of the confirmation password.
	 *
	 * @var string
	 * @since 1.8.0
	 */
	protected $input_confirm_label;

	/**
	 * Unique slug for the confirmation password field.
	 *
	 * @var string
	 * @since 1.8.0
	 */
	protected $unique_confirm_slug;

	/**
	 * Check password strength flag.
	 *
	 * @var bool
	 * @since 1.8.0
	 */
	protected $check_password_strength;

	/**
	 * Check password visibility flag.
	 *
	 * @var bool
	 * @since 1.8.0
	 */
	protected $toggle_password_visibility;

	/**
	 * Retains a copy of Confirmation Password input label.
	 *
	 * @var string
	 * @since 1.8.0
	 */
	protected $confirm_label;

	/**
	 * Initialize the properties based on block attributes.
	 *
	 * @param array<mixed> $attributes Block attributes.
	 * @since 1.8.0
	 */
	public function __construct( $attributes ) {
		$this->set_properties( $attributes );
		$this->set_error_msg( 'srfm_password_block_required_text', $attributes );
		$this->set_input_label( __( 'Password', 'sureforms-pro' ) );
		$this->slug                = 'password';
		$this->is_confirm_password = $attributes['isConfirmPassword'] ?? false;
		$this->placeholder_attr    = $this->placeholder ? ' placeholder="' . esc_attr( $this->placeholder ) . '" ' : '';

		// translators: Placeholder %s represents the label being confirmed.
		$this->input_confirm_label        = '-lbl-' . Helper::encrypt( sprintf( __( 'Confirm %s', 'sureforms-pro' ), $this->input_label_fallback ) );
		$this->unique_confirm_slug        = 'srfm-' . $this->slug . '-confirm-' . $this->block_id . $this->input_confirm_label;
		$this->check_password_strength    = $attributes['checkPasswordStrength'] ?? false;
		$this->toggle_password_visibility = $attributes['togglePasswordVisibility'] ?? false;
		// translators: Placeholder %s represents the label being confirmed.
		$this->confirm_label = ! empty( $attributes['confirmLabel'] ) ? sanitize_text_field( $attributes['confirmLabel'] ) : sprintf( __( 'Confirm %s', 'sureforms-pro' ), $this->label );
		$this->set_unique_slug();
		$this->set_field_name( $this->unique_slug );
		$this->set_markup_properties( $this->input_label, true );
		$this->set_aria_described_by();
		$this->set_label_as_placeholder( $this->label );
	}

	/**
	 * Render HTML markup of a block.
	 * This function renders the markup of the password input block.
	 *
	 * @since 1.8.0
	 * @return string|bool
	 */
	public function markup() {
		$classes = $this->get_field_classes();
		ob_start(); ?>
		<div
			data-block-id="<?php echo esc_attr( $this->block_id ); ?>"
			class="<?php echo esc_attr( $classes ); ?>"
			<?php if ( $this->check_password_strength ) { ?>
				data-password-strength-check=<?php echo esc_attr( Helper::get_string_value( $this->check_password_strength ) ); ?>
			<?php } ?>
			<?php if ( $this->toggle_password_visibility ) { ?>
				data-password-visibility=<?php echo esc_attr( Helper::get_string_value( $this->toggle_password_visibility ) ); ?>
			<?php } ?>
			<?php if ( $this->is_confirm_password ) { ?>
				data-confirm-password=<?php echo esc_attr( Helper::get_string_value( $this->is_confirm_password ) ); ?>
			<?php } ?>
		>
			<div class="<?php echo esc_attr( Helper::join_strings( [ "srfm-{$this->slug}-block", "srfm-{$this->slug}-{$this->block_id}-block" ] ) ); ?>">
				<?php echo wp_kses_post( $this->label_markup ); ?>
				<?php echo wp_kses_post( $this->help_markup ); ?>
				<div class="srfm-block-wrap srfm-with-icon">
					<input class="<?php echo esc_attr( Helper::join_strings( [ 'srfm-input-common', "srfm-input-{$this->slug}" ] ) ); ?>" type="password" id="<?php echo esc_attr( $this->unique_slug ); ?>"
					<?php echo ! empty( $this->aria_described_by ) ? "aria-describedby='" . esc_attr( trim( $this->aria_described_by ) ) . "'" : ''; ?>
					aria-required="<?php echo esc_attr( $this->data_require_attr ); ?>" data-required="<?php echo esc_attr( $this->data_require_attr ); ?>" <?php echo wp_kses_post( $this->placeholder_attr ); ?> name="<?php echo esc_attr( $this->field_name ); ?>" />
					<?php
					if ( $this->toggle_password_visibility ) {
						echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
							'showPassword',
							'srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
							' role="button" tabindex="0" id="srfm-show-password" aria-label="' . esc_attr__( 'Show password', 'sureforms-pro' ) . '"',
							'span'
						);
						echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
							'hidePassword',
							'srfm-hide srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
							' role="button" tabindex="0" id="srfm-hide-password" aria-label="' . esc_attr__( 'Hide password', 'sureforms-pro' ) . '"',
							'span'
						);
					}
					?>
				</div>
				<div class="srfm-error-wrap">
					<?php echo wp_kses_post( $this->error_msg_markup ); ?>
				</div>
			</div>
		<?php
		if ( true === $this->is_confirm_password ) {
			$confirm_label_markup = Helper::generate_common_form_markup( $this->form_id, 'label', $this->confirm_label, $this->slug . '-confirm', $this->block_id . $this->input_confirm_label, boolval( $this->required ) );
			$this->set_label_as_placeholder( $this->confirm_label );
			if ( ! empty( $this->placeholder_attr ) ) {
				$confirm_label_markup = '';
			}
			?>
			<div class="srfm-<?php echo esc_attr( $this->slug ); ?>-confirm-block srfm-<?php echo esc_attr( $this->slug ); ?>-<?php echo esc_attr( $this->block_id ); ?>-confirm-block">
				<?php echo wp_kses_post( $confirm_label_markup ); ?>
				<div class="srfm-block-wrap srfm-with-icon">
					<input class="srfm-input-common srfm-input-<?php echo esc_attr( $this->slug ); ?>-confirm" type="password" id="<?php echo esc_attr( $this->unique_confirm_slug ); ?>"
					<?php echo ! empty( $this->aria_described_by ) ? "aria-describedby='" . esc_attr( trim( $this->aria_described_by ) ) . "'" : ''; ?>
					aria-required="<?php echo esc_attr( $this->data_require_attr ); ?>" data-required="<?php echo esc_attr( $this->data_require_attr ); ?>" <?php echo wp_kses_post( $this->placeholder_attr ); ?> />
					<?php
					if ( $this->toggle_password_visibility ) {
						echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
							'showPassword',
							'srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
							' role="button" tabindex="0" id="srfm-show-confirm-password" aria-label="' . esc_attr__( 'Show confirm password', 'sureforms-pro' ) . '"',
							'span'
						);
						echo Pro_Helper::fetch_pro_svg( //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Ignored to render svg
							'hidePassword',
							'srfm-hide srfm-' . esc_attr( $this->slug ) . '-icon srfm-input-icon',
							' role="button" tabindex="0" id="srfm-hide-confirm-password" aria-label="' . esc_attr__( 'Hide confirm password', 'sureforms-pro' ) . '"',
							'span'
						);
					}
					?>
				</div>
				<div class="srfm-error-wrap">
					<?php echo wp_kses_post( $this->error_msg_markup ); ?>
				</div>
			</div>
		<?php } ?>
		</div>
		<?php

		$markup = ob_get_clean();

		return apply_filters(
			'srfm_block_field_markup',
			$markup,
			[
				'slug'       => $this->slug,
				'is_editing' => $this->is_editing,
				'field_name' => $this->field_name,
				'attributes' => $this->attributes,
			]
		);
	}
}
