<?php

/**
 * JSON Web Token implementation, based on this spec:
 * http://tools.ietf.org/html/draft-ietf-oauth-json-web-token-06
 *
 * PHP version 5
 *
 * @category Authentication
 * @package  Authentication_JWT
 * @author   Neuman Vong <neuman@twilio.com>
 * @author   Anant Narayanan <anant@php.net>
 * @license  http://opensource.org/licenses/BSD-3-Clause 3-clause BSD
 * @link     https://github.com/firebase/php-jwt
 */
class JWT
{

    public static $methods = [
        'HS256' => [
            'hash_hmac',
            'SHA256',
        ],
        'HS512' => [
            'hash_hmac',
            'SHA512',
        ],
        'HS384' => [
            'hash_hmac',
            'SHA384',
        ],
        'RS256' => [
            'openssl',
            'SHA256',
        ],
    ];


    /**
     * Decodes a JWT string into a PHP object.
     *
     * @param string            $jwt    The JWT
     * @param string|array|null $key    The secret key, or map of keys
     * @param boolean           $verify Don't skip verification process
     *
     * @return object      The JWT's payload as a PHP object
     *
     * @throws DomainException              Algorithm was not provided
     * @throws UnexpectedValueException     Provided JWT was invalid
     * @throws SignatureInvalidException    Provided JWT was invalid because the signature verification failed
     * @throws BeforeValidException         Provided JWT is trying to be used before it's eligible as defined by 'nbf'
     * @throws BeforeValidException         Provided JWT is trying to be used before it's been created as defined by 'iat'
     * @throws ExpiredException             Provided JWT has since expired, as defined by the 'exp' claim
     *
     * @uses jsonDecode
     * @uses urlsafeB64Decode
     */
    public static function decode($jwt, $key=null, $verify=true)
    {
        $tks = explode('.', $jwt);
        if (count($tks) != 3) {
            throw new UnexpectedValueException('Wrong number of segments');
        }

        list( $headb64, $bodyb64, $cryptob64 ) = $tks;
        if (null === ($header = self::jsonDecode(self::urlsafeB64Decode($headb64)))) {
            throw new UnexpectedValueException('Invalid header encoding');
        }

        if (null === $payload = self::jsonDecode(self::urlsafeB64Decode($bodyb64))) {
            throw new UnexpectedValueException('Invalid claims encoding');
        }

        $sig = self::urlsafeB64Decode($cryptob64);
        if ($verify) {
            if (empty($header->alg)) {
                throw new DomainException('Empty algorithm');
            }

            if (is_array($key)) {
                if (isset($header->kid)) {
                    $key = $key[$header->kid];
                } else {
                    throw new DomainException('"kid" empty, unable to lookup correct key');
                }
            }

            // Check the signature
            if (! self::verify("$headb64.$bodyb64", $sig, $key, $header->alg)) {
                throw new SignatureInvalidException('Signature verification failed');
            }

            // Check if the nbf if it is defined. This is the time that the
            // token can actually be used. If it's not yet that time, abort.
            if (isset($payload->nbf) && $payload->nbf > time()) {
                throw new BeforeValidException(
                    'Cannot handle token prior to '.date(DateTime::ISO8601, $payload->nbf)
                );
            }

            // Check that this token has been created before 'now'. This prevents
            // using tokens that have been created for later use (and haven't
            // correctly used the nbf claim).
            if (isset($payload->iat) && $payload->iat > time()) {
                throw new BeforeValidException(
                    'Cannot handle token prior to '.date(DateTime::ISO8601, $payload->iat)
                );
            }

            // Check if this token has expired.
            if (isset($payload->exp) && time() >= $payload->exp) {
                throw new ExpiredException('Expired token');
            }
        }//end if

        return $payload;

    }//end decode()


    /**
     * Converts and signs a PHP object or array into a JWT string.
     *
     * @param object|array $payload PHP object or array
     * @param string       $key     The secret key
     * @param string       $algo    The signing algorithm. Supported
     *                              algorithms are 'HS256', 'HS384' and 'HS512'
     *
     * @return string      A signed JWT
     * @uses   jsonEncode
     * @uses   urlsafeB64Encode
     */
    public static function encode($payload, $key, $algo='HS256', $keyId=null)
    {
        $header = [
            'typ' => 'JWT',
            'alg' => $algo,
        ];
        if ($keyId !== null) {
            $header['kid'] = $keyId;
        }

        $segments      = [];
        $segments[]    = self::urlsafeB64Encode(self::jsonEncode($header));
        $segments[]    = self::urlsafeB64Encode(self::jsonEncode($payload));
        $signing_input = implode('.', $segments);
        $signature     = self::sign($signing_input, $key, $algo);
        $segments[]    = self::urlsafeB64Encode($signature);
        return implode('.', $segments);

    }//end encode()


    /**
     * Sign a string with a given key and algorithm.
     *
     * @param string          $msg    The message to sign
     * @param string|resource $key    The secret key
     * @param string          $method The signing algorithm. Supported algorithms
     *                                are 'HS256', 'HS384', 'HS512' and 'RS256'
     *
     * @return string          An encrypted message
     * @throws DomainException Unsupported algorithm was specified
     */
    public static function sign($msg, $key, $method='HS256')
    {
        if (empty(self::$methods[$method])) {
            throw new DomainException('Algorithm not supported');
        }

        list( $function, $algo ) = self::$methods[$method];
        switch ($function) {
            case 'hash_hmac':
                return hash_hmac($algo, $msg, $key, true);

            case 'openssl':
                $signature = '';
                $success   = openssl_sign($msg, $signature, $key, $algo);
                if (! $success) {
                    throw new DomainException('OpenSSL unable to sign data');
                } else {
                    return $signature;
                }
        }

    }//end sign()


    /**
     * Verify a signature with the mesage, key and method. Not all methods
     * are symmetric, so we must have a separate verify and sign method.
     *
     * @param string          $msg       the original message
     * @param string          $signature
     * @param string|resource $key       for HS*, a string key works. for RS*, must be a resource of an openssl public key
     * @param string          $method
     *
     * @return boolean
     * @throws DomainException Invalid Algorithm or OpenSSL failure
     */
    public static function verify($msg, $signature, $key, $method='HS256')
    {
        if (empty(self::$methods[$method])) {
            throw new DomainException('Algorithm not supported');
        }

        list( $function, $algo ) = self::$methods[$method];
        switch ($function) {
            case 'openssl':
                $success = openssl_verify($msg, $signature, $key, $algo);
                if (! $success) {
                    throw new DomainException('OpenSSL unable to verify data: '.openssl_error_string());
                } else {
                    return $signature;
                }

            case 'hash_hmac':
            default:
                $hash   = hash_hmac($algo, $msg, $key, true);
                $len    = min(strlen($signature), strlen($hash));
                $status = 0;
                for ($i = 0; $i < $len; $i++) {
                    $status |= (ord($signature[$i]) ^ ord($hash[$i]));
                }

                $status |= (strlen($signature) ^ strlen($hash));
                return ($status === 0);
        }//end switch

    }//end verify()


    /**
     * Decode a JSON string into a PHP object.
     *
     * @param string $input JSON string
     *
     * @return object          Object representation of JSON string
     * @throws DomainException Provided string was invalid JSON
     */
    public static function jsonDecode($input)
    {
        if (version_compare(PHP_VERSION, '5.4.0', '>=') && ! (defined('JSON_C_VERSION') && PHP_INT_SIZE > 4)) {
            /*
             * In PHP >=5.4.0, json_decode() accepts an options parameter, that allows you
             * to specify that large ints (like Steam Transaction IDs) should be treated as
             * strings, rather than the PHP default behaviour of converting them to floats.
             */
            $obj = json_decode($input, false, 512, JSON_BIGINT_AS_STRING);
        } else {
            /*
             * Not all servers will support that, however, so for older versions we must
             * manually detect large ints in the JSON string and quote them (thus converting
             *them to strings) before decoding, hence the preg_replace() call.
             */
            $max_int_length       = (strlen((string) PHP_INT_MAX) - 1);
            $json_without_bigints = preg_replace('/:\s*(-?\d{'.$max_int_length.',})/', ': "$1"', $input);
            $obj                  = json_decode($json_without_bigints);
        }

        if (function_exists('json_last_error') && $errno = json_last_error()) {
            self::handleJsonError($errno);
        } else if ($obj === null && $input !== 'null') {
            throw new DomainException('Null result with non-null input');
        }

        return $obj;

    }//end jsonDecode()


    /**
     * Encode a PHP object into a JSON string.
     *
     * @param object|array $input A PHP object or array
     *
     * @return string          JSON representation of the PHP object or array
     * @throws DomainException Provided object could not be encoded to valid JSON
     */
    public static function jsonEncode($input)
    {
        $json = json_encode($input);
        if (function_exists('json_last_error') && $errno = json_last_error()) {
            self::handleJsonError($errno);
        } else if ($json === 'null' && $input !== null) {
            throw new DomainException('Null result with non-null input');
        }

        return $json;

    }//end jsonEncode()


    /**
     * Decode a string with URL-safe Base64.
     *
     * @param string $input A Base64 encoded string
     *
     * @return string A decoded string
     */
    public static function urlsafeB64Decode($input)
    {
        $remainder = (strlen($input) % 4);
        if ($remainder) {
            $padlen = (4 - $remainder);
            $input .= str_repeat('=', $padlen);
        }

        return base64_decode(strtr($input, '-_', '+/'));

    }//end urlsafeB64Decode()


    /**
     * Encode a string with URL-safe Base64.
     *
     * @param string $input The string you want encoded
     *
     * @return string The base64 encode of what you passed in
     */
    public static function urlsafeB64Encode($input)
    {
        return str_replace('=', '', strtr(base64_encode($input), '+/', '-_'));

    }//end urlsafeB64Encode()


    /**
     * Helper method to create a JSON error.
     *
     * @param integer $errno An error number from json_last_error()
     *
     * @return void
     */
    private static function handleJsonError($errno)
    {
        $messages = [
            JSON_ERROR_DEPTH     => 'Maximum stack depth exceeded',
            JSON_ERROR_CTRL_CHAR => 'Unexpected control character found',
            JSON_ERROR_SYNTAX    => 'Syntax error, malformed JSON',
        ];
        throw new DomainException(
            isset($messages[$errno]) ? $messages[$errno] : 'Unknown JSON error: '.$errno
        );

    }//end handleJsonError()


}//end class
