<?php
/* Prohibit direct script loading */
defined('ABSPATH') || die('No direct script access allowed!');

use Joomunited\Queue\JuMainQueue;

/**
 * Class WpmfAddonOneDriveAdmin
 * This class that holds most of the admin functionality for OneDrive
 */
class WpmfAddonOwncloudAdmin
{
    /**
     * Dav URL
     *
     * @var string
     */
    public $davUrl = '';
    /**
     * WpmfAddonOneDriveAdmin constructor.
     */
    public function __construct()
    {
        $configs = get_option('_wpmfAddon_owncloud_config');
        if (!empty($configs['username']) && !empty($configs['owncloudurl'])) {
            $this->davUrl = trim($configs['owncloudurl'], '/') . '/remote.php/webdav/';
        }
        add_action('admin_init', array($this, 'createRootFolder'));
        add_filter('wpmfaddon_owncloud_settings', array($this, 'renderSettings'), 10, 1);
        add_action('wp_ajax_wpmf_owncloud_add_queue', array($this, 'ajaxAddToQueue'));
        add_action('wpmfSyncOwnCloud', array($this, 'autoSyncWithCrontabMethod'));
        add_action('wpmf_create_folder', array($this, 'createFolderLibrary'), 10, 4);
        add_action('wpmf_before_delete_folder', array($this, 'deleteFolder'), 10, 1);
        add_action('wpmf_move_folder', array($this, 'moveFolder'), 10, 3);
        add_action('wpmf_update_folder_name', array($this, 'updateFolderName'), 10, 2);
        add_action('add_attachment', array($this, 'addAttachment'), 10, 1);
        add_action('wpmf_add_attachment', array($this, 'doAddAttachment'), 10, 2);
        add_action('wpmf_delete_attachment_cloud', array($this, 'deleteAttachment'), 10, 1);
        add_filter('wp_update_attachment_metadata', array($this, 'wpGenerateAttachmentMetadata'), 10, 2);
        add_action('wpmf_attachment_set_folder', array($this, 'moveFileLibrary'), 10, 3);
        add_filter('wpmf_owncloud_render_thumbnail', array($this, 'renderThumbnail'), 10, 3);
        add_filter('wpmf_remove_server_file', array($this, 'removeServerFile'), 10, 3);
        add_filter('wpmf_sync_owncloud', array($this, 'doSync'), 10, 3);
        add_filter('wpmf_owncloud_remove', array($this, 'syncRemoveItems'), 10, 3);
        add_filter('wpmf_owncloud_import', array($this, 'importFile'), 10, 5);
        add_action('wp_ajax_wpmf_owncloud_get_content', array($this, 'getContentFile'));
        add_action('wp_ajax_nopriv_wpmf_owncloud_get_content', array($this, 'getContentFile'));
        add_filter('wpmfAddonReplaceFileOwncloud', array($this, 'replaceFile'), 10, 2);
    }

    /**
     * Onedrive settings html
     *
     * @param string $html HTML
     *
     * @return string
     */
    public function renderSettings($html)
    {
        $owncloud_config = get_option('_wpmfAddon_owncloud_config');
        if (empty($owncloud_config)) {
            $owncloud_config = array('username' => '', 'password' => '', 'owncloudurl' => '', 'rootfoldername' => '');
        }

        if (!empty($_POST['OwnCloudUsername']) && !empty($_POST['OwnCloudPassword']) && !empty($_POST['OwnCloudUrl'])) {
            if (empty($_POST['wpmf_nonce'])
                || !wp_verify_nonce($_POST['wpmf_nonce'], 'wpmf_nonce')) {
                die();
            }

            $owncloud_config['username']     = trim($_POST['OwnCloudUsername']);
            $owncloud_config['password'] = trim($_POST['OwnCloudPassword']);
            $owncloud_config['owncloudurl'] = trim($_POST['OwnCloudUrl']);
            $new_root_foldername = $_POST['RootFolderName'];
            if (!empty($_POST['RootFolderName']) && $new_root_foldername !== $owncloud_config['rootfoldername']) {
                $owncloud_config['rootfoldername'] = $new_root_foldername;
                delete_option('wpmf_owncloud_create_root');
            }
            update_option('_wpmfAddon_owncloud_config', $owncloud_config);
            $owncloud_config = get_option('_wpmfAddon_owncloud_config');
        }
        $connect_owncloud = wpmfGetOption('connect_owncloud');
        ob_start();
        require_once 'templates/settings_owncloud.php';
        $html = ob_get_contents();
        ob_end_clean();
        return $html;
    }

    /**
     * Check connect
     *
     * @return boolean
     */
    public function isConnected()
    {
        $params = get_option('_wpmfAddon_owncloud_config');
        $connect_owncloud = wpmfGetOption('connect_owncloud');
        if (!empty($params['username']) && !empty($params['password']) && !empty($params['owncloudurl']) && !empty($params['rootfoldername']) && !empty($connect_owncloud)) {
            return true;
        }

        return false;
    }

    /**
     * Get folder or file id from path
     *
     * @param string $path Path
     *
     * @return boolean|integer
     */
    public function getFileIdByPath($path)
    {
        $valid_path = $this->getValidPath($path);
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
<d:propfind xmlns:d="DAV:" xmlns:oc="http://owncloud.org/ns">
  <d:prop>
    <d:getetag/>
    <oc:fileid/>
    <d:getlastmodified/>
  </d:prop>
</d:propfind>';
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($valid_path, '/') . '/');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PROPFIND');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: text/xml', 'Depth: 0'));
            curl_setopt($ch, CURLOPT_POSTFIELDS, $xml);
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $response = curl_exec($ch);
            curl_close($ch);
            $pattern = '/<oc:fileid ?.*>(.*)<\/oc:fileid>/';
            preg_match($pattern, $response, $matches);
            return (int)$matches[1];
        }

        return false;
    }

    /**
     * Get valid path
     *
     * @param string $path Path
     *
     * @return string
     */
    public function getValidPath($path)
    {
        $path = rawurldecode($path);
        $sub_paths = explode('/', $path);
        $valid_subpaths = array();
        foreach ($sub_paths as $sub_path) {
            $valid_subpaths[] = rawurlencode($sub_path);
        }

        $valid_path = implode('/', $valid_subpaths);
        return trim($valid_path);
    }

    /**
     * Get folder or file id from path
     *
     * @param string $path Path
     *
     * @return array
     */
    public function getChildsInFolder($path)
    {
        $valid_path = $this->getValidPath($path);
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
<d:propfind xmlns:d="DAV:" xmlns:oc="http://owncloud.org/ns" xmlns:nc="http://owncloud.org/ns">
  <d:prop>
  <oc:fileid />
  <oc:size />
  <d:getcontenttype />
  <d:resourcetype />
  </d:prop>
</d:propfind>';
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($valid_path, '/') . '/');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PROPFIND');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: text/xml', 'Depth: 1'));
            curl_setopt($ch, CURLOPT_POSTFIELDS, $xml);
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $response = curl_exec($ch);
            curl_close($ch);

            $response = str_replace('oc:', 'd:', $response);
            $items = $this->convertXmlToArray($response);
            if (!empty($items)) {
                return $items['response'];
            }
        }

        return array();
    }

    /**
     * Add root to queue
     *
     * @return void
     */
    public function createRootFolder()
    {
        if (!class_exists('\Joomunited\Queue\JuMainQueue')) {
            require_once WP_MEDIA_FOLDER_PLUGIN_DIR . 'queue/JuMainQueue.php';
        }

        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $rootfoldername = rawurlencode($params['rootfoldername']);
            // insert root folder on Media library
            $id = $this->createFolder($rootfoldername);
            $inserted = wp_insert_term('Owncloud', WPMF_TAXO, array('parent' => 0, 'slug' => 'owncloud'));
            if (is_wp_error($inserted)) {
                $folder_id = (int)$inserted->error_data['term_exists'];
            } else {
                $folder_id = (int)$inserted['term_id'];
            }
            update_term_meta($folder_id, 'wpmf_drive_type', 'owncloud');
            update_term_meta($folder_id, 'wpmf_drive_path', $rootfoldername);
            update_term_meta($folder_id, 'wpmf_drive_root_id', $id);
            add_option('wpmf_owncloud_create_root', 1, '', 'yes');

            if (!get_option('wpmf_create_owncloud_thumbs_folder', false)) {
                $this->createFolder($rootfoldername . '/wpmfthumbs');
                add_option('wpmf_create_owncloud_thumbs_folder', 1, '', 'yes');
            }
        }
    }

    /**
     * Sync folders and files with crontab method
     *
     * @return void
     */
    public function autoSyncWithCrontabMethod()
    {
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if (!$isConnected) {
            return;
        }
        if (!class_exists('\Joomunited\Queue\JuMainQueue')) {
            require_once WP_MEDIA_FOLDER_PLUGIN_DIR . 'queue/JuMainQueue.php';
        }
        $args = wpmfGetQueueOptions(true);
        $wpmfQueue = JuMainQueue::getInstance('wpmf');
        $wpmfQueue->init($args);
        $this->doAddToQueue($params);
        $wpmfQueue->proceedQueueAsync();
    }

    /**
     * Sync folders with media library
     *
     * @return void
     */
    public function ajaxAddToQueue()
    {
        if (empty($_POST['wpmf_nonce'])
            || !wp_verify_nonce($_POST['wpmf_nonce'], 'wpmf_nonce')) {
            die();
        }

        set_time_limit(0);
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if (!$isConnected) {
            wp_send_json(array('status' => false));
        }
        $this->doAddToQueue($params);
        wp_send_json(array('status' => true));
    }

    /**
     * Do add to queue
     *
     * @param array $params Configs details
     *
     * @return void
     */
    public function doAddToQueue($params)
    {
        $isConnected = $this->isConnected();
        if (!$isConnected) {
            return;
        }

        $id = $this->getFileIdByPath(trim($params['rootfoldername'], '/'));
        if (!empty($id)) {
            $datas = array(
                'id' => $id,
                'path' => trim($params['rootfoldername'], '/'),
                'folder_parent' => 0,
                'name' => 'Owncloud',
                'action' => 'wpmf_sync_owncloud',
                'type' => 'folder'
            );

            WpmfAddonHelper::addToQueue($datas);
        }
    }

    /**
     * Create folder
     *
     * @param string $folder_path Folder path
     *
     * @return integer
     */
    public function createFolder($folder_path)
    {
        $folder_path = $this->getValidPath($folder_path);
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($folder_path, '/'));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MKCOL');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $response = curl_exec($ch);
            curl_close($ch);

            // get folder id
            $id = $this->getFileIdByPath(trim($folder_path, '/'));
            return $id;
        }
    }

    /**
     * Create cloud folder from media library
     *
     * @param integer $folder_id    Local folder ID
     * @param string  $name         Folder name
     * @param integer $parent_id    Local folder parent ID
     * @param array   $informations Informations
     *
     * @return boolean
     */
    public function createFolderLibrary($folder_id, $name, $parent_id, $informations)
    {
        try {
            $path = $this->getFolderPath($parent_id);
            if ($path) {
                $cloud_type = wpmfGetCloudFolderType($parent_id);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $id = $this->createFolder($path . '/' . rawurlencode($name));
                    if (!empty($id)) {
                        add_term_meta($folder_id, 'wpmf_drive_id', $id);
                        add_term_meta($folder_id, 'wpmf_drive_path', $path . '/' . rawurlencode($name));
                        add_term_meta($folder_id, 'wpmf_drive_type', 'owncloud');
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Delete cloud folder from media library
     *
     * @param object $folder Local folder info
     *
     * @return boolean
     */
    public function deleteFolder($folder)
    {
        try {
            $path = $this->getFolderPath($folder->term_id);
            if ($path) {
                $cloud_type = wpmfGetCloudFolderType($folder->term_id);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        if ($params['rootfoldername'] !== $path) {
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($path, '/'));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                            $response = curl_exec($ch);
                            curl_close($ch);
                        }
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Delete cloud attachment
     *
     * @param integer $pid Attachment ID
     *
     * @return boolean
     */
    public function deleteAttachment($pid)
    {
        try {
            $path = get_post_meta($pid, 'wpmf_drive_path', true);
            if ($path) {
                $thumb_ids = get_post_meta($pid, 'cloud_thumb_ids', true);
                $cloud_type = get_post_meta($pid, 'wpmf_drive_type', true);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        if ($params['rootfoldername'] !== $path) {
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($path, '/'));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                            $response = curl_exec($ch);
                            curl_close($ch);

                            if (!empty($thumb_ids)) {
                                foreach ($thumb_ids as $thumb_id) {
                                    $ch = curl_init();
                                    curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($thumb_id, '/'));
                                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                                    curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                                    curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                                    $response = curl_exec($ch);
                                    curl_close($ch);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Move cloud folder from media library
     *
     * @param integer $fileid       Local file ID
     * @param integer $parent_id    Local folder new parent ID
     * @param array   $informations Informations
     *
     * @return boolean
     */
    public function moveFileLibrary($fileid, $parent_id, $informations)
    {
        try {
            $path = get_post_meta($fileid, 'wpmf_drive_path', true);
            $path = $this->getValidPath($path);
            if ($path) {
                $cloud_type = wpmfGetCloudFolderType($parent_id);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        $destination_path = $this->getFolderPath($parent_id);
                        $destination_path = trim($destination_path, '/') . '/' . rawurlencode(basename($path));
                        $ch = curl_init();
                        curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($path, '/'));
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                        curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MOVE');
                        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                            'Destination:' . $this->davUrl . $destination_path
                        ));
                        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                        curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                        $response = curl_exec($ch);
                        curl_close($ch);
                        update_post_meta($fileid, 'wpmf_drive_path', $destination_path);
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Move cloud folder from media library
     *
     * @param integer $folder_id    Local folder ID
     * @param integer $parent_id    Local folder new parent ID
     * @param array   $informations Informations
     *
     * @return boolean
     */
    public function moveFolder($folder_id, $parent_id, $informations)
    {
        try {
            $path = $this->getFolderPath($folder_id);
            if ($path) {
                $cloud_type = wpmfGetCloudFolderType($folder_id);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        if ($params['rootfoldername'] !== $path) {
                            $folder = get_term($folder_id, WPMF_TAXO);
                            $destination_path = $this->getFolderPath($parent_id);
                            $destination_path = trim($destination_path, '/') . '/' . rawurlencode($folder->name);
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($path, '/'));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MOVE');
                            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                                'Destination:' . $this->davUrl . $destination_path
                            ));
                            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                            $response = curl_exec($ch);
                            curl_close($ch);
                            update_term_meta($folder_id, 'wpmf_drive_path', $destination_path);
                        }
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Get folder path
     *
     * @param integer $folder_id Folder ID on media library
     * @param string  $type      Folder type
     *
     * @return mixed|string
     */
    public function getFolderPath($folder_id, $type = '')
    {
        if ($type === 'wpmfthumbs') {
            $params = get_option('_wpmfAddon_owncloud_config');
            $path = $params['rootfoldername'];
        } else {
            $path = get_term_meta($folder_id, 'wpmf_drive_path', true);
        }

        $path = $this->getValidPath($path);
        return $path;
    }

    /**
     * Rename cloud folder from media library
     *
     * @param integer $folder_id Local folder ID
     * @param string  $name      New name
     *
     * @return boolean
     */
    public function updateFolderName($folder_id, $name)
    {
        try {
            $path = $this->getFolderPath($folder_id);
            if ($path) {
                $cloud_type = wpmfGetCloudFolderType($folder_id);
                if ($cloud_type && $cloud_type === 'owncloud') {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        if ($params['rootfoldername'] !== $path) {
                            $pathinfo = pathinfo($path);
                            $destination_path = str_replace($pathinfo['basename'], rawurlencode($name), $path);
                            $ch = curl_init();
                            curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($path, '/'));
                            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MOVE');
                            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                                'Destination:' . $this->davUrl . $destination_path
                            ));
                            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                            $response = curl_exec($ch);
                            curl_close($ch);
                            if ($response === '') {
                                update_term_meta($folder_id, 'wpmf_drive_path', $destination_path);

                                // update path of sub folders
                                $sub_folders = get_term_children((int)$folder_id, WPMF_TAXO);
                                foreach ($sub_folders as $f_id) {
                                    $folder_path = get_term_meta($f_id, 'wpmf_drive_path', true);
                                    if (!empty($folder_path)) {
                                        $new_f_path = str_replace($path, $destination_path, $folder_path);
                                        update_term_meta($f_id, 'wpmf_drive_path', $new_f_path);
                                    }
                                }

                                // update path of childrent files
                                $query = new WP_Query(array(
                                    'posts_per_page' => -1,
                                    'post_type' => 'attachment',
                                    'post_status' => 'inherit',
                                    'fields' => 'ids',
                                    'tax_query' => array(
                                        array(
                                            'taxonomy' => WPMF_TAXO,
                                            'field' => 'term_id',
                                            'terms' => $folder_id,
                                            'operator' => 'IN',
                                            'include_children' => true
                                        )
                                    )
                                ));
                                $files = $query->get_posts();
                                foreach ($files as $file) {
                                    $file_path = get_post_meta($file, 'wpmf_drive_path', true);
                                    if (!empty($file_path)) {
                                        $new_f_path = str_replace($path, $destination_path, $file_path);
                                        update_post_meta($file, 'wpmf_drive_path', $new_f_path);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception $ex) {
            return false;
        }

        return true;
    }

    /**
     * Add attachment to cloud
     *
     * @param integer $attachment_id Attachment ID
     *
     * @return void
     */
    public function addAttachment($attachment_id)
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- No action, nonce is not required
        if (!empty($_POST['wpmf_folder'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- No action, nonce is not required
            $folder_id = (int)$_POST['wpmf_folder'];
            $this->doAddAttachment($attachment_id, $folder_id);
        }
    }

    /**
     * Add attachment to cloud
     *
     * @param integer $attachment_id Attachment ID
     * @param integer $folder_id     Folder ID
     *
     * @return void
     */
    public function doAddAttachment($attachment_id, $folder_id)
    {
        $cloud_type = wpmfGetCloudFolderType($folder_id);
        if ($cloud_type && $cloud_type === 'owncloud') {
            $filePath = get_attached_file($attachment_id);
            $response = $this->doUpload($folder_id, $filePath, 'upload');
            if ($response !== '') {
                $configs = get_option('_wpmfAddon_owncloud_config');
                $filePath = get_attached_file($attachment_id);
                $token = $this->getShareToken($response);
                $large_size_w = get_option('large_size_w');
                $large_size_h = get_option('large_size_h');
                $link = $this->getLink(trim($configs['owncloudurl'], '/'), $filePath, $large_size_w, $large_size_h, $token);
                // get file id
                $id = $this->getFileIdByPath(trim($response, '/'));
                if (!empty($id)) {
                    global $wpdb;
                    add_post_meta($attachment_id, 'wpmf_drive_id', $id);
                    add_post_meta($attachment_id, 'wpmf_drive_path', $response);
                    add_post_meta($attachment_id, 'wpmf_drive_type', 'owncloud');
                    $where = array('ID' => $attachment_id);
                    $wpdb->update($wpdb->posts, array('guid' => $link), $where);
                    add_post_meta($attachment_id, 'wpmf_drive_link', $link);
                }
            }
        }
    }

    /**
     * Do upload File
     *
     * @param string $folder_id Folder id
     * @param string $filePath  Full path of attachment
     * @param string $action    Action
     *
     * @return boolean|string
     */
    public function doUpload($folder_id, $filePath, $action = 'upload')
    {
        if ($action === 'upload_thumbnail') {
            $path = $this->getFolderPath($folder_id, 'wpmfthumbs');
        } else {
            $path = $this->getFolderPath($folder_id);
        }

        if ($path) {
            $cloud_type = wpmfGetCloudFolderType($folder_id);
            if ($cloud_type && $cloud_type === 'owncloud') {
                try {
                    $params = get_option('_wpmfAddon_owncloud_config');
                    $isConnected = $this->isConnected();
                    if ($isConnected) {
                        if (file_exists($filePath)) {
                            $info = pathinfo($filePath);
                            $chunk_size = 4000000;

                            $davUrl = trim($params['owncloudurl'], '/') . '/remote.php/webdav/';
                            $data = file_get_contents($filePath);
                            $name = basename($filePath);

                            $ch = curl_init();
                            $fileSize = filesize($filePath);

                            if ($action === 'upload_thumbnail') {
                                $destination = trim($path, '/') . '/wpmfthumbs/' . $name;
                            } else {
                                $destination = trim($path, '/') . '/' . $name;
                            }

                            if ($fileSize < $chunk_size) {
                                curl_setopt($ch, CURLOPT_URL, $this->davUrl .  $destination);
                                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                                curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                                $response = curl_exec($ch);
                            } else { // Chunked large file to upload
                                // folder for chunks
                                curl_setopt($ch, CURLOPT_URL, $davUrl . 'uploads/'. $params['username'] .'/' . $info['filename']);
                                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MKCOL');
                                curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                                curl_exec($ch);

                                // upload chunks
                                $chunks = str_split($data, $chunk_size);
                                $start = 0;
                                $end = $chunk_size;

                                foreach ($chunks as $key => $chunk) {
                                    curl_setopt($ch, CURLOPT_URL, $davUrl . 'uploads/' . $params['username'] . '/' . $info['filename'] . '/' . $start . '-' . $end);
                                    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                                    curl_setopt($ch, CURLOPT_POSTFIELDS, $chunk);
                                    $res = curl_exec($ch);
                                    $start = 1 + $end;
                                    $end = $chunk_size + $start;
                                }

                                // try to move the chunk and use the folder as destination
                                curl_setopt($ch, CURLOPT_URL, $davUrl . 'uploads/' . $params['username'] . '/' . $info['filename'] . '/.file');
                                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'MOVE');
                                curl_setopt($ch, CURLOPT_HTTPHEADER, array('Destination:' . $this->davUrl . $destination));
                                $response = curl_exec($ch);
                            }


                            curl_close($ch);
                            return $destination;
                        }
                    }
                    return false;
                } catch (Exception $e) {
                    return false;
                }
            }
        }
        return false;
    }

    /**
     * Update metadata for cloud file
     *
     * @param array   $meta          Meta data
     * @param integer $attachment_id Attachment ID
     *
     * @return mixed
     */
    public function wpGenerateAttachmentMetadata($meta, $attachment_id)
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- No action, nonce is not required
        if (!empty($_POST['wpmf_folder'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- No action, nonce is not required
            $type = get_term_meta($_POST['wpmf_folder'], 'wpmf_drive_type', true);
            if ($type === 'owncloud') {
                $file_paths = WpmfAddonHelper::getAttachmentFilePaths($attachment_id, $meta);
                update_post_meta($attachment_id, 'wpmf_origin_file_paths', $file_paths);
                // add to queue
                $datas = array(
                    'attachment_id' => $attachment_id,
                    'folder_id' => $_POST['wpmf_folder'], // phpcs:ignore WordPress.Security.NonceVerification.Missing -- No action, nonce is not required
                    'action' => 'wpmf_owncloud_render_thumbnail'
                );

                WpmfAddonHelper::addToQueue($datas);
            }
        }

        return $meta;
    }

    /**
     * Render thumbnail
     *
     * @param boolean $result     Result
     * @param array   $datas      Data details
     * @param integer $element_id ID of queue element
     *
     * @return boolean|integer
     */
    public function renderThumbnail($result, $datas, $element_id)
    {
        $this->doUploadThumbnail($datas['attachment_id'], $datas['folder_id']);
        return $result;
    }

    /**
     * Render thumbnail
     *
     * @param integer $attachment_id Attachment ID
     * @param integer $folder_id     Folder ID
     *
     * @return void
     */
    public function doUploadThumbnail($attachment_id, $folder_id)
    {
        $old_meta = get_post_meta($attachment_id, '_wp_attachment_metadata', true);
        $meta = get_post_meta($attachment_id, '_wp_attachment_metadata', true);
        $pub_meta = get_post_meta($attachment_id, '_wp_attachment_metadata', true);
        $file_paths = WpmfAddonHelper::getAttachmentFilePaths($attachment_id, $meta);
        $thumb_ids = array();
        foreach ($file_paths as $size => $file_path) {
            if ($size === 'original') {
                continue;
            }

            $response = $this->doUpload($folder_id, $file_path, 'upload_thumbnail');
            // upload attachment to cloud
            if (!empty($response)) {
                $configs = get_option('_wpmfAddon_owncloud_config');
                $token = $this->getShareToken($response);
                $link = $this->getLink(trim($configs['owncloudurl'], '/'), $file_path, 0, 0, $token);
                $meta['sizes'][$size]['file'] = $link;
                $thumb_ids[] = $response;
            }
        }

        // remove file from server
        $datas = array(
            'attachment_id' => $attachment_id,
            'action' => 'wpmf_remove_server_file',
            'meta' => $old_meta
        );

        WpmfAddonHelper::addToQueue($datas);

        update_post_meta($attachment_id, '_wp_attachment_metadata', $meta);
        update_post_meta($attachment_id, '_wpmf_attachment_metadata_pub', $pub_meta);
        update_post_meta($attachment_id, 'cloud_thumb_ids', $thumb_ids);
    }

    /**
     * Get full share link
     *
     * @param string  $owncloudurl Next cloud url
     * @param string  $file_path   File path
     * @param integer $width       Width
     * @param integer $height      Height
     * @param string  $token       Token
     *
     * @return string
     */
    public function getLink($owncloudurl, $file_path, $width = 0, $height = 0, $token = '')
    {
        if ($token) {
            $isImage = $this->isImage($file_path);
            $info = pathinfo($file_path);
            $isSupportedVideoAudio = false;
            if (in_array(strtolower($info['extension']), array('mp4', 'webm', 'ogg', 'mp3'))) {
                $isSupportedVideoAudio = true;
            }
            if ($isImage || $isSupportedVideoAudio) {
                if (empty($width) || empty($height)) {
                    list($width, $height) = getimagesize($file_path);
                }
                $link = $owncloudurl . '/index.php/s/'. $token .'/download?path=&files=';
            } else {
                $link = $owncloudurl . '/index.php/s/'. $token;
            }

            return $link;
        }
    }

    /**
     * Check a file is image
     *
     * @param string $file_path File path
     *
     * @return boolean
     */
    public function isImage($file_path)
    {
        $info = pathinfo($file_path);
        if (in_array(strtolower($info['extension']), array('jpg', 'jpeg', 'png', 'webp'))) {
            return true;
        }

        return false;
    }
    /**
     * Remove server file
     *
     * @param boolean $result     Result
     * @param array   $datas      Data details
     * @param integer $element_id ID of queue element
     *
     * @return boolean|integer
     */
    public function removeServerFile($result, $datas, $element_id)
    {
        if (!empty($datas['meta']['sizes'])) {
            $file_paths = WpmfAddonHelper::getAttachmentFilePaths($datas['attachment_id'], $datas['meta']);
            foreach ($file_paths as $file_path) {
                unlink($file_path);
            }
        } else {
            $file_path = get_attached_file($datas['attachment_id']);
            unlink($file_path);
        }

        return $result;
    }

    /**
     * Create share link of file
     *
     * @param string $path Path of file
     *
     * @return array|mixed
     */
    public function createShareLink($path)
    {
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $params['owncloudurl'] . '/ocs/v2.php/apps/files_sharing/api/v1/shares');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: multipart/form-data', 'OCS-APIRequest:true'));
            curl_setopt($ch, CURLOPT_POSTFIELDS, array('path' => $path, 'shareType' => 3));
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $response = curl_exec($ch);
            curl_close($ch);

            $datas = $this->convertXmlToArray($response);
            if (!empty($datas['meta']) && !empty($datas['meta']['statuscode'])) {
                if (!empty($datas['data'])) {
                    return $datas['data'];
                }
            }
        }
        return false;
    }

    /**
     * Get share token
     *
     * @param string $path File path
     *
     * @return mixed
     */
    public function getShareToken($path)
    {
        $path = rawurldecode($path);
        $get_shared_links = $this->getShareLink($path);
        if (empty($get_shared_links)) {
            $res = $this->createShareLink($path);
            $token = $res['token'];
        } else {
            if (!empty($get_shared_links['token'])) {
                $token = $get_shared_links['token'];
            } else {
                $token = $get_shared_links[count($get_shared_links) - 1]['token'];
            }
        }

        return $token;
    }
    /**
     * Get share link of file
     *
     * @param string $path Path of file
     *
     * @return array|mixed
     */
    public function getShareLink($path)
    {
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $params['owncloudurl'] . '/ocs/v2.php/apps/files_sharing/api/v1/shares?path=' . $path);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array('OCS-APIRequest:true'));
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $response = curl_exec($ch);
            curl_close($ch);

            $datas = $this->convertXmlToArray($response);
            if (!empty($datas['meta']) && !empty($datas['meta']['statuscode']) && (int)$datas['meta']['statuscode'] === 200) {
                if (!empty($datas['data']) && !empty($datas['data']['element'])) {
                    return $datas['data']['element'];
                }
            }
        }
        return array();
    }

    /**
     * Convert xml string to array
     *
     * @param string $xml Xml string
     *
     * @return array
     */
    public function convertXmlToArray($xml)
    {
        $content = str_replace(array('\n', '\r', '\t', 'd:'), '', $xml);
        $content = trim(str_replace('"', "'", $content));
        $simpleXml = simplexml_load_string($content);
        $json = json_encode($simpleXml);
        $return = json_decode($json, true);
        return $return;
    }

    /**
     * Sync cloud folder and file from queue
     *
     * @param boolean $result     Result
     * @param array   $datas      Data details
     * @param integer $element_id ID of queue element
     *
     * @return boolean|integer
     */
    public function doSync($result, $datas, $element_id)
    {
        $isConnected = $this->isConnected();
        if (!$isConnected) {
            return -1;
        }

        global $wpdb;
        $name = html_entity_decode(rawurldecode($datas['name']), ENT_COMPAT, 'UTF-8');
        if (strpos($name, 'wpmfthumbs') !== false) {
            return true;
        }
        if ($datas['type'] === 'folder') {
            // check folder exists
            if ($datas['id'] === '') {
                $meta_key = 'wpmf_drive_root_id';
            } else {
                $meta_key = 'wpmf_drive_id';
            }
            $row = $wpdb->get_row($wpdb->prepare('SELECT term_id, meta_value FROM ' . $wpdb->termmeta . ' WHERE meta_key = %s AND BINARY meta_value = BINARY %s', array($meta_key, $datas['id'])));
            // if folder not exists
            if (!$row) {
                $inserted = wp_insert_term($name, WPMF_TAXO, array('parent' => (int)$datas['folder_parent']));
                if (is_wp_error($inserted)) {
                    $folder_id = (int)$inserted->error_data['term_exists'];
                } else {
                    $folder_id = (int)$inserted['term_id'];
                }
                
                if ((int)$datas['folder_parent'] === 0) {
                    update_term_meta($folder_id, 'wpmf_drive_root_id', $datas['id']);
                } else {
                    update_term_meta($folder_id, 'wpmf_drive_id', $datas['id']);
                }
                update_term_meta($folder_id, 'wpmf_drive_path', $datas['path']);
            } else {
                $folder_id = (int)$row->term_id;
                $exist_folder = get_term($folder_id, WPMF_TAXO);
                // if folder exists, then update parent and name
                if (!empty($datas['folder_parent']) && (int)$exist_folder->parent !== (int)$datas['folder_parent']) {
                    $parent_exist = get_term((int)$datas['folder_parent'], WPMF_TAXO);
                    if (!is_wp_error($parent_exist)) {
                        wp_update_term($folder_id, WPMF_TAXO, array('parent' => (int) $datas['folder_parent']));
                    }
                }

                if ($name !== $exist_folder->name) {
                    wp_update_term($folder_id, WPMF_TAXO, array('name' => $name));
                }
            }

            // find childs element to add to queue
            if (!empty($folder_id)) {
                $responses = array();
                $responses['folder_id'] = (int)$folder_id;
                update_term_meta($responses['folder_id'], 'wpmf_drive_type', 'owncloud');
                $wpmfQueue = JuMainQueue::getInstance('wpmf');
                $wpmfQueue->updateQueueTermMeta((int)$responses['folder_id'], (int)$element_id);
                $wpmfQueue->updateResponses((int)$element_id, $responses);
                $this->addChildsToQueue($datas['id'], $datas['path'], $folder_id);
            }
        } else {
            $is_thumb_or_scaled = preg_match('/(-scaled|[_-]\d+x\d+)|@[2-6]\x(?=\.[a-z]{3,4}$)/im', $name) === true;
            if ($is_thumb_or_scaled) {
                return true;
            }

            $upload_path = wp_upload_dir();
            $info = pathinfo($name);
            $row = $wpdb->get_row($wpdb->prepare('SELECT post_id, meta_value FROM ' . $wpdb->postmeta . ' WHERE meta_key = %s AND BINARY meta_value = BINARY %s', array('wpmf_drive_id', $datas['id'])));
            $params = get_option('_wpmfAddon_owncloud_config');
            if (!$row) {
                $token = $this->getShareToken($datas['path']);
                $large_size_w = get_option('large_size_w');
                $large_size_h = get_option('large_size_h');
                $link = $this->getLink(trim($params['owncloudurl'], '/'), $datas['path'], $large_size_w, $large_size_h, $token);
                if (!$link) {
                    return false;
                }

                // insert attachment
                $attachment = array(
                    'guid'           => $link,
                    'post_mime_type' => $datas['file']['mimeType'],
                    'post_title'     => $info['filename'],
                    'post_author'   => (int)$datas['user_id'],
                    'post_type'     => 'attachment',
                    'post_status'    => 'inherit'
                );

                $file_id   = wp_insert_post($attachment);
                $attached = trim($upload_path['subdir'], '/') . '/' . $name;
                wp_set_object_terms((int) $file_id, (int)$datas['folder_parent'], WPMF_TAXO);

                update_post_meta($file_id, '_wp_attached_file', $attached);
                update_post_meta($file_id, 'wpmf_size', $datas['size']);
                update_post_meta($file_id, 'wpmf_filetype', $info['extension']);
                update_post_meta($file_id, 'wpmf_order', 0);
                update_post_meta($file_id, 'wpmf_drive_id', $datas['id']);
                update_post_meta($file_id, 'wpmf_drive_path', $datas['path']);
                update_post_meta($file_id, 'wpmf_drive_type', 'owncloud');

                $meta = array();
                if (strpos($datas['file']['mimeType'], 'image') !== false) {
                    if (isset($child['image']) && isset($child['image']['width']) && isset($datas['image']['height'])) {
                        $meta['width'] = $datas['image']['width'];
                        $meta['height'] = $datas['image']['height'];
                    } else {
                        list($width, $heigth) = wpmfGetImgSize($link);
                        $meta['width'] = $width;
                        $meta['height'] = $heigth;
                    }

                    $meta['file'] = $link;
                }

                if (isset($datas['size'])) {
                    $meta['filesize'] = $datas['size'];
                }

                $sizes = $this->renderMetadata($info['extension']);
                $meta['sizes'] = $sizes;

                update_post_meta($file_id, '_wp_attachment_metadata', $meta);
            } else {
                // update attachment
                $file_id = $row->post_id;

                $metadata = get_post_meta($file_id, '_wp_attachment_metadata', true);
                if (empty($metadata) || !is_array($metadata)) {
                    $metadata = array();
                }
                $update = false;
                if (empty($metadata['width']) || empty($metadata['height'])) {
                    if (isset($datas['image']['width']) && isset($datas['image']['height'])) {
                        $metadata['width'] = $datas['image']['width'];
                        $metadata['height'] = $datas['image']['height'];
                        $update = true;
                    }
                }

                if (empty($metadata['filesize'])) {
                    if (isset($datas['size'])) {
                        $metadata['filesize'] = $datas['size'];
                        $update = true;
                    }
                }

                if (empty($metadata['sizes'])) {
                    $sizes = $this->renderMetadata($info['extension']);
                    $metadata['sizes'] = $sizes;
                    $update = true;
                }

                if ($update) {
                    update_post_meta($file_id, '_wp_attachment_metadata', $metadata);
                }

                $this->updateAttachment($info, $file_id, $datas['folder_parent']);
                $file = get_post($file_id);
                // update author
                if (empty($file->post_author)) {
                    $my_post = array(
                        'ID'           => $file_id,
                        'post_author'   => (int)$datas['user_id']
                    );

                    wp_update_post($my_post);
                }
                // update file URL
                $token = $this->getShareToken($datas['path']);
                $large_size_w = get_option('large_size_w');
                $large_size_h = get_option('large_size_h');
                $link = $this->getLink(trim($params['owncloudurl'], '/'), $datas['path'], $large_size_w, $large_size_h, $token);
                if (!$link) {
                    return false;
                }

                $wpdb->update(
                    $wpdb->posts,
                    array(
                        'guid' => $link
                    ),
                    array('ID' => $file_id),
                    array(
                        '%s'
                    ),
                    array('%d')
                );

                update_post_meta($file_id, 'wpmf_drive_link', $link);

                // thumbnail
                /*$sizes = $this->renderMetaSizes($info['extension'], $file_id, $datas['id']);
                if (!empty($sizes)) {
                    $meta['sizes'] = $sizes;
                }*/

                //update_post_meta($file_id, '_wp_attachment_metadata', $meta);
            }

            if (!empty($file_id)) {
                $responses = array();
                $responses['attachment_id'] = (int)$file_id;
                $wpmfQueue = JuMainQueue::getInstance('wpmf');
                $wpmfQueue->updateResponses((int)$element_id, $responses);
                $wpmfQueue->updateQueuePostMeta((int)$file_id, (int)$element_id);
            }
        }

        return true;
    }

    /**
     * Render meta data
     *
     * @param string $extension File extension
     *
     * @return array
     */
    public function renderMetadata($extension)
    {
        $configs = get_option('_wpmfAddon_cloud_config');
        if (empty($configs['generate_thumbnails'])) {
            return array();
        }
        $sizes = array();
        $thumbnail_size_w = get_option('thumbnail_size_w');
        $thumbnail_size_h = get_option('thumbnail_size_h');
        
        switch (strtolower($extension)) {
            case 'avif':
                $size_info = array();
                $size_info['file'] = get_site_url() . '/wp-includes/images/media/default.svg';
                $size_info['url'] = get_site_url() . '/wp-includes/images/media/default.svg';
                $size_info['width'] = $thumbnail_size_w;
                $size_info['height'] = $thumbnail_size_h;
                $sizes['thumbnail'] = $size_info;
                break;
        }
        return $sizes;
    }

    /**
     * Update attachment
     *
     * @param array   $info    File info
     * @param integer $file_id Attachment ID
     * @param integer $parent  Parent folder
     *
     * @return void
     */
    public function updateAttachment($info, $file_id, $parent)
    {
        $curent_parents = get_the_terms($file_id, WPMF_TAXO);
        if (isset($parent)) {
            if (empty($curent_parents)) {
                wp_set_object_terms((int) $file_id, (int)$parent, WPMF_TAXO);
            } else {
                foreach ($curent_parents as $curent_parent) {
                    if (!empty($parent) && (int)$curent_parent->term_id !== (int)$parent) {
                        wp_set_object_terms((int) $file_id, (int)$parent, WPMF_TAXO);
                    }
                }
            }
        }

        $attached_file = get_post_meta($file_id, '_wp_attached_file', true);
        $attached_info = pathinfo($attached_file);
        if ($info['filename'] !== $attached_info['filename']) {
            $new_path = str_replace($attached_info['filename'], $info['filename'], $attached_file);
            update_post_meta($file_id, '_wp_attached_file', $new_path);
        }
    }

    /**
     * Add child items to queue
     *
     * @param string  $folderID      ID of cloud folder
     * @param string  $folder_path   Path of cloud folder
     * @param integer $folder_parent ID of folder parent on media library
     *
     * @return void
     */
    public function addChildsToQueue($folderID, $folder_path, $folder_parent)
    {
        $error = false;
        $childs = array();

        try {
            $childs = $this->getChildsInFolder($folder_path);
        } catch (Exception $e) {
            $error = true;
        }

        if ($error) {
            return;
        }

        include_once 'includes/mime-types.php';
        // get folder childs list on cloud
        $cloud_folders_list = array();
        // get file childs list on cloud
        $cloud_files_list = array();
        $params = get_option('_wpmfAddon_owncloud_config');
        foreach ($childs as $i => $child) {
            if ((int)$i === 0) {
                continue;
            }
            $s = 'remote.php/webdav/';
            $paths = explode($s, $child['href']);
            $path = trim($paths[1], '/');
            $datas = array(
                'path' => $path,
                'folder_parent' => $folder_parent,
                'name' => mb_convert_encoding(basename($path), 'HTML-ENTITIES', 'UTF-8'),
                'action' => 'wpmf_sync_owncloud',
                'cloud_parent' => $folderID
            );

            $fileExtension = pathinfo($child['href'], PATHINFO_EXTENSION);
            if ($fileExtension !== '') {
                // ignore thumbnail
                $name = mb_convert_encoding(basename($path), 'HTML-ENTITIES', 'UTF-8');
                $is_thumb_or_scaled = preg_match('/(-scaled|[_-]\d+x\d+)|@[2-6]\x(?=\.[a-z]{3,4}$)/im', $name) === true;
                if ($is_thumb_or_scaled || strpos($name, 'wpmfthumbs') !== false) {
                    continue;
                }

                $cloud_files_list[] = $child['propstat']['prop']['fileid'];
                $mimeType   = getMimeType($fileExtension);
                $datas['id'] = $child['propstat']['prop']['fileid'];
                $datas['type'] = 'file';
                $datas['file'] = array('mimeType' => $mimeType);
                $datas['image'] = array();
                $datas['size'] = $child['propstat']['prop']['size'];
                /*if (strpos($mimeType, 'image') !== false) {
                    $dimensions = array('width' => 0, 'height' => 0);
                    if (isset($child['media_info'])) {
                        if (empty($child['media_info']['metadata']['dimensions'])) {
                            $dimensions = array(
                                'width' => $child['media_info']['metadata']['dimensions']['width'],
                                'height' => $child['media_info']['metadata']['dimensions']['height']
                            );
                        }
                    }
                    $datas['image'] = $dimensions;
                }*/
            } else {
                $datas['id'] = $child['propstat'][0]['prop']['fileid'];
                $cloud_folders_list[] = $child['propstat'][0]['prop']['fileid'];
                $datas['type'] = 'folder';
            }

            WpmfAddonHelper::addToQueue($datas);
        }

        // then remove the file and folder not exist
        $datas = array(
            'id' => '',
            'media_folder_id' => $folder_parent,
            'cloud_folder_id' => $folderID,
            'action' => 'wpmf_owncloud_remove',
            'cloud_files_list' => $cloud_files_list,
            'cloud_folders_list' => $cloud_folders_list
        );

        WpmfAddonHelper::addToQueue($datas);
    }

    /**
     * Remove the files/folders when sync
     *
     * @param boolean $result     Result
     * @param array   $datas      Data details
     * @param integer $element_id ID of queue element
     *
     * @return boolean|integer
     */
    public function syncRemoveItems($result, $datas, $element_id)
    {
        if (defined('WPMFADDON_SYNC')) {
            return true;
        }
        global $wpdb;
        remove_action('delete_attachment', array($this, 'deleteAttachment'));
        remove_action('wpmf_before_delete_folder', array($this, 'deleteFolderLibrary'));
        $args = array(
            'post_type' => 'attachment',
            'fields' => 'ids',
            'posts_per_page' => -1,
            'post_status' => 'any',
            'meta_query' => array(
                array(
                    'key'       => 'wpmf_drive_type',
                    'value'     => 'owncloud',
                    'compare'   => '='
                )
            ),
            'tax_query'      => array(
                array(
                    'taxonomy'         => WPMF_TAXO,
                    'field'            => 'term_id',
                    'terms'            => (int)$datas['media_folder_id'],
                    'include_children' => false
                )
            ),
        );
        $postIDs = get_posts($args);
        if (!empty($postIDs)) {
            if (!empty($datas['cloud_files_list'])) {
                $results = $wpdb->get_col(
                    $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.QuotedDynamicPlaceholderGeneration
                        'Select post_id from '. $wpdb->postmeta . ' WHERE post_id IN  (' . implode(',', esc_sql($postIDs)) . ") AND meta_key=%s AND meta_value NOT IN ('".  implode("','", esc_sql($datas['cloud_files_list'])) ."')",
                        array( 'wpmf_drive_id')
                    )
                );
                if (!empty($results)) {
                    foreach ($results as $postID) {
                        wp_delete_attachment($postID);
                    }
                }
            } else {
                foreach ($postIDs as $postID) {
                    wp_delete_attachment($postID);
                }
            }
        }


        // get media library files in current folder, then remove the folder not exist
        $folders = get_categories(array('hide_empty' => false, 'taxonomy' => WPMF_TAXO, 'parent' => (int)$datas['media_folder_id']));
        foreach ($folders as $folder) {
            $drive_id = get_term_meta($folder->term_id, 'wpmf_drive_id', true);
            if (is_array($datas['cloud_folders_list']) && !empty($datas['cloud_folders_list']) && !empty($drive_id) && !in_array($drive_id, $datas['cloud_folders_list'])) {
                wp_delete_term($folder->term_id, WPMF_TAXO);
            }
        }
        return true;
    }

    /**
     * Import file to media library
     *
     * @param string  $cloud_id  Cloud file ID
     * @param integer $term_id   Folder target ID
     * @param boolean $imported  Check imported
     * @param string  $filename  File name
     * @param string  $extension File extension
     *
     * @return boolean
     */
    public function importFile($cloud_id, $term_id, $imported, $filename, $extension)
    {
        $valid_path = $this->getValidPath($cloud_id);
        $params = get_option('_wpmfAddon_owncloud_config');
        $isConnected = $this->isConnected();
        if ($isConnected) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->davUrl . $valid_path);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            //curl_setopt($ch, CURLOPT_HTTPHEADER, array('OCS-APIRequest:true'));
            curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
            $content = curl_exec($ch);
            curl_close($ch);

            if ($content) {
                $info = pathinfo($valid_path);
                $extension = strtolower($info['extension']);
                $upload_dir = wp_upload_dir();
                require_once 'includes/mime-types.php';
                $getMimeType = getMimeType($extension);
                $file_exists = WpmfAddonHelper::checkExistPost($term_id, $cloud_id);
                if (!empty($file_exists)) {
                    return false;
                }

                $attach_id = $this->insertAttachmentMetadata(
                    $upload_dir['path'],
                    $upload_dir['url'],
                    $filename,
                    $content,
                    $getMimeType,
                    $extension,
                    $term_id
                );

                if ($attach_id) {
                    update_post_meta($attach_id, 'wpmf_import_cloud_id', $cloud_id);
                    return true;
                }
            }
        }

        return $imported;
    }

    /**
     * Insert a attachment to database
     *
     * @param string  $upload_path Wordpress upload path
     * @param string  $upload_url  Wordpress upload url
     * @param string  $file        File name
     * @param string  $content     Content of file
     * @param string  $mime_type   Mime type of file
     * @param string  $ext         Extension of file
     * @param integer $term_id     Media folder id to set file to folder
     *
     * @return boolean
     */
    public function insertAttachmentMetadata(
        $upload_path,
        $upload_url,
        $file,
        $content,
        $mime_type,
        $ext,
        $term_id
    ) {
        $file   = wp_unique_filename($upload_path, $file);
        $upload = file_put_contents($upload_path . '/' . $file, $content);
        if ($upload) {
            $attachment = array(
                'guid'           => $upload_url . '/' . $file,
                'post_mime_type' => $mime_type,
                'post_title'     => str_replace('.' . $ext, '', $file),
                'post_status'    => 'inherit'
            );

            $image_path = $upload_path . '/' . $file;
            // Insert attachment
            $attach_id   = wp_insert_attachment($attachment, $image_path);
            $attach_data = wp_generate_attachment_metadata($attach_id, $image_path);
            wp_update_attachment_metadata($attach_id, $attach_data);
            // set attachment to term
            wp_set_object_terms((int) $attach_id, (int) $term_id, WPMF_TAXO, true);
            return true;
        }

        return false;
    }

    /**
     * Get content file by URL
     *
     * @return boolean
     */
    public function getContentFile()
    {
        $isConnected = $this->isConnected();
        if ($isConnected) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- download URL inserted post content
            if (empty($_REQUEST['url'])) {
                return false;
            }
            // phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.EscapeOutput.OutputNotEscaped -- download URL inserted post content
            $url = $_REQUEST['url'];
            $content = file_get_contents($url);
            echo $content;
            die();
        }

        return false;
    }

    /**
     * Replace file
     *
     * @param string  $content  Content file
     * @param integer $filepath File path
     *
     * @return boolean
     */
    public function replaceFile($content, $filepath)
    {
        try {
            $valid_path = $this->getValidPath($filepath);
            $params = get_option('_wpmfAddon_owncloud_config');
            $isConnected = $this->isConnected();
            if ($isConnected) {
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $this->davUrl . trim($valid_path, '/') . '/');
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($ch, CURLOPT_USERPWD, $params['username'] . ':' . $params['password']);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $content);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                $response = curl_exec($ch);
                curl_close($ch);

                return true;
            }
        } catch (Exception $e) {
            return false;
        }
    }
}
