<?php

namespace WPPayFormPro\Classes\Export;

use WPPayForm\App\Models\Form;
use WPPayForm\App\Models\OrderItem;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Subscription;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\SubscriptionTransaction;
use WPPayForm\App\Modules\Entry\Entry;

use WPPayForm\App\Modules\Builder\Helper;


class Export
{
    public function exportData($request, $formId)
    {

        $paymentStatus = sanitize_text_field($request->payment_status);
        if (!$formId) {
            exit('No Form Found');
        }
        $type = 'csv';

        if ($request->doc_type) {
            $type = sanitize_text_field($request->doc_type);
        }

        $searchString = '';
        // if (isset($request->search_string) && $request->search_string) {
        //     $searchString = sanitize_text_field($request->search_string);
        // }

        if ($request->get('search_string')) {
            $searchString = sanitize_text_field($request->get('search_string'));
        }

        $startDate = null;
        if ($request->get('start_date') && $request->get('start_date') != 'null') {
            $startDate = sanitize_text_field($request->get('start_date'));
        }

        $endDate = null;
        if ($request->get('end_date') && $request->get('end_date') != 'null') {
            $endDate = sanitize_text_field($request->get('end_date'));
        }

        if (!in_array($type, ['csv', 'ods', 'xlsx', 'json'])) {
            exit('Invalid requested format');
        }

        $form = Form::getForm($formId);

        if (!$form) {
            exit('No Form Found');
        }

        if ($type == 'json') {
            $this->exportAsJSON($formId, $paymentStatus, $searchString, $startDate, $endDate);
        }

        $formattedData = $this->getExportDataArray(
            $this->getSubmissions($formId, $paymentStatus, $searchString, $startDate, $endDate),
            $formId
        );

        $this->downloadOfficeDoc(
            $formattedData,
            $type,
            sanitize_title($form->post_title, 'export', 'view') . '-' . current_time('Y-m-d')
        );
    }

    public function exportAsJSON($formId, $paymentStatus, $searchString = false, $startDate = null, $endDate = null)
    {
        $form = Form::getForm($formId);
        if (!$form) {
            exit('No Form Found');
        }
        $formattedData = $this->getDataObjects(
            $this->getSubmissions($formId, $paymentStatus, $searchString, $startDate, $endDate),
            $formId
        );

        header('Content-disposition: attachment; filename=' . sanitize_title($form->post_title, 'export', 'view') . '-' . date('Y-m-d') . '.json');
        header('Content-type: application/json');
        echo json_encode($formattedData);
        exit();
    }

    private function getSubmissions($formId, $paymentStatus, $search = false, $start_date = null, $end_date = null)
    {
        $wheres = [];
        if ($paymentStatus) {
            $wheres['payment_status'] = $paymentStatus;
        }
        $submissionModel = new Submission();
        $submissions = $submissionModel->getAll($formId, $wheres, false, false, 'ASC', $search, $start_date, $end_date);
        return $submissions->items;
    }

    private function getExportDataArray($submissions, $formId)
    {
        $inputLabels = (array) Form::getFormInputLabels($formId);
        $hasPaymentInputs = Form::hasPaymentFields($formId);

        $submissionColumns = array(
            'id' => __('ID', 'wp-payment-form-pro'),
            'created_at' => __('Submission Date', 'wp-payment-form-pro')
        );

        $hasTaxFields = false;
        if ($hasPaymentInputs) {
            $hasTaxFields = Form::hasTheFields($formId, 'tax_payment_input');
            $hasCouponFields = Form::hasTheFields($formId, 'coupon');
            if ($hasTaxFields) {
                $paymentColumns = [
                    'payment_status' => __('Payment Status', 'wp-payment-form-pro'),
                    'sub_total' => __('Sub Total', 'wp-payment-form-pro'),
                    'tax_total' => __('Tax Total', 'wp-payment-form-pro'),
                    'payment_mode' => __('Payment Mode', 'wp-payment-form-pro'),
                    'payment_method' => __('Payment Method', 'wp-payment-form-pro'),
                    'payment_items' => __('Payment Items', 'wp-payment-form-pro'),
                    'payment_total_in_decimal' => __('Payment Total', 'wp-payment-form-pro'),
                ];
            } else {
                $paymentColumns = [
                    'payment_status' => __('Payment Status', 'wp-payment-form-pro'),
                    'payment_mode' => __('Payment Mode', 'wp-payment-form-pro'),
                    'payment_method' => __('Payment Method', 'wp-payment-form-pro'),
                    'payment_items' => __('Payment Items', 'wp-payment-form-pro'),
                    'payment_total_in_decimal' => __('Payment Total', 'wp-payment-form-pro'),
                ];
            }

            if ($hasCouponFields) {
                $new_item = array('total_discount' => __('Total Discount', 'wp-payment-form-pro'));
                $paymentColumns =
                    array_slice($paymentColumns, 0, 2, true) + $new_item + array_slice($paymentColumns, 2, NULL, true);

                // create applied coupon codes column
                $new_item = array('applied_coupons' => __('Applied Coupons', 'wp-payment-form-pro'));
                $paymentColumns =
                    array_slice($paymentColumns, 0, 3, true) + $new_item + array_slice($paymentColumns, 3, NULL, true);
            }

            $submissionColumns = array_merge($submissionColumns, $paymentColumns);
        }

        $subscriptionModel = new Subscription();

        $hasRecurring = Form::hasRecurring($formId);
        if ($hasRecurring) {
            $subscriptionColumns = [
                'subscription_items' => 'Subscription Items'
            ];
            $submissionColumns = array_merge($submissionColumns, $subscriptionColumns);
        }

        $submissionColumns = apply_filters('wppayform/exportdata_submission_columns', $submissionColumns, $formId);

        $header = array_merge(array_values($submissionColumns), array_values($inputLabels));

        $formattedData = [];
        $formattedData[] = $header;

        foreach ($submissions as $submission) {
            $entry = new Entry($submission);
            $trasubscriptionTransactionModel = new SubscriptionTransaction();
            $hasSubscription = $trasubscriptionTransactionModel->hasSubscription($submission->id);
            $entry->default = '';
            $discount = (new OrderItem())->getDiscountTotal($submission->id);
            $discountItems = (new OrderItem())->getDiscountItems($submission->id)->toArray();
            $couponCodes = '';
            if ($discountItems) {
                foreach ($discountItems as $key => $value) {

                    if ($key != count($discountItems) - 1 && count($discountItems) > 1) {
                        $couponCodes .= isset($value['item_name']) ? $value['item_name'] : '';
                        $couponCodes .= ', ';
                    } else {
                        $couponCodes .= isset($value['item_name']) ? $value['item_name'] : '';
                    }
                }
            }
            if ($hasTaxFields) {
                $taxTotal = $entry->getTaxTotal();
            } else {
                $taxTotal = 0;
            }

            $data = [];
            foreach ($submissionColumns as $columnName => $column) {
                if ($columnName == 'payment_items') {
                    $data[] = $entry->getOrderItemsAsText();
                } elseif ($columnName == 'subscription_items') {
                    $data[] = $entry->getSubscriptionsAsText();
                } elseif ($columnName == 'total_discount') {
                    $data[] = number_format($discount / 100, 2);
                } elseif ($columnName == 'applied_coupons') {
                    $data[] = $couponCodes;
                } elseif ($columnName == 'tax_total') {
                    $data[] = number_format($taxTotal / 100, 2);
                } elseif ($columnName == 'sub_total') {
                    $data[] = number_format(($submission->payment_total - $taxTotal) / 100, 2);
                } elseif ($columnName == 'payment_total_in_decimal') {
                    if ($hasSubscription) {
                        $data[] = number_format(($submission->payment_total) / 100, 2);
                    } else {
                        //  $data[] = number_format(($submission->payment_total - $discount + $taxTotal ) / 100, 2);
                        $data[] = number_format(($submission->payment_total - $discount) / 100, 2);
                    }
                } else {
                    $data[] = $entry->{$columnName};
                }
            }

            foreach ($inputLabels as $inputKey => $item) {
                $data[] = Helper::sanitizeForCSV($entry->getInput($inputKey, ''));
            }
            $formattedData[] = $data;
        }

        return $formattedData;
    }

    private function getDataObjects($submissions, $formId)
    {
        $formattedData = [];
        $hasPaymentInputs = Form::hasPaymentFields($formId);
        $transactionModel = new Transaction();
        $orderItemModel = new OrderItem();

        $subscriptionModel = new Subscription();

        $hasRecurring = Form::hasRecurring($formId);

        foreach ($submissions as $submission) {
            $data = [
                'id' => $submission->id,
                'user_id' => $submission->user_id,
                'customer_name' => $submission->customer_name,
                'customer_email' => $submission->customer_email,
                'input_data' => $submission->form_data_formatted,
                'created_at' => $submission->created_at,
                'ip_address' => $submission->ip_address,
                'browser' => $submission->browser,
                'device' => $submission->device
            ];
            if ($hasPaymentInputs) {
                $data['currency'] = $submission->currency;
                $data['payment_status'] = $submission->payment_status;
                $data['payment_total'] = number_format($submission->payment_total / 100, 2);
                $data['payment_method'] = $submission->payment_method;
                $data['transactions'] = $transactionModel->getTransactions($submission->id);
                $data['order_items'] = $orderItemModel->getOrderItems($submission->id);
                $data['tax_items'] = $orderItemModel->getTaxOrderItems($submission->id);
            }
            if ($hasRecurring) {
                $data['subscriptions'] = $subscriptionModel->getSubscriptions($submission->id);
            }
            $formattedData[] = $data;
        }
        return $formattedData;
    }

    private function downloadOfficeDoc($data, $type = 'csv', $fileName = null)
    {
        $data = array_map(function ($item) {
            return array_map(function ($itemValue) {
                if (is_array($itemValue)) {
                    return implode(', ', $itemValue);
                }
                return $itemValue;
            }, $item);
        }, $data);
        require_once WPPAYFORMPRO_DIR_PATH . 'src/libs/Spout/Autoloader/autoload.php';
        $fileName = ($fileName) ? $fileName . '.' . $type : 'export-data-' . current_time('d-m-Y') . '.' . $type;
        $writer = \Box\Spout\Writer\WriterFactory::create($type);
        $writer->openToBrowser($fileName);
        $writer->addRows($data);
        $writer->close();
        die();
    }
}
