<?php

namespace WPPayFormPro\GateWays\Paystack;

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\AccessControl;
use WPPayFormPro\GateWays\BasePaymentMethod;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class PaystackSettings extends BasePaymentMethod
{
    public function __construct()
    {
        parent::__construct(
            'paystack',
            'Paystack',
            [],
            WPPAYFORM_URL .'assets/images/gateways/paystack.svg'
        );
    }

     /**
     * @function mapperSettings, To map key => value before store
     * @function validateSettings, To validate before save settings
     */

    public function init()
    {
        add_filter('wppayform_payment_method_settings_mapper_'.$this->key, array($this, 'mapperSettings'));
        add_filter('wppayform_payment_method_settings_validation_' . $this->key, array($this, 'validateSettings'), 10, 2);
    }

    /**
     * @return Array of global fields
     */
    public function globalFields() : array
    {
        return array(
            'is_active' => array(
                'value' => 'no',
                'label' => __('Enable/Disable', 'wp-payment-form'),
            ),
            'payment_mode' => array(
                'value' => 'test',
                'label' => __('Payment Mode', 'wp-payment-form'),
                'options' => array(
                    'test' => __('Test Mode', 'wp-payment-form'),
                    'live' => __('Live Mode', 'wp-payment-form')
                ),
                'type' => 'payment_mode'
            ),
            'live_api_key' => array(
                'value' => 'live',
                'label' => __('Live api key', 'wp-payment-form'),
                'type' => 'live_pub_key',
                'placeholder' => __('Live api key', 'wp-payment-form')
            ),
            'test_api_key' => array(
                'value' => 'test',
                'label' => __('Test api key', 'wp-payment-form'),
                'type' => 'test_pub_key',
                'placeholder' => __('Test api key', 'wp-payment-form')
            ),
            'live_api_secret' => array(
                'value' => '',
                'label' => __('Live Secret Key', 'wp-payment-form'),
                'type' => 'live_secret_key',
                'placeholder' => __('Live Secret Key', 'wp-payment-form')
            ),
            'test_api_secret' => array(
                'value' => '',
                'label' => __('Test Secret Key', 'wp-payment-form'),
                'type' => 'test_secret_key',
                'placeholder' => __('Test Secret Key', 'wp-payment-form')
            ),
            'payment_channels' => array(
                'value' => [],
                'label' => __('Payment Channels', 'wp-payment-form')
            ),
            'is_pro_item' => array(
                'value' => 'yes',
                'label' => __('PayPal', 'wp-payment-form'),
            ),
            'desc' => array(
                'value' => '<p>See our <a href="https://paymattic.com/docs/how-to-integrate-paystack-in-wordpress-with-paymattic/" target="_blank" rel="noopener">documentation</a> to get more information about paystack setup.</p>',
                'type' => 'html_attr',
            ),
        );
    }

    /**
     * @return Array of default fields
     */
    public static function settingsKeys()
    {
        return array(
            'is_active' => 'no',
            'payment_mode' => 'test',
            'checkout_type' => 'modal',
            'test_api_key' => '',
            'test_api_secret' => '',
            'live_api_key' => '',
            'live_api_secret' => '',
            'payment_channels' => []
        );
    }

    /**
     * @return Array of global_payments settings fields
     */
    public function getPaymentSettings()
    {
        $settings = $this->mapper(
            $this->globalFields(), 
            static::getSettings()
        );

        return array(
            'settings' => $settings,
            'is_key_defined' => self::isPaystackKeysDefined()
        );
    }

    public static function getSettings()
    {
        $settings = get_option('wppayform_payment_settings_paystack', array());
        $defaults = [
            'is_active' => 'no',
            'payment_mode' => 'test',
            'checkout_type' => 'modal',
            'test_api_key' => '',
            'test_api_secret' => '',
            'live_api_key' => '',
            'live_api_secret' => '',
            'payment_channels' => []
        ];
        return wp_parse_args($settings, $defaults);
    }

    public function mapperSettings ($settings)
    {
        return $this->mapper(
            static::settingsKeys(), 
            $settings, 
            false
        );
    }

    public static function isPaystackKeysDefined()
    {
        return defined('WP_PAY_FORM_PAYSTACK_SECRET_KEY') && defined('WP_PAY_FORM_PAYSTACK_PUB_KEY');
    }

    public function validateSettings($errors, $settings)
    {
        AccessControl::checkAndPresponseError('set_payment_settings', 'global');

        $mode = Arr::get($settings, 'payment_mode');

        if ($mode == 'test') {
            if (empty(Arr::get($settings, 'test_api_key'))) {
                $errors['test_api_key'] = __('Please provide Test Publishable key and Test Secret Key', 'wp-payment-form-pro');
            }
        }

        if ($mode == 'live') {
            if (empty(Arr::get($settings, 'live_api_key'))) {
                $errors['live_api_key'] = __('Please provide Live Publishable key and Live Secret Key', 'wp-payment-form-pro');
            }
        }
        return $errors;
    }

    public static function isLive($formId = false)
    {
        $settings = self::getSettings();
        $mode = Arr::get($settings, 'payment_mode');
        return $mode == 'live';
    }

    public static function getApiKeys($formId = false)
    {
        $isLive = self::isLive($formId);
        $settings = self::getSettings();
        if ($isLive) {
            return array(
                'api_key' => Arr::get($settings, 'live_api_key'),
                'api_secret' => Arr::get($settings, 'live_api_secret')
            );
        }
        return array(
            'api_key' => Arr::get($settings, 'test_api_key'),
            'api_secret' => Arr::get($settings, 'test_api_secret')
        );
    }
}
