<?php

namespace WPPayFormPro\GateWays\Razorpay;

use WPPayFormPro\GateWays\BasePaymentMethod;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\AccessControl;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class RazorpaySettings extends BasePaymentMethod
{
    /**
     * Automatically create global payment settings page
     * @param  String: key, title, routes_query, 'logo')
     */
    public function __construct()
    {
        parent::__construct(
            'razorpay',
            'Razorpay',
            [],
            WPPAYFORM_URL .'assets/images/gateways/razorpay.svg'
        );
    }
    /**
     * @function mapperSettings, To map key => value before store
     * @function validateSettings, To validate before save settings
     */
    public function init()
    {
        add_filter('wppayform_payment_method_settings_mapper_'.$this->key, array($this, 'mapperSettings'));
        add_filter('wppayform_payment_method_settings_validation_'.$this->key, array($this, 'validateSettings'), 10, 2);
    }

    /**
     * @return Array of global fields
     */
    public function globalFields()
    {
        return array(
            'payment_mode' => array(
                'value' => 'test',
                'label' => __('Payment Mode', 'wp-payment-form'),
                'options' => array(
                    'test' => __('Test Mode', 'wp-payment-form'),
                    'live' => __('Live Mode', 'wp-payment-form')
                ),
                'type' => 'payment_mode'
            ),
            'checkout_type' => array(
                'value' => 'modal',
                'label' => __('Checkout Style', 'wp-payment-form'),
                'options' => ['modal' => 'Modal Checkout', 'hosted' => 'Hosted Checkout'],
                'type' => 'radio'
            ),
            'live_pub_key' => array(
                'value' => 'live',
                'label' => __('Live public key', 'wp-payment-form'),
                'type' => 'live_pub_key',
                'placeholder' => __('Live public key', 'wp-payment-form')
            ),
            'test_pub_key' => array(
                'value' => 'test',
                'label' => __('Test public key', 'wp-payment-form'),
                'type' => 'test_pub_key',
                'placeholder' => __('Test public key', 'wp-payment-form')
            ),
            'live_secret_key' => array(
                'value' => '',
                'label' => __('Live Secret Key', 'wp-payment-form'),
                'type' => 'live_secret_key',
                'placeholder' => __('Live Secret Key', 'wp-payment-form')
            ),
            'test_secret_key' => array(
                'value' => '',
                'label' => __('Test Secret Key', 'wp-payment-form'),
                'type' => 'test_secret_key',
                'placeholder' => __('Test Secret Key', 'wp-payment-form')
            ),
            'notification' => array(
                'value' => [],
                'label' => __('Razorpay notifications', 'wp-payment-form'),
                'type' => 'checkbox_group',
                'desc' => __('Select if you want to enable SMS and Email Notification from razorpay', 'wp-payment-form'),
                'title' => __('Razorpay notifications', 'wp-payment-form'),
                'tooltip' => __('Select if you want to enable SMS and Email Notification from razorpay', 'wp-payment-form'),
                'options' => ['sms' => __('SMS', 'wp-payment-form'), 'email' => __('Email', 'wp-payment-form')],
                'placeholder' => __('Razorpay notifications', 'wp-payment-form'),
            ),
            'is_pro_item' => array(
                'value' => 'yes',
                'label' => __('PayPal', 'wp-payment-form'),
            ),
        );
    }

    /**
     * @return Array of default fields
     */
    public static function settingsKeys()
    {
        return array(
            'payment_mode' => 'test',
            'checkout_type' => 'modal',
            'live_pub_key' => '',
            'live_secret_key' => '',
            'test_pub_key' => '',
            'test_secret_key' => '',
            'notification' => []
        );
    }

    /**
     * @return Array of global_payments settings fields
     */
    public function getPaymentSettings()
    {
        $settings = $this->mapper(
            $this->globalFields(), 
            static::getSettings()
        );

        return array(
            'settings' => $settings,
            'is_key_defined' => self::isRazorpayKeysDefined()
        );
    }

    /**
     * @return Array of settings for checkout
     * Set defaults fields
     */
    public static function getSettings()
    {
        $settings = get_option('wppayform_payment_settings_razorpay', []);
        return wp_parse_args($settings, static::settingsKeys());
    }

    public function mapperSettings ($settings)
    {
        return $this->mapper(
            static::settingsKeys(), 
            $settings, 
            false
        );
    }

    public static function isRazorpayKeysDefined()
    {
        return defined('WP_PAY_FORM_RAZORPAY_SECRET_KEY') && defined('WP_PAY_FORM_RAZORPAY_PUB_KEY');
    }

    public function validateSettings($errors, $settings)
    {
        AccessControl::checkAndPresponseError('set_payment_settings', 'global');
        $mode = Arr::get($settings, 'payment_mode');

        if ($mode == 'test') {
            if (empty(Arr::get($settings, 'test_pub_key')) || empty(Arr::get($settings, 'test_secret_key'))) {
                $errors['test_pub_key'] = __('Please provide Test Publishable key and Test Secret Key', 'wp-payment-form-pro');
            }
        }

        if ($mode == 'live') {
            if (empty(Arr::get($settings, 'live_pub_key')) || empty(Arr::get($settings, 'live_secret_key'))) {
                $errors['live_pub_key'] = __('Please provide Live Publishable key and Live Secret Key', 'wp-payment-form-pro');
            }
        }

        return $errors;
    }

    public static function isLive($formId = false)
    {
        $settings = static::getSettings();
        $mode = Arr::get($settings, 'payment_mode');
        return $mode == 'live';
    }

    public static function getApiKeys($formId = false)
    {
        $isLive = static::isLive($formId);
        $settings = static::getSettings();
        // dd($settings);
        if ($isLive) {
            return array(
                'api_key' => Arr::get($settings, 'live_pub_key'),
                'api_secret' => Arr::get($settings, 'live_secret_key')
            );
        }
        return array(
            'api_key' => Arr::get($settings, 'test_pub_key'),
            'api_secret' => Arr::get($settings, 'test_secret_key')
        );
    }
}
