<?php

namespace WPPayFormPro\GateWays\Square;

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\SubmissionActivity;
use WPPayForm\App\Models\Subscription;
use WPPayForm\App\Models\SubscriptionTransaction;
use WPPayForm\App\Models\Transaction;
use WPPayFormPro\GateWays\Square\SquareProcessor;

class SquareIpn 
{

    public function init() 
    {
        add_action('init', array($this, 'verifyIPN'));
    }

     public function verifyIPN()
    {
        if (!isset($_REQUEST['wpf_square_listener'])) {
            return;
        }

        if (defined('PAYFORM_SQUARE_IPN_DEBUG')) {
            error_log(json_encode($_REQUEST));
        }

        // Check the request method is POST
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] != 'POST') {
            return;
        }

        // Set initial post data to empty string
        $body = file_get_contents('php://input');

       // decode post data to json object
        $event = json_decode($body);

        $eventType = str_replace('.', '_', $event->type);
        $data = $event->data;

        if ($event) {  
            if ($eventType == 'payment_updated') {
                $paymentStatus = $data->object->payment->status;
                if ($paymentStatus === 'COMPLETED') {
                    $this->handlePaymentCompleted($data);
                } else {
                    // for all other payment status changes likes of PENDING, COMPLETED, CANCELED, or FAILED.
                    $this->handlePaymentStatusChange($data);
                }
            } else if ($eventType == 'subscription_created') {
                $this->handleSubscriptionCreation($data);
            } else if ($eventType == 'subscription_updated') {
                $this->handleSubscriptionUpdate($data);
            } else if ($eventType == 'invoice_created') {
                $this->handleInvoiceCreation($data);
            } else if($eventType == 'invoice_payment_made') {
                $this->handleInvoicePaymentMade($data);
            } else if($eventType == 'payment_failed' || $eventType == 'payment_declined') {
                $this->handlePaymentFailed($data);
            }
        } else {
            status_header(500);
            die('-1'); // Failed
        }
        die('1');  
    }

    public function handlePaymentCompleted($data)
    {
        $payment = $data->object->payment;
        $orderId = $payment->order_id;

        //get transaction from database
        $transaction = Transaction::where('charge_id', $orderId)
            ->where('payment_method', 'square')
            ->first();

        if (!$transaction) {
            // Not our transaction
            return;
        }

        $order = (new API())->makeApiCall('orders/'. $orderId, [], $transaction->form_id, '');

        if (!$order || is_wp_error($order) || isset($order['errors'])) {
            return;
        }

        $squareProcessor = new SquareProcessor();
        $squareProcessor->handlePaid(safeUnserialize($transaction), $payment);

    }

    public function handlePaymentStatusChange($data) 
    {
        $payment = $data->object->payment;
        $orderId = $payment->order_id;

        //get transaction from database
        $transaction = Transaction::where('charge_id', $orderId)
            ->where('payment_method', 'square')
            ->first();

        if (!$transaction) {
            // Not our transaction
            return;
        }

        $formId = SquareSettings::getApiKeys($transaction->form_id);

        $order = (new API())->makeApiCall('orders/'. $orderId, [], $formId, '');

        if (!$order || is_wp_error($order) || isset($order['errors'])) {
            return;
        }

        $status = strtolower($data->object->payment->status);

        if (!$transaction || $transaction->payment_method != "square" || $transaction->status === $status) {
            return;
        }
        
        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $updateData = [
            'payment_note'     => maybe_serialize($payment),
            'charge_id'        => $transaction->charge_id,
        ];

        $submissionModel = new Submission();
        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);

        $transactionModel = new Transaction();
        $updateDate = array(
            'charge_id' => $updateData['charge_id'],
            'payment_note' => $updateData['payment_note'],
            'status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );
        
        $formId = Arr::get($transaction, 'form_id');
        $submissionId = Arr::get($transaction, 'submission_id');

        $transactionModel->where('id', $transaction->id)->update($updateDate);
        $transaction = $transactionModel->getTransaction($transaction->id);

        do_action('wppayform_log_data', [
            'form_id' => $formId,
            'submission_id' => $submissionId,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction status updated and Square Transaction ID: %s', 'wp-payment-form-pro'), $updateDate['charge_id'])
        ]);

    }


    public function handleSubscriptionCreation($data) 
    {
        $subscription = $data->object->subscription;
        $planId = Arr::get($subscription, 'plan_id');
        $status = Arr::get($subscription, 'status');
        $subscriptionId = Arr::get($subscription, 'id');

        if (!$subscription || !$planId) {
            return;
        }

        $sub = Subscription::where('vendor_plan_id', $planId)
            ->first();
        $formId = $sub->form_id;

        //verifying subscription
        $checkSubscription = (new API())->makeApiCall('subscriptions/'. $subscriptionId, [], $formId, '');
        if (!$checkSubscription || is_wp_error($checkSubscription) || isset($checkSubscription['errors'])) {
            return;
        }

        $updatedSubscription = Subscription::where('vendor_plan_id', $planId)
            ->update([
                'vendor_subscriptipn_id' => $subscriptionId,
                'status' => strtolower($status),
                'vendor_response' => maybe_serialize($subscription)
            ]);
        
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($updatedSubscription->submission_id);

        SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => __('Square recurring subscription created in square dashboard.', 'wp-payment-form')
        ));

        do_action('wppayform/subscription_created_with_square', $updatedSubscription, $submission);

    }

    public function handleSubscriptionUpdate($data) 
    {
        $subscription = $data->object->subscription;
        $id = Arr::get($subscription, 'id');
        $status = Arr::get($subscription, 'status');
        $data = $subscription;

        $sub = Subscription::where('vendor_subscriptipn_id', $id)
            ->first();
        $formId = $sub->form_id;

        $checkSubscription = (new API())->makeApiCall('subscriptions/'. $id, [], $formId, '');

        if (!$checkSubscription || is_wp_error($checkSubscription) || isset($checkSubscription['errors'])) {
            return;
        }

        $updatedSubscription = Subscription::where('vendor_subscriptipn_id', $id)
            ->update(['status' => strtolower($status)]);

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($updatedSubscription->submission_id);

        SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => __('Square recurring subscription status upadated.', 'wp-payment-form')
        ));

        do_action('wppayform/subscription_status_updated_to_' . $status, $submission, $updatedSubscription, $submission->form_id, $data);
        do_action('wppayform/subscription_payment_' . $status . '_square', $submission, $updatedSubscription, $submission->form_id, $data);
    }

    public function handleInvoiceCreation($data)
    {
        $invoice = $data->object->invoice;
        $vendorSubscriptionId = Arr::get($invoice, 'subscription_id');
        $invoiceId = Arr::get($invoice, 'id');

        if (!$vendorSubscriptionId) {
            return;
        }

        $subscription = Subscription::where('vendor_subscriptipn_id', $vendorSubscriptionId)
            ->first();

        $formId = $subscription->form_id;

        if(!$subscription) {
            // not our subscription
            return;
        }

        // verifying invoice
        $checkInvoice = (new API())->makeApiCall('invoices/'. $invoiceId, [], $formId, '');

        if (!$checkInvoice || is_wp_error($checkInvoice) || isset($checkInvoice['errors'])) {
            return;
        }

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($subscription->submission_id);

        $formId = Arr::get($submission, 'form_id');
        $submissionId = Arr::get($submission, 'id');
        $paymentMode = Arr::get($submission, 'payment_mode');
        $subscriptionId = Arr::get($subscription, 'id');

        $paymentTotal = Arr::get($invoice, 'payment_requests.computed_amount_money.amount');
        $currency = Arr::get($invoice, 'payment_requests.computed_amount_money.currency');

        $currentUserId = get_current_user_id();
        $paymentData = [
            'form_id' => $formId,
            'submission_id' => $submissionId,
            'user_id' => $currentUserId,
            'subscription_id' => $subscriptionId,
            'transaction_type' => 'subscription',
            'payment_method' => 'square',
            'charge_id' => $invoiceId,
            'payment_total' => $paymentTotal,
            'status' => 'pending',
            'currency' => $currency,
            'payment_mode' => $paymentMode,
            'payment_note' => maybe_serialize($invoice)
        ];

        $subscriptionTransactionModel = new SubscriptionTransaction();
        $subscriptionTransactionModel->maybeInsertCharge($paymentData);  
        
        do_action('wppayform/subscription_transaction_creates', $subscriptionId, $invoiceId, $submissionId, $formId);
        do_action('wppayform/subscription_transaction_creates_with_square', $subscriptionId, $invoiceId, $submissionId);
    }

    public function handleInvoicePaymentMade($data) 
    {
        $invoice = $data->object->invoice;
        $vendorSubscriptionId = Arr::get($invoice, 'subscription_id');
        $invoiceId = Arr::get($invoice, 'id');

        if (!$vendorSubscriptionId) {
            return;
        }

        $subscriptionTransaction = SubscriptionTransaction::where('charge_id', $invoiceId)
            ->where('transaction_type', 'subscription')
            ->first();
        
        $formId = $subscriptionTransaction->form_id;

        if (!$subscriptionTransaction) {
            // not our transaction
            return;
        }

        //verifying invoice
        $checkInvoice = (new API())->makeApiCall('invoices/'. $invoiceId, [], $formId, '');
        if (!$checkInvoice || is_wp_error($checkInvoice) || isset($checkInvoice['errors'])) {
            return;
        }
        
        $subscriptionTransactionModel = new SubscriptionTransaction();
        $subscriptionTransaction = $subscriptionTransactionModel->updateSubsTransaction($subscriptionTransaction->id, [
            'status' => 'paid'
        ]);

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($subscriptionTransaction->submission_id);

        SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => __('Square recurring subscription payment made.', 'wp-payment-form')
        ));

        do_action('wppayform/suscription_payment_made', $subscriptionTransaction, $subscriptionTransaction->form_id, $subscriptionTransaction->submission_id);
        do_action('wppayform/suscription_payment_made_square', $subscriptionTransaction, $subscriptionTransaction->form_id, $subscriptionTransaction->submission_id);

    }

    public function handlePaymentFailed($data) 
    {
        $payment = $data->object->payment;
        $orderId = $payment->order_id;

        //get transaction from database
        $transaction = Transaction::where('charge_id', $orderId)
            ->where('payment_method', 'square')
            ->first();

        if (!$transaction) {
            // Not our transaction
            return;
        }

        $formId = $transaction->form_id;

        $order = (new API())->makeApiCall('orders/'. $orderId, [], $formId, '');

        if (!$order || is_wp_error($order) || isset($order['errors'])) {
            return;
        }

        $squareProcessor = new SquareProcessor();
        $squareProcessor->handleFailed(safeUnserialize($transaction), $order);
    }
}