<?php

namespace WPPayFormPro\GateWays\Square;

use WPPayFormPro\GateWays\BasePaymentMethod;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\AccessControl;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

 /**
 * Automatically create global payment settings page
 * @param  String: key, title, routes_query, 'logo')
 */

class SquareSettings extends BasePaymentMethod
{
    public function __construct()
    {
        parent::__construct(
            'square',
            'Square',
            [],
            WPPAYFORM_URL .'assets/images/gateways/square.svg'
        );
    }

    /**
     * @function mapperSettings, To map key => value before store
     * @function validateSettings, To validate before save settings
     */

    public function init()
    {
        add_filter('wppayform_payment_method_settings_mapper_'.$this->key, array($this, 'mapperSettings'));
        add_filter('wppayform_payment_method_settings_validation_' . $this->key, array($this, 'validateSettings'), 10, 2);
    }

        /**
     * @return Array of global fields
     */
    public function globalFields()
    {
        return array (
            'payment_mode' => array(
                'value' => 'test',
                'label' => __('Payment Mode', 'wp-payment-form'),
                'options' => array(
                    'test' => __('Test Mode', 'wp-payment-form'),
                    'live' => __('Live Mode', 'wp-payment-form')
                ),
                'type' => 'payment_mode'
            ),
            'live_access_key' => array(
                'value' => 'live',
                'label' => __('Live Access key', 'wp-payment-form'),
                'type' => 'live_pub_key',
                'placeholder' => __('Live Access key', 'wp-payment-form')
            ),
            'test_access_key' => array(
                'value' => 'test',
                'label' => __('Test Access key', 'wp-payment-form'),
                'type' => 'test_pub_key',
                'placeholder' => __('Test Access key', 'wp-payment-form')
            ),
            'live_location_id' => array(
                'value' => '',
                'label' => __('Live Location Id', 'wp-payment-form'),
                'type' => 'live_secret_key',
                'placeholder' => __('Live Location Id', 'wp-payment-form')
            ),
            'test_location_id' => array(
                'value' => '',
                'label' => __('Test Location Id', 'wp-payment-form'),
                'type' => 'test_secret_key',
                'placeholder' => __('Test Location Id', 'wp-payment-form')
            ),
            'payment_channels' => array(
                'value' => [],
                'label' => __('Payment Channels', 'wp-payment-form')
            ),
            'webhook_desc' => array(
                'value' => "<h3><span style='color: #ef680e; margin-right: 2px'>*</span>Required Square Webhook Setup</h3> <p>In order for Square to function completely for all kind of payments (one time/recurring) , you must configure your Square webhooks. Visit your <a href='https://squareup.com/login?app=developer&return_to=https://developer.squareup.com/' target='_blank' rel='noopener'>account dashboard</a> to configure them. Please add a webhook endpoint for the URL below. </p> <p><b>Webhook URL: </b><code> ". site_url('?wpf_square_listener=1') . "</code></p><div> <p><b>Please subscribe to these following Webhook events for this URL:</b></p> <ul> <li><code>payment.updated</code></li> <li><code>subscription.created</code></li> <li><code>subscription.updated</code></li> <li><code>invoice.created</code></li> <li><code>invoice.payment_made</code></li></ul> </div>",
                'label' => __('Webhook URL', 'wp-payment-form'),
                'type' => 'html_attr',
            ),
            'is_pro_item' => array(
                'value' => 'yes',
                'label' => __('PayPal', 'wp-payment-form'),
            ),
            'desc' => array(
                'value' => '<p>See our <a href="https://paymattic.com/docs/how-to-integrate-square-with-paymattic-in-wordpress/" target="_blank" rel="noopener">documentation</a> to get more information about square setup.</p>',
                'type' => 'html_attr',
            ), 
            // no longer needed
            // 'warning' => array(
            //     'value' => '<p style="color: #ec3609">Due to changes made on sqaure API, now you need to set webhook for all kinds of Payments. See how to configure webhook <a href="https://paymattic.com/docs/how-to-integrate-square-with-paymattic-in-wordpress/#webhook" target="_blank" rel="noopener">here</a>. and please select <span style="color: #000"><b>2022-11-16</b> </span> as your API version For a smooth transaction.</p>',
            //     'type' => 'html_attr',
            // ), 

        );
    }

    /**
     * @return Array of default fields
     */
    public static function settingsKeys()
    {
        return array(
            'payment_mode' => 'test',
            'live_access_key' => '',
            'live_location_id' => '',
            'test_access_key' => '',
            'test_location_id' => ''
        );
    }

    /**
     * @return Array of global_payments settings fields
     */
    public function getPaymentSettings()
    {
        $settings = $this->mapper(
            $this->globalFields(), 
            static::getSettings()
        );

        return array(
            'settings' => $settings,
            'webhook_url' => site_url('?wpf_square_listener=1'),
            'is_key_defined' => self::isSquareKeysDefined()
        );
    }

    public static function getSettings()
    {
        $settings = get_option('wppayform_payment_settings_square', array());
        return wp_parse_args($settings, static::settingsKeys());
    }

    public function mapperSettings ($settings) {
        return $this->mapper(
            static::settingsKeys(), 
            $settings, 
            false
        );
    }

    public static function isSquareKeysDefined()
    {
        return defined('WP_PAY_FORM_SQUARE_SECRET_KEY') && defined('WP_PAY_FORM_SQUARE_PUB_KEY');
    }

    public function validateSettings($errors, $settings)
    {
        AccessControl::checkAndPresponseError('set_payment_settings', 'global');
        $mode = Arr::get($settings, 'payment_mode');

        if ($mode == 'test') {
            if (empty(Arr::get($settings, 'test_access_key')) || empty(Arr::get($settings, 'test_location_id'))) {
                $errors['test_access_key'] = __('Please provide Test Access key and Test Location Id', 'wp-payment-form-pro');
            }
        }

        if ($mode == 'live') {
            if (empty(Arr::get($settings, 'live_access_key')) || empty(Arr::get($settings, 'live_location_id'))) {
                $errors['live_access_key'] = __('Please provide Live Access key and Live Location Id', 'wp-payment-form-pro');
            }
        }

        return $errors;
    }

    public static function isLive($formId = false)
    {
        $settings = self::getSettings();
        $mode = Arr::get($settings, 'payment_mode');
        return $mode == 'live';
    }

    /**
     * deprecated api_url
     * 
     * API version upgraded to v2
     * from https://connect.squareup.com/v2/locations/$liveLocation/
     * to https://connect.squareup.com/v2/
     * till 4.3.5
     */
    public static function getApiKeys($formId = false)
    {
        $isLive = self::isLive($formId);
        $settings = self::getSettings();
        if ($isLive) {
            $liveLocation = Arr::get($settings, 'live_location_id');
            $keys =  array(
                'access_key' => Arr::get($settings, 'live_access_key'),
                'location_id' => $liveLocation,
                'api_url' => "https://connect.squareup.com/v2/"
            );
            return $keys = apply_filters('wppayform/square_api_settings', $keys);
        }

        $testLocation = Arr::get($settings, 'test_location_id');
        $keys = array(
            'access_key' => Arr::get($settings, 'test_access_key'),
            'location_id' => $testLocation,
            'api_url' => "https://connect.squareupsandbox.com/v2/"
        );
        return $keys = apply_filters('wppayform/square_api_settings', $keys);
    }
}
