<?php

namespace WPPayFormPro\Integrations\GoogleSheet;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use WPPayForm\App\Services\Integrations\IntegrationManager;
use WPPayForm\Framework\Foundation\App;
use WPPayForm\Framework\Support\Arr;
use WPPayFormPro\Integrations\GoogleSheet\API\API;
use WPPayFormPro\Integrations\GoogleSheet\API\Sheet;

class Bootstrap extends IntegrationManager
{

    public function __construct()
    {
        parent::__construct(
            App::getInstance(),
            'Google Sheets',
            'google_sheet',
            '_wppayform_google_sheet_settings',
            'google_sheet_notification_feed',
            26
        );

        $this->logo = WPPAYFORM_URL . 'assets/images/integrations/google-sheets.png';
        $this->description = 'Add Paymattic Forms Submission to Google sheets when a form is submitted.';
        $this->registerAdminHooks();
        
        // Occasionally needed this action to be called specially in case of google sheet to receive important credentials to move forward
        add_action('wppayform_authenticate_global_credentials_google_sheet', array($this, 'authenticateCredentials'), 10, 1);
        add_filter('wppayform_save_integration_settings_google_sheet', array($this, 'checkColumnSlugs'), 10, 2);

        // add_filter('wppayform_notifying_async_' . $this->integrationKey, '__return_false');
    }


    public function getGlobalFields($fields)
    {
        $api = new API();

        return [
            'logo' => $this->logo,
            'menu_title' => __('Google Sheets', 'wp-payment-form-pro'),
            'menu_description' => __('Follow these links/docs step by step to get your credentials. Copy them and paste it here respectively, then click on Verify Code button. <a href="https://paymattic.com/docs/integrate-google-sheets-in-wordpress-with-paymattic/" target="_blank">Documentation</a>', 'wp-payment-form-pro'),
            'valid_message' => __('Your Google Access Code is valid', 'wp-payment-form-pro'),
            'invalid_message' => __('Your Google Access Code is not valid', 'wp-payment-form-pro'),
            'save_button_text' => __('Verify Code', 'wp-payment-form-pro'),
            'fields' => [
                'access_code' => [
                    'type' => 'text',
                    'placeholder' => 'Access Code',
                    'label_tips' => __("Enter Google Access Code. Please find this by clicking 'Get Google Sheet Access Code' Button", 'wp-payment-form-pro'),
                    'label' => __('Access Code', 'wp-payment-form-pro'),
                ],
                'button_link' => [
                    'type' => 'link',
                    'link_text' => __('Get Google Sheet Access Code', 'w-payment-form'),
                    'link' => $api->getAUthUrl(),
                    'target' => '_blank',
                    'tips' => __('Please click on this link get get Access Code From Google', 'wp-payment-form'),
                ]
            ],
            'hide_on_valid' => true,
            'discard_settings' => [
                'section_description' => 'Your Google Sheet integration is up and running',
                'button_text' => 'Disconnect Google Sheet',
                'data' => [
                    'access_code' => ''
                ],
                'show_verify' => false
            ]
        ];
    }

    public function getGlobalSettings($settings)
    {
        $globalSettings = get_option($this->optionKey);
        if (!$globalSettings) {
            $globalSettings = [];
        }
        $defaults = [
            'access_code' => ''
        ];

        return wp_parse_args($globalSettings, $defaults);
    }

    public function saveGlobalSettings($settings)
    {

        if (empty($settings['access_code'])) {
           $integrationSettings = [
                'client_id' => '',
                'client_secret' => '',
                'access_code' => '',
                'status' => false
            ];
    
            update_option($this->optionKey, $integrationSettings, 'no');
            wp_send_json_success([
                'message' => __('Your settings has been updated and discarded', 'wp-payment-form-pro'),
                'status' => false
            ], 200);
        }

        // Verify API key now
        try {
            $accessCode = sanitize_textarea_field($settings['access_code']);
            $api = new API();

            $result = $api->generateAccessKey($accessCode);

            if (is_wp_error($result)) {
                throw new \Exception($result->get_error_message());
            }
            

            $previousSettings = get_option($this->optionKey);
            $clientId = $previousSettings['client_id'];
            $clientSecret = $previousSettings['client_secret'];

            $result['client_id'] = $clientId;
            $result['client_secret'] = $clientSecret;
            $result['access_code'] = $accessCode;
            $result['created_at'] = time();
            $result['status'] = true;

            update_option($this->optionKey, $result, 'no');

        } catch (\Exception $exception) {
            wp_send_json_error([
                'message' => $exception->getMessage()
            ], 400);
        }

        wp_send_json_success([
            'message' => __('Your Google Sheet api key has been verified and successfully set', 'wp-payment-form-pro'),
            'status' => true
        ], 200);
    }

    public function pushIntegration($integrations, $formId)
    {
        $integrations[$this->integrationKey] = [
            'title' => 'Google Sheet',
            'logo' => $this->logo,
            'is_active' => $this->isConfigured(),
            'configure_title' => 'Configuration required!',
            'config_url' => admin_url('admin.php?page=wppayform.php#/integrations/google-sheet'),
            'global_configure_url' => admin_url('admin.php?page=wppayform.php#/integrations/google-sheet'),
            'configure_message' => 'Google Sheet is not configured yet! Please configure your Google Sheet api first',
            'configure_button_text' => 'Set Google Sheet API'
        ];
        return $integrations;
    }

    public function getIntegrationDefaults($settings, $formId)
    {
        return [
            'name' => '',
            'spreadsheet_id' => '',
            'work_sheet_id' => '',
            'meta_fields' => [
                (object)array()
            ],
            'conditionals' => [
                'conditions' => [],
                'status' => false,
                'type' => 'all'
            ],
            'trigger_on_payment' => false,
            'enabled' => true
        ];
    }

    public function getSettingsFields($settings, $formId)
    {
        return [
            'fields' => [
                [
                    'key' => 'name',
                    'label' => 'Name',
                    'required' => true,
                    'placeholder' => 'Your Feed Name',
                    'component' => 'text'
                ],
                [
                    'key' => 'spreadsheet_id',
                    'label' => 'Spreadsheet ID',
                    'required' => true,
                    'placeholder' => 'Spreadsheet ID',
                    'component' => 'text',
                    'inline_tip' => '<a href="https://paymattic.com/docs/integrate-google-sheets-in-wordpress-with-paymattic/#1-toc-title" target="blank">Check documentation</a> for how to find google Spreadsheet ID'
                ],
                [
                    'key' => 'work_sheet_id',
                    'label' => 'Worksheet Name',
                    'required' => true,
                    'placeholder' => 'Worksheet Name',
                    'component' => 'text',
                    'inline_tip' => '<a href="https://paymattic.com/docs/integrate-google-sheets-in-wordpress-with-paymattic/#2-toc-title" target="blank">Check documentation</a> for how to find google Worksheet Name'
                ],
                [
                    'key' => 'meta_fields',
                    'label' => 'Spreadsheet Fields',
                    'sub_title' => 'Please specify the meta ',
                    'required' => true,
                    'component' => 'dropdown_label_repeater',
                ],
                [
                    'key' => 'conditionals',
                    'label' => 'Conditional Logics',
                    'tips' => 'Push data to google sheet conditionally based on your submission values',
                    'component' => 'conditional_block'
                ],
                [
                    'key' => 'trigger_on_payment',
                    'require_list' => false,
                    'label' => 'Trigger On',
                    'checkbox_label' => __('Payment Success', 'wp-payment-form-pro'),
                    'component' => 'checkbox-single',
                    'tips' => 'Push data to google sheet when payment is successful, helpful for payment data tracking.'
                ],
                [
                    'require_list' => false,
                    'key' => 'enabled',
                    'label' => 'Status',
                    'component' => 'checkbox-single',
                    'checkbox_label' => 'Enable This feed'
                ]
            ],
            'button_require_list' => false,
            'integration_title' => $this->title
        ];
    }

    public function checkColumnSlugs($settings, $integrationId)
    {
        $settingsTobeReturnedAfterValidation = $settings;

        // check google sheet settings for validation, which is in meta_value as currently constructed.
        $settings = json_decode($settings['meta_value'], true);
        $message = 'Validation Failed';
        // Validate First
        $errors = [];
        if (empty($settings['spreadsheet_id'])) {
            $errors['spreadsheet_id'] = ['Please Provide spreadsheet ID'];
        }
        if (empty($settings['work_sheet_id'])) {
            $errors['work_sheet_id'] = ['Please Provide Worksheet Name'];
        }
        if (empty($settings['meta_fields'])) {
            $errors['meta_fields'] = ['Please Provide Meta Fields Values'];
        }

        if (count($settings['meta_fields']) > 208) {
            $errors['meta_fields'] = ['Spreadsheet Fields can not bet greater than 104'];
            $message = 'Spreadsheet Fields can not bet greater than 104';
        }

        if ($errors) {
            wp_send_json_error([
                'message' => $message,
                'errors' => $errors
            ], 423);
        }

        $keys = [];
        foreach ($settings['meta_fields'] as $index => $meta) {
            if (empty($meta['slug'])) {
                $slug = sanitize_title($meta['label'], 'column_' . $index, 'display');
                if (isset($keys[$slug])) {
                    $slug = $slug . '_' . time() . '_' . mt_rand(1, 100);
                }
                $settings['meta_fields'][$index]['slug'] = $slug;
                $keys[$slug] = $meta['label'];
            } else {
                $keys[$meta['slug']] = $meta['label'];
            }
        }


        // Let's get the sheet Header Now
        $sheet = new Sheet();
        $sheetId = $settings['spreadsheet_id'];
        $workId = $settings['work_sheet_id'];
        $response = $sheet->insertHeader($sheetId, $workId, $keys);
    
        if (is_wp_error($response)) {
            wp_send_json_error([
                'message' => $response->get_error_message(),
                'errors' => $response
            ], 423);
        }

        // we are done here
        return $settingsTobeReturnedAfterValidation;
    }

    public function getMergeFields($list, $listId, $formId)
    {
        return [];
    }

    /*
     * Form Submission Hooks Here
     */
    public function notify($feed, $formData, $entry, $formId)
    {
        // especially for asynchronous notifications
        if( null == gettype($formData) || !$formData) {
            $formData = Arr::get($entry, 'form_data_formatted');
        }
        
        $feedData = $feed['processedValues'];
        $row = [];
        $metaFields = $feedData['meta_fields'];

        if(!$metaFields) {
            return do_action('wppayform_integration_action_result', $feed, 'failed', 'No meta fields found');
        }

        foreach ($metaFields as $field) {
            $row[] = wp_unslash(sanitize_textarea_field(Arr::get($field, 'item_value')));
        }

        $row = apply_filters('wppayform_integration_data_'.$this->integrationKey, $row, $feed, $entry);

        $sheet = new Sheet();
        $response = $sheet->insertRow($feedData['spreadsheet_id'], $feedData['work_sheet_id'], $row);

        if (is_wp_error($response)) {
            $this->addLog($response->get_error_message(), $formId, $entry->id, 'failed');
        } else {
            $this->addLog('Successfully data pushed to google-sheet!', $formId, $entry->id, 'success');
        }
    }
}
