<?php

namespace WPPayFormPro\Integrations\Webhook;

use WPPayForm\App\Services\GeneralSettings;
use WPPayForm\App\Services\Integrations\IntegrationManager;
use WPPayForm\Framework\Foundation\App;
use WPPayForm\App\Models\Meta;
use WPPayForm\Framework\Support\Arr;

class Bootstrap extends IntegrationManager
{
    public $hasGlobalMenu = false;

    public $disableGlobalSettings = 'yes';

    public function __construct()
    {
        parent::__construct(
            App::getInstance(),
            'Webhook',
            'webhook',
            '_wppayform_webhook_settings',
            'webhook_feeds',
            10
        );

        $this->logo = WPPAYFORM_URL . 'assets/images/integrations/webhook.png';

        $this->description = 'Broadcast your Paymattic Submission to any web api endpoint with the powerful webhook module.';
 
        $this->registerAdminHooks();

        $this->category = 'crm';

        add_filter('wppayform_global_notification_feeds', array($this, 'getWebhookFeeds'), 20, 2);
        
        add_filter('wppayform_notifying_async_webhook', '__return_false');

        // add_filter('wppayform_global_addons', array($this, 'addInstruction'));
    }

    public function addInstruction($addOns)
    {
        $instruction = array(
            'webhook'   => array(
                'title'        => 'Webhook',
                'description'  => 'Broadcast your Paymattic Submission to any web api endpoint with the powerful webhook module.',
                'logo'         =>  WPPAYFORM_URL . 'assets/images/integrations/webhook.png',
                'enabled'      => 'no',
                'category'     => $this->category,
                'btnTxt'       => ''
            ),
        );
        return array_merge($addOns, $instruction);
    }

    public function pushIntegration($integrations, $formId)
    {
        $integrations[$this->integrationKey] = [
            'title' => $this->title . ' Integration',
            'logo' => $this->logo,
            'is_active' => $this->isConfigured(),
            'configure_title' => __('Configuration required!', 'wp-payment-form-pro'),
            'global_configure_url' => '#',
            'configure_message' => __('Webhook is not configured yet! Please configure your Webhook api first', 'wp-payment-form-pro'),
            'configure_button_text' => __('Set Webhook', 'wp-payment-form-pro')
        ];
        return $integrations;
    }

    public function getIntegrationDefaults($settings, $formId)
    {
        return [
            'name' => '',
            'request_url' => '',
            'request_method' => 'POST',
            'request_format' => 'FORM',
            'with_header' => 'nop',
            'with_selected_fields' => 'nop', // 'yup' or 'nop
            'request_headers' => [array('key' => '', 'value' => '')],
            'selected_request_fields' => [array('key' => '', 'value' => '')],
            'conditionals' => [
                'conditions' => [],
                'status' => false,
                'type' => 'all'
            ],
            'trigger_on_payment' => false,
            'enabled' => true
        ];
    }

    public function getSettingsFields($settings, $formId)
    {
        return [
            'fields' => [
                [
                    'key' => 'name',
                    'label' => __('Feed Name', 'wp-payment-form-pro'),
                    'required' => true,
                    'placeholder' => __('Your Feed Name', 'wp-payment-form-pro'),
                    'component' => 'text'
                ],
                [
                    'key' => 'request_url',
                    'label' => __('Webhook URL', 'wp-payment-form-pro'),
                    'required' => true,
                    'placeholder' => __('Enter your Webhook URL', 'wp-payment-form-pro'),
                    'tips' => __('Enter the URL where you want to send the Paymattic submission data.', 'wp-payment-form-pro'),
                    'component' => 'text'
                ],
                [
                    'key' => 'request_method',
                    'label' => __('Request Method', 'wp-payment-form-pro'),
                    'required' => false,
                    'placeholder' => __('Select Request Method', 'wp-payment-form-pro'),
                    'tips' => __('Select the HTTP method used for the webhook request.', 'wp-payment-form-pro'),
                    'component' => 'select',
                    'options' => [
                        'POST' => 'POST',
                        'GET' => 'GET',
                        'PUT' => 'PUT',
                        'PATCH' => 'PATCH',
                        'DELETE' => 'DELETE'
                    ]
                ],
                [
                    'key' => 'request_format',
                    'label' => __('Request Format', 'wp-payment-form-pro'),
                    'required' => false,
                    'placeholder' => __('Select the format for the webhook request.', 'wp-payment-form-pro'),
                    'tips' => __('Select the format for the webhook request.', 'wp-payment-form-pro'),
                    'component' => 'select',
                    'options' => [
                        'JSON' => 'JSON',
                        'FORM' => 'FORM',
                    ],
                ],
                [
                    'key' => 'with_header',
                    'label' => __('Request Header', 'wp-payment-form-pro'),
                    'required' => false,
                    'component' => 'radio_choice',
                    'placeholder' => __('Request header', 'wp-payment-form-pro'),
                    'tips' => __('Select with headers if any headers should be sent with the webhook request.', 'wp-payment-form-pro'),
                    'options'      => [
                        'yup' => 'Yes',
                        'nop' => 'No'
                    ],

                ],
                [
                    'key' => 'request_headers',
                    'label' => __('Request Headers', 'wp-payment-form-pro'),
                    'keys'=> ['Header Name', 'Header Value'],
                    'placeholders' => ['Enter Header Name', 'Enter Header Value'],
                    'required' => false,
                    'component' => 'request_info',
                    'options' => Client::getHeaders(),
                ],
                [
                    'key' => 'with_selected_fields',
                    'label' => __('Request Body', 'wp-payment-form-pro'),
                    'required' => true,
                    'component' => 'radio_choice',
                    'placeholder' => __('Request Body', 'wp-payment-form-pro'),
                    'tips' => __('Select Selected body if any specific body should be sent with the webhook request/ Otherwise All fields will be sent.', 'wp-payment-form-pro'),
                    'options'      => [
                        'nop' => 'All Fields',
                        'yup' => 'Selected Fields'
                    ],

                ],
                [
                    'key' => 'selected_request_fields',
                    'label' => __('Request Fields', 'wp-payment-form-pro'),
                    'keys'=> ['Field Name', 'Field Value'],
                    'placeholders' => ['Enter Field Name', 'Enter Field Value'],
                    'required' => false,
                    'component' => 'request_info',
                    'options' => [],
                ],
                [
                    'require_list' => false,
                    'key'          => 'conditionals',
                    'label'        => __('Conditional Logics', 'wp-payment-form-pro'),
                    'tips'         => __('Allow Webhook integration conditionally based on your submission values', 'wp-payment-form-pro'),
                    'component'    => 'conditional_block'
                ],
                [
                    'require_list' => false,
                    'key' => 'enabled',
                    'label' => 'Status',
                    'component' => 'checkbox-single',
                    'checkbox_label' => __('Enable This feed', 'wp-payment-form-pro')
                ],
                [
                    'key' => 'trigger_on_payment',
                    'require_list' => false,
                    'label' => 'Trigger On',
                    'checkbox_label' => __('On Payment Success', 'wp-payment-form-pro'),
                    'component' => 'checkbox-single',
                    'tips' => 'Push data to webhook when payment is successful, helpful for payment data tracking.'
                ],
            ],
            'button_require_list' => false,
            'integration_title' => $this->title
        ];
    }

    public function getMergeFields($list, $listId, $formId)
    {
        return [];
    }

    public function isConfigured()
    {
        return true;
    }

    public function isEnabled()
    {
        return true;
    }

    public function getWebhookFeeds($feeds, $formId)
    {
        $webhookFeeds = Meta::where('form_id', $formId)
        ->where('meta_key', 'webhook')->get();

        if(!$webhookFeeds){
            return $feeds;
        }

        foreach ($webhookFeeds as $feed) {
            $feeds[] = $feed;
        }
        return $feeds;
    }

    public function notify($feed, $formData, $entry, $formId)
    {
        // especially for asynchronous notifications
        if( null == gettype($formData) || !$formData) {
            $formData = Arr::get($entry, 'form_data_formatted');
        }

        $isEnabled = GeneralSettings::isModuleEnabled('webhook');
        if (!$isEnabled) {
            return;
        }
        $response = $this->getApiClient()->notify($feed, $formData, $entry, $formId);

        if (Arr::get($response, 'response.code') === 200) {
            do_action('wppayform_log_data', [
                'form_id' => $formId,
                'submission_id' => $entry->id,
                'type' => 'success',
                'created_by' => 'Paymattic BOT',
                'title' => 'Webhook',
                'content' => "Successfully Webhook payload has been fired."
            ]);
        } else {
            do_action('wppayform_log_data', [
                'form_id' => $formId,
                'submission_id' => $entry->id,
                'type' => 'failed',
                'created_by' => 'Paymattic BOT',
                'title' => 'Webhook',
                'content' => "Webhook feed failed (" . Arr::get($response, 'response.message') . ")"
            ]);
        }
    }

    protected function getApiClient()
    {
        return new Client();
    }
}
