<?php

namespace WPPayFormPro\Integrations\Webhook;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Exception;
use WPPayForm\App\Models\Meta;
use WPPayForm\App\Models\Form;
use WPPayForm\Framework\Foundation\App;

class Client
{
    use NotifyTrait;

    protected $app = null;
    protected $metaKey = 'webhook';
    private $isJsonValue = true;

    public function __construct()
    {
        $this->app = App::getInstance();
        $this->app->request->get('form_id', false);
    }

    public function getAll($formId)
    {
        $settingsQuery = Meta::where('form_id', $formId)
            ->where('meta_key', $this->metaKey)->get();

        foreach ($settingsQuery as $setting) {
            $setting->meta_value = $this->getFormattedValue($setting);
        }
        return $settingsQuery;
    }

    private function getFormattedValue($setting)
    {
        if ($this->isJsonValue) {
            return json_decode($setting->meta_value, true);
        }

        return $setting->value;
    }

    /**
     * Get all notifications/feeds
     * @return JSON Response
     */
    public function getWebHooks($formId)
    {
        $response = [
            'integrations' => $this->getAll($formId),
            'request_headers' => $this->getHeaders(),
            'inputs' => Form::getInputShortcode($formId)    
        ];
        wp_send_json_success($response);
    }

    /**
     * Save GetResponse Feed
     * @return JSON Response
     */
    public function saveWebHook($request)
    {
        $data = $request->data;
        $notification = $data['notification'];
        $notification_id = $data['notification_id'];

        $notification = json_decode($notification, true);

        $notification = $this->validate($notification);

        if ($notification_id) {
            $this->update($notification_id, $notification);
            $message = __('Feed successfully updated', 'wp-payment-form-pro');
        } else {
            $notification_id = $this->save($notification, $request);
            $message = __('Feed successfully created', 'wp-payment-form-pro');
        }

        wp_send_json_success(array(
            'message' => $message,
            'notification_id' => $notification_id
        ), 200);
    }

    /**
     * Delete GetResponse Feed
     * @return JSON Response
     */
    public function deleteWebHook($request, $formId)
    {
        $this->delete($request->data["id"], $formId);
        wp_send_json_success(array(
            'message' => __('Selected WebHook Feed is deleted', 'wppayformpro'),
            'integrations' => $this->getAll($formId)
        ));
    }

    public function save($settings, $request)
    {
        return Meta::insert([
            'meta_key' => $this->metaKey,
            'meta_group' => 'integration',
            'form_id' => $request->data['form_id'],
            'meta_value' => json_encode($settings),
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ]);
    }

    public function update($settingsId, $settings)
    {

        return Meta::where('id', $settingsId)
            ->update(array(
                'meta_value' => json_encode($settings)
            ));
    }

    public function get($settingsId, $formId)
    {
        $settings = Meta::where('form_id', $formId)
            ->where('meta_key', $this->metaKey)
            ->find($settingsId)
            ->get();
        $settings->meta_value = $this->getFormattedValue($settings);
        return $settings;
    }

    public function delete($settingsId, $formId)
    {
        return Meta::where('meta_key', $this->metaKey)
            ->where('form_id', $formId)
            ->where('id', $settingsId)
            ->delete();
    }

    /**
     * Validate inputs
     * @param array $notification
     * @return array $notification
     */
    protected function validate($notification)
    {
        $errors = [];

        if(empty($notification['name'])){
            $errors['name']['required'] = __('Feed name is required', 'wppayformpro');
        }

        if(empty($notification['request_url'])){
            $errors['url']['required'] = __('Url is required', 'wppayformpro');
        }

        if (!$this->validateFields($notification)) {
            $errors['fields']['required'] = __('Field name is required', 'wppayformpro');
        }

        if (!$this->validateHeaders($notification)) {
            $errors['headers']['required'] = __('Header name is required', 'wppayformpro');
        }

        if ($errors) {
            wp_send_json_error(array(
                'errors' => $errors,
                'message' => __('Please fix the errors', 'wppayformpro')
            ), 400);
        }

        return $notification;
    }

    protected function validateFields($notification)
    {
        if ($notification['request_body'] != 'all_fields') {
            foreach ($notification['fields'] as $field) {
                if (empty($field['key'])) {
                    return false;
                }
            }
        }
        return true;
    }

    protected function validateHeaders(&$notification)
    {
        if ($notification['with_header'] == 'yup') {
            foreach ($notification['request_headers'] as $key => &$header) {
                if (empty($header['key'])) {
                    return false;
                }
            }
        }
        return true;
    }

    public static function getHeaders()
    {
        return array(
            array(
                'label' => 'Accept',
                'value' => 'Accept',
                'possible_values' => [
                    'title' => 'Accept Header Samples',
                    'shortcodes' => [
                        'Accept: text/plain' => 'text/plain',
                        'Accept: text/html' => 'text/html',
                        'Accept: text/*' => 'text/*'
                    ]
                ]
            ),
            array(
                'label' => 'Accept-Charset',
                'value' => 'Accept-Charset',
                'possible_values' => [
                    'title' => 'Accept-Charset Header Samples',
                    'shortcodes' => [
                        'Accept-Charset: utf-8' => 'utf-8',
                        'Accept-Charset: iso-8859-1' => 'iso-8859-1'
                    ]
                ]
            ),
            array(
                'label' => 'Accept-Encoding',
                'value' => 'Accept-Encoding',
                'possible_values' => [
                    'title' => 'Accept-Encoding Header Samples',
                    'shortcodes' => [
                        'Accept-Encoding: gzip' => 'gzip',
                        'Accept-Encoding: compress' => 'compress',
                        'Accept-Encoding: deflate' => 'deflate',
                        'Accept-Encoding: br' => 'br',
                        'Accept-Encoding: identity' => 'identity',
                        'Accept-Encoding: *' => '*'
                    ]
                ]
            ),
            array(
                'label' => 'Accept-Language',
                'value' => 'Accept-Language',
                'possible_values' => [
                    'title' => 'Accept-Language Header Samples',
                    'shortcodes' => [
                        'Accept-Language: en' => 'en',
                        'Accept-Language: en-US' => 'en-US',
                        'Accept-Language: en-GR' => 'en-GR',
                        'Accept-Language: en-US,en;q=0.5' => 'en-US,en;q=0.5'
                    ]
                ]
            ),
            array(
                'label' => 'Accept-Datetime',
                'value' => 'Accept-Datetime',
            ),
            array(
                'label' => 'Authorization',
                'value' => 'Authorization',
            ),
            array(
                'label' => 'Cache-Control',
                'value' => 'Cache-Control',
            ),
            array(
                'label' => 'Connection',
                'value' => 'Connection',
            ),
            array(
                'label' => 'Cookie',
                'value' => 'Cookie',
            ),
            array(
                'label' => 'Content-Length',
                'value' => 'Content-Length',
            ),
            array(
                'label' => 'Content-Type',
                'value' => 'Content-Type',
            ),
            array(
                'label' => 'Date',
                'value' => 'Date',
            ),
            array(
                'label' => 'Expect',
                'value' => 'Expect',
            ),
            array(
                'label' => 'Forwarded',
                'value' => 'Forwarded',
            ),
            array(
                'label' => 'From',
                'value' => 'From',
            ),
            array(
                'label' => 'Host',
                'value' => 'Host',
            ),
            array(
                'label' => 'If-Match',
                'value' => 'If-Match',
            ),
            array(
                'label' => 'If-Modified-Since',
                'value' => 'If-Modified-Since',
            ),
            array(
                'label' => 'If-None-Match',
                'value' => 'If-None-Match',
            ),
            array(
                'label' => 'If-Range',
                'value' => 'If-Range',
            ),
            array(
                'label' => 'If-Unmodified-Since',
                'value' => 'If-Unmodified-Since',
            ),
            array(
                'label' => 'Max-Forwards',
                'value' => 'Max-Forwards',
            ),
            array(
                'label' => 'Origin',
                'value' => 'Origin',
            ),
            array(
                'label' => 'Pragma',
                'value' => 'Pragma',
            ),
            array(
                'label' => 'Proxy-Authorization',
                'value' => 'Proxy-Authorization',
            ),
            array(
                'label' => 'Range',
                'value' => 'Range',
            ),
            array(
                'label' => 'Referer',
                'value' => 'Referer',
            ),
            array(
                'label' => 'TE',
                'value' => 'TE',
            ),
            array(
                'label' => 'User-Agent',
                'value' => 'User-Agent',
            ),
            array(
                'label' => 'Upgrade',
                'value' => 'Upgrade',
            ),
            array(
                'label' => 'Via',
                'value' => 'Via',
            ),
            array(
                'label' => 'Warning',
                'value' => 'Warning',
            ),
        );
    }
}