<?php
/**
 * Zynith SEO - Deferment Manager (safe rewrite)
 */

defined('ABSPATH') || exit;

/* -------------------------------------------------------------------------
 * Defaults & helpers
 * ---------------------------------------------------------------------- */

// Global settings defaults.
function zynith_seo_dm_default_globals() {
    return [
        'lazy_loading'           => 'disabled', // 'enabled' => apply defer globally to front-end scripts
        'critical_preloading'    => array(),    // array of absolute/relative script URLs to <link rel="preload">
        'media_lcp_optimization' => 'disabled', // 'enabled' => LCP media preloading
        'custom_media_selector'  => '',         // optional CSS selector to hint the LCP element
    ];
}

// Load global settings with sane defaults.
function zynith_seo_dm_get_globals() {
    $raw = get_option('zynith_global_settings', []);
    return wp_parse_args(is_array($raw) ? $raw : [], zynith_seo_dm_default_globals());
}

/**
 * Load per-script deferment map (src => none|defer|async).
 */
function zynith_seo_dm_get_defer_map() {
    $raw = get_option( 'zynith_deferment_settings', array() );
    return is_array( $raw ) ? $raw : array();
}

/**
 * Load detected HTML scripts list (array of src strings).
 */
function zynith_seo_dm_get_detected() {
    $raw = get_option( 'zynith_detected_scripts', array() );
    return is_array( $raw ) ? $raw : array();
}

/* -------------------------------------------------------------------------
 * Admin menu & page
 * ---------------------------------------------------------------------- */

add_action( 'admin_menu', 'zynith_seo_dm_menu' );
function zynith_seo_dm_menu() {
    add_submenu_page(
        'zynith_seo_dashboard',
        __( 'Deferment Manager', 'zynith-seo' ),
        __( 'Deferment Manager', 'zynith-seo' ),
        'manage_options',
        'zynith-seo-deferment-manager',
        'zynith_seo_dm_render_page'
    );
}

/**
 * Render admin page.
 */
function zynith_seo_dm_render_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $globals    = zynith_seo_dm_get_globals();
    $defer_map  = zynith_seo_dm_get_defer_map();
    $detected   = zynith_seo_dm_get_detected();

    // Handle save.
    if ( 'POST' === $_SERVER['REQUEST_METHOD'] && isset( $_POST['zynith_dm_nonce'] ) ) {
        check_admin_referer( 'zynith_dm_save', 'zynith_dm_nonce' );

        // ---- Global settings ----
        $new_globals                     = zynith_seo_dm_default_globals();
        $new_globals['lazy_loading']     = ! empty( $_POST['global_lazy_loading'] ) ? 'enabled' : 'disabled';
        $new_globals['media_lcp_optimization'] = ! empty( $_POST['global_media_lcp_optimization'] ) ? 'enabled' : 'disabled';
        $new_globals['custom_media_selector']  = isset( $_POST['custom_media_selector'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_media_selector'] ) ) : '';

        $preload_raw = isset( $_POST['global_critical_preloading'] ) ? (string) wp_unslash( $_POST['global_critical_preloading'] ) : '';
        $preload_arr = array_filter( array_map( 'trim', preg_split( "/\r\n|\r|\n/", $preload_raw ) ) );
        $new_globals['critical_preloading'] = array_map( 'esc_url_raw', $preload_arr );

        update_option( 'zynith_global_settings', $new_globals );

        // ---- Per-script settings ----
        $new_map = array();
        if ( ! empty( $_POST['zynith_deferment_settings'] ) && is_array( $_POST['zynith_deferment_settings'] ) ) {
            foreach ( $_POST['zynith_deferment_settings'] as $src => $mode ) {
                $src  = esc_url_raw( wp_unslash( $src ) );
                $mode = sanitize_key( $mode );
                if ( ! $src ) {
                    continue;
                }
                if ( ! in_array( $mode, array( 'none', 'defer', 'async' ), true ) ) {
                    $mode = 'none';
                }
                $new_map[ $src ] = $mode;
            }
        }
        update_option( 'zynith_deferment_settings', $new_map );

        // Refresh in-memory copies for render.
        $globals   = $new_globals;
        $defer_map = $new_map;

        echo '<div class="updated"><p>' . esc_html__( 'Settings saved.', 'zynith-seo' ) . '</p></div>';
    }

    // Gather currently enqueued WP scripts (handles + src).
    global $wp_scripts;
    $enqueued = array();
    if ( $wp_scripts instanceof WP_Scripts ) {
        foreach ( (array) $wp_scripts->queue as $handle ) {
            $src = '';
            if ( isset( $wp_scripts->registered[ $handle ] ) ) {
                $src = $wp_scripts->registered[ $handle ]->src;
                $src = $src ? wp_make_link_relative( $src ) : $src;
                $src = $src ? $wp_scripts->base_url . $src : $wp_scripts->registered[ $handle ]->src; // best effort
            }
            $enqueued[ $handle ] = $src ? $src : $handle;
        }
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__( 'Zynith SEO JS Deferment Manager', 'zynith-seo' ) . '</h1>';
    echo '<form method="post">';
    wp_nonce_field( 'zynith_dm_save', 'zynith_dm_nonce' );

    // ---- Global Settings ----
    echo '<h2>' . esc_html__( 'Global Settings', 'zynith-seo' ) . '</h2>';
    echo '<table class="form-table"><tbody>';

    // Lazy loading (implemented as defer) toggle.
    echo '<tr><th scope="row">' . esc_html__( 'Global Defer', 'zynith-seo' ) . '</th><td>';
    echo '<label><input type="checkbox" name="global_lazy_loading" ' . checked( $globals['lazy_loading'], 'enabled', false ) . ' /> ';
    echo esc_html__( 'Add defer to all front-end scripts (excluding admin).', 'zynith-seo' ) . '</label>';
    echo '</td></tr>';

    // Critical preloading.
    echo '<tr><th scope="row">' . esc_html__( 'Critical Script Preloading', 'zynith-seo' ) . '</th><td>';
    echo '<textarea name="global_critical_preloading" rows="4" cols="70" placeholder="https://example.com/app.js&#10;/wp-content/themes/child/js/hero.js">' . esc_textarea( implode( "\n", $globals['critical_preloading'] ) ) . '</textarea>';
    echo '<p class="description">' . esc_html__( 'One script URL per line. These will be <link rel="preload" as="script"> in <head>.', 'zynith-seo' ) . '</p>';
    echo '</td></tr>';

    // Media LCP optimization.
    echo '<tr><th scope="row">' . esc_html__( 'Media LCP Optimization', 'zynith-seo' ) . '</th><td>';
    echo '<label><input type="checkbox" name="global_media_lcp_optimization" ' . checked( $globals['media_lcp_optimization'], 'enabled', false ) . ' /> ';
    echo esc_html__( 'Enable LCP media preloading (server & JS fallback).', 'zynith-seo' ) . '</label>';
    echo '<div style="margin-top:8px">';
    echo '<label>' . esc_html__( 'Optional custom selector (e.g. .hero-img, #main-hero)', 'zynith-seo' ) . ' ';
    echo '<input type="text" name="custom_media_selector" value="' . esc_attr( $globals['custom_media_selector'] ) . '" class="regular-text" /></label>';
    echo '</div>';
    echo '</td></tr>';

    echo '</tbody></table>';

    // ---- WordPress-enqueued scripts ----
    echo '<h2>' . esc_html__( 'WordPress-Enqueued Scripts', 'zynith-seo' ) . '</h2>';
    echo '<table class="widefat fixed striped"><thead><tr>';
    echo '<th>' . esc_html__( 'Handle', 'zynith-seo' ) . '</th>';
    echo '<th>' . esc_html__( 'Source', 'zynith-seo' ) . '</th>';
    echo '<th>' . esc_html__( 'Setting', 'zynith-seo' ) . '</th>';
    echo '</tr></thead><tbody>';

    if ( $enqueued ) {
        foreach ( $enqueued as $handle => $src ) {
            $src_key = $src ? $src : (string) $handle;
            $setting = isset( $defer_map[ $src_key ] ) ? $defer_map[ $src_key ] : 'none';
            echo '<tr>';
            echo '<td>' . esc_html( $handle ) . '</td>';
            echo '<td style="word-break:break-all">' . esc_html( $src_key ) . '</td>';
            echo '<td><select name="zynith_deferment_settings[' . esc_attr( $src_key ) . ']">';
            echo '<option value="none"' . selected( $setting, 'none', false ) . '>' . esc_html__( 'None', 'zynith-seo' ) . '</option>';
            echo '<option value="defer"' . selected( $setting, 'defer', false ) . '>' . esc_html__( 'Defer', 'zynith-seo' ) . '</option>';
            echo '<option value="async"' . selected( $setting, 'async', false ) . '>' . esc_html__( 'Async', 'zynith-seo' ) . '</option>';
            echo '</select></td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="3">' . esc_html__( 'No enqueued scripts found on this admin load.', 'zynith-seo' ) . '</td></tr>';
    }
    echo '</tbody></table>';

    // ---- HTML-detected scripts ----
    echo '<h2>' . esc_html__( 'HTML-Detected Scripts', 'zynith-seo' ) . '</h2>';
    echo '<table class="widefat fixed striped"><thead><tr>';
    echo '<th>' . esc_html__( 'Script Source', 'zynith-seo' ) . '</th>';
    echo '<th>' . esc_html__( 'Setting', 'zynith-seo' ) . '</th>';
    echo '</tr></thead><tbody>';

    if ( $detected ) {
        foreach ( $detected as $src ) {
            $setting = isset( $defer_map[ $src ] ) ? $defer_map[ $src ] : 'none';
            echo '<tr>';
            echo '<td style="word-break:break-all">' . esc_html( $src ) . '</td>';
            echo '<td><select name="zynith_deferment_settings[' . esc_attr( $src ) . ']">';
            echo '<option value="none"' . selected( $setting, 'none', false ) . '>' . esc_html__( 'None', 'zynith-seo' ) . '</option>';
            echo '<option value="defer"' . selected( $setting, 'defer', false ) . '>' . esc_html__( 'Defer', 'zynith-seo' ) . '</option>';
            echo '<option value="async"' . selected( $setting, 'async', false ) . '>' . esc_html__( 'Async', 'zynith-seo' ) . '</option>';
            echo '</select></td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="2">' . esc_html__( 'No scripts detected from HTML yet.', 'zynith-seo' ) . '</td></tr>';
    }
    echo '</tbody></table>';

    submit_button( __( 'Save Changes', 'zynith-seo' ) );
    echo '</form>';
    echo '</div>';
}

/* -------------------------------------------------------------------------
 * Apply async/defer (and optional global defer)
 * ---------------------------------------------------------------------- */

/**
 * Adds async/defer attributes per per-script settings map.
 * If global "lazy_loading" is enabled, adds defer to all front-end scripts
 * unless already async/defer by map.
 */
add_filter( 'script_loader_tag', 'zynith_seo_dm_script_attrs', 10, 3 );
function zynith_seo_dm_script_attrs( $tag, $handle, $src ) {
    if ( is_admin() ) {
        return $tag;
    }

    $globals   = zynith_seo_dm_get_globals();
    $defer_map = zynith_seo_dm_get_defer_map();

    // Normalize key to match how we store map keys (raw URL string).
    $key = $src;

    $mode = isset( $defer_map[ $key ] ) ? $defer_map[ $key ] : 'none';

    // Per-script wins.
    if ( 'async' === $mode ) {
        return str_replace( '<script ', '<script async ', $tag );
    }
    if ( 'defer' === $mode ) {
        return str_replace( '<script ', '<script defer ', $tag );
    }

    // Global lazy (defer everything) if enabled.
    if ( 'enabled' === $globals['lazy_loading'] ) {
        return str_replace( '<script ', '<script defer ', $tag );
    }

    return $tag;
}

/* -------------------------------------------------------------------------
 * Preload critical scripts
 * ---------------------------------------------------------------------- */
add_action( 'wp_head', 'zynith_seo_dm_preload_critical', 1 );
function zynith_seo_dm_preload_critical() {
    $globals = zynith_seo_dm_get_globals();
    if ( empty( $globals['critical_preloading'] ) || ! is_array( $globals['critical_preloading'] ) ) {
        return;
    }
    foreach ( $globals['critical_preloading'] as $url ) {
        if ( $url ) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '<link rel="preload" href="' . esc_url( $url ) . '" as="script">' . "\n";
        }
    }
}

/* -------------------------------------------------------------------------
 * Media LCP optimization (server hint + JS fallback)
 * ---------------------------------------------------------------------- */
add_action( 'wp_head', 'zynith_seo_dm_media_lcp', 2 );
function zynith_seo_dm_media_lcp() {
    $globals = zynith_seo_dm_get_globals();
    $media_lcp = $globals['media_lcp_optimization'] ?? 'disabled';
    if ($media_lcp !== 'enabled') return;

    $custom_selector = $globals['custom_media_selector'];

    // Output buffer to prepend discovered <link> tags. We RETURN the modified HTML.
    ob_start( function ( $html ) use ( $custom_selector ) {
        $prepend = '';

        // Background-image detector for common hero elements.
        $regex = '/
            <(?:img|div|section|header|video)[^>]*                              
            (?:class=["\'][^"\']*(hero|main-image|banner|cover|featured|background|back-ground|bckgrnd)[^"\']*["\']|
               id=["\'][^"\']*(hero|main-image|featured-image|banner)[^"\']*["\']|
               style=["\'][^"\']*background-image:\s*url\((["\']?)([^"\')]+)\2\)[^"\']*["\'])
            [^>]*>
        /ix';

        if ( preg_match( $regex, $html, $m ) ) {
            $bg = isset( $m[3] ) ? $m[3] : '';
            if ( $bg ) {
                $prepend .= '<link rel="preload" href="' . esc_url( $bg ) . '" as="image">' . "\n";
            }
        }

        // Optional custom selector hint.
        if ( $custom_selector ) {
            $sel = preg_quote( $custom_selector, '/' );
            if ( preg_match( '/<[^>]*' . $sel . '[^>]*>/i', $html, $match ) ) {
                // Try src
                if ( preg_match( '/src=["\']([^"\']+)["\']/', $match[0], $sm ) ) {
                    $prepend .= '<link rel="preload" href="' . esc_url( $sm[1] ) . '" as="image">' . "\n";
                }
                // Or background
                elseif ( preg_match( '/background-image:\s*url\((["\']?)([^"\')]+)\1\)/i', $match[0], $bm ) ) {
                    $prepend .= '<link rel="preload" href="' . esc_url( $bm[2] ) . '" as="image">' . "\n";
                }
            }
        }

        // First <video src> found.
        if ( preg_match( '/<video[^>]*src=["\']([^"\']+)["\']/', $html, $vm ) ) {
            $prepend .= '<link rel="preload" href="' . esc_url( $vm[1] ) . '" as="video">' . "\n";
        }

        return $prepend . $html;
    } );

    // JS fallback (safe to echo here).
    ?>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        if (!('PerformanceObserver' in window)) return;
        try {
            const observer = new PerformanceObserver((list) => {
                const entries = list.getEntries();
                const lcp = entries.find((e) => {
                    if (!e.element) return false;
                    const el  = e.element;
                    const cls = (el.className || '').toString();
                    const id  = (el.id || '').toString();
                    const tag = (el.tagName || '').toUpperCase();
                    if (/hero|main-image|banner|featured|background|back-ground|bckgrnd/i.test(cls)) return true;
                    if (/hero|main-image|banner|featured|background|back-ground|bckgrnd/i.test(id)) return true;
                    if (tag === 'IMG' || tag === 'VIDEO') return true;
                    const bg = getComputedStyle(el).backgroundImage;
                    return bg && bg.includes('url(');
                });
                if (!lcp) return;

                const el = lcp.element;
                const add = (href, as) => {
                    if (!href) return;
                    var link = document.createElement('link');
                    link.rel = 'preload';
                    link.as = as;
                    link.href = href;
                    document.head.appendChild(link);
                };

                if (el.tagName === 'IMG')  add(el.getAttribute('src'), 'image');
                if (el.tagName === 'VIDEO') add(el.getAttribute('src'), 'video');

                const bg = getComputedStyle(el).backgroundImage;
                const m  = bg && bg.match(/url\(["']?(.*?)["']?\)/);
                if (m && m[1]) add(m[1], 'image');
            });
            observer.observe({ type: 'largest-contentful-paint', buffered: true });
        } catch (e) { /* no-op */ }
    });
    </script>
    <?php
}

/* -------------------------------------------------------------------------
 * Detect <script src="..."> in raw HTML and store (front-end only)
 * ---------------------------------------------------------------------- */

add_action( 'template_redirect', 'zynith_seo_dm_detect_html_scripts', 0 );
function zynith_seo_dm_detect_html_scripts() {
    if ( is_admin() || is_feed() || ( defined('DOING_AJAX') && DOING_AJAX ) ) {
        return;
    }

    ob_start( function ( $html ) {
        $regex = '/<script[^>]*\s+src=["\']([^"\']+)["\']/i';
        if ( ! preg_match_all( $regex, $html, $matches ) ) {
            return $html;
        }

        $existing = zynith_seo_dm_get_detected();
        $found    = array_map( 'esc_url_raw', array_filter( array_map( 'trim', $matches[1] ) ) );
        $merged   = array_values( array_unique( array_merge( $existing, $found ) ) );

        // Save only if changed to avoid unnecessary writes.
        if ( $merged !== $existing ) {
            update_option( 'zynith_detected_scripts', $merged );
        }

        return $html;
    } );
}