<?php

namespace Workdo\MobileServiceManagement\DataTables;

use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

use App\Models\Invoice;
use Workdo\MobileServiceManagement\Entities\MobileServiceRequest;

class MobileServiceRequestDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $rowColumn = ['service_id', 'priority', 'tracking_status', 'is_technician_asign', 'payment_status', 'getServiceCreatedName', 'created_at'];

        $dataTable = (new EloquentDataTable($query))
            ->addIndexColumn()
            ->editColumn('service_id', function (MobileServiceRequest $data) {
                $href = \Laratrust::hasPermission('mobileservice_servicerequest show') ? route('mobileservice.service-request.show', ['serviceId' => encrypt($data->service_id)]) : "#";
                return '<a class="btn btn-outline-primary"  href="' . $href . '" >' . $data->service_id . '</a>';
            })
            ->editColumn('priority', function (MobileServiceRequest $data) {
                if ($data->priority == 'priority') {
                    $class = "bg-warning";
                } elseif ($data->priority == 'emergency') {
                    $class = "bg-danger";
                } else {
                    $class = "bg-secondary";
                }
                return '<span class="badge fix_badge ' . $class . '  p-2 px-3">' . __(ucfirst($data->priority)) . '</span>';
            })
            ->editColumn('tracking_status', function (MobileServiceRequest $data) {
                $statusColor = isset($data->getServiceTrackingStatus->status_color) ? '#' . $data->getServiceTrackingStatus->status_color : '#ffc107';
                $statusName = isset($data->getServiceTrackingStatus->status_name) ? $data->getServiceTrackingStatus->status_name : 'pending';
                return '<span class="badge fix_badge p-2 px-3" style="background-color: ' . $statusColor . '; color: white;">' . $statusName . '</span>';
            })
            ->filterColumn('tracking_status', function ($query, $keyword) {
                if (stripos('pending', $keyword) !== false) {
                    $query->where('tracking_status', 'pending');
                } else {
                    $query->whereHas('getServiceTrackingStatus', function ($q) use ($keyword) {
                        $q->where('status_name', 'LIKE', "%$keyword%")
                            ->orwhere('status_name', '=', "%$keyword%");
                    });
                }
            })
            ->editColumn('is_technician_asign', function (MobileServiceRequest $data) {
                $technicianColor = (!is_null($data->getAssignTechnician) && !is_null($data->getAssignTechnician->getTechnicianDetails) && !is_null($data->getAssignTechnician->getTechnicianDetails->name)) ? 'bg-success' : 'bg-warning';
                $technicianName = (!is_null($data->getAssignTechnician) && !is_null($data->getAssignTechnician->getTechnicianDetails) && !is_null($data->getAssignTechnician->getTechnicianDetails->name)) ? $data->getAssignTechnician->getTechnicianDetails->name : __('Not Assigned');
                return '<span class="badge fix_badge ' . $technicianColor . ' p-2 px-3">' . $technicianName . '</span>';
            })
            ->filterColumn('is_technician_asign', function ($query, $keyword) {
                if (stripos('Not Assigned', $keyword) !== false) {
                    $query->where('is_technician_asign', 0);
                } else {
                    $query->whereHas('getAssignTechnician', function ($q) use ($keyword) {
                        $q->whereHas('getTechnicianDetails', function ($q) use ($keyword) {
                            $q->where('name', 'LIKE', "%$keyword%");
                        });
                    });
                }
            })
            ->editColumn('payment_status', function (MobileServiceRequest $data) {
                if (isset($data->getInvoiceStatus->status)) {
                    if ($data->getInvoiceStatus->status == 0) {
                        $class = 'bg-info';
                    } elseif ($data->getInvoiceStatus->status == 1) {
                        $class = 'bg-primary';
                    } elseif ($data->getInvoiceStatus->status == 2) {
                        $class = 'bg-secondary';
                    } elseif ($data->getInvoiceStatus->status == 3) {
                        $class = 'bg-warning';
                    } elseif ($data->getInvoiceStatus->status == 4) {
                        $class = 'bg-success';
                    }
                    $paymentStatus = Invoice::$statues[$data->getInvoiceStatus->status];
                } else {
                    $class         = 'bg-dark';
                    $paymentStatus = __('No Invoice');
                }

                $html = '<span class="badge fix_badges ' . $class . ' p-2 px-3">' . $paymentStatus . '</span>';
                return $html;
            })
            ->filterColumn('payment_status', function ($query, $keyword) {
                if (stripos('Draft', $keyword) !== false) {
                    $query->whereHas('getInvoiceStatus', function ($q) use ($keyword) {
                        $q->where('status', '=', 0);
                    });
                } elseif (stripos('Sent', $keyword) !== false) {
                    $query->whereHas('getInvoiceStatus', function ($q) use ($keyword) {
                        $q->where('status', '=', 1);
                    });
                } elseif (stripos('Unpaid', $keyword) !== false) {
                    $query->whereHas('getInvoiceStatus', function ($q) use ($keyword) {
                        $q->where('status', '=', 2);
                    });
                } elseif (stripos('Partialy Paid', $keyword) !== false) {
                    $query->whereHas('getInvoiceStatus', function ($q) use ($keyword) {
                        $q->where('status', '=', 3);
                    });
                } elseif (stripos('Paid', $keyword) !== false) {
                    $query->whereHas('getInvoiceStatus', function ($q) use ($keyword) {
                        $q->where('status', '=', 4);
                    });
                }
            })
            ->editColumn('getServiceCreatedName', function (MobileServiceRequest $data) {
                $getServiceCreatedName = $data->getServiceCreatedName ? $data->getServiceCreatedName->name ?? '--' : '---';
                return '<span class="white-space text-primary">' . $getServiceCreatedName . '</span>';
            })
            ->editColumn('created_at', function (MobileServiceRequest $data) {
                $html = '<span data-bs-toggle="tooltip" title="' . $data->created_at->diffForHumans() . '" data-title="' . $data->created_at->diffForHumans() . '">' . company_date_formate($data->created_at) . '</span>';
                return $html;
            })
            ->filterColumn('created_at', function ($query, $keyword) {
                try {
                    if (\Carbon\Carbon::hasFormat($keyword, 'd-m-Y')) {
                        $date = \Carbon\Carbon::createFromFormat('d-m-Y', $keyword)->format('Y-m-d');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } elseif (\Carbon\Carbon::hasFormat($keyword, 'm-Y')) {
                        $date = \Carbon\Carbon::createFromFormat('m-Y', $keyword)->format('Y-m');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } elseif (\Carbon\Carbon::hasFormat($keyword, 'd-m')) {
                        $date = \Carbon\Carbon::createFromFormat('d-m', $keyword)->format('m-d');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } else {
                        $hasDay   = false;
                        $hasMonth = false;
                        $hasYear = false;
                        if (\Carbon\Carbon::hasFormat($keyword, 'd') && strlen($keyword) <= 2) {
                            $day = \Carbon\Carbon::createFromFormat('d', $keyword)->format('d');
                            $query->whereRaw('DAY(created_at) = ?', [$day]);
                            $hasDay = true;
                        }
                        if (\Carbon\Carbon::hasFormat($keyword, 'm') && strlen($keyword) <= 2) {
                            $month = \Carbon\Carbon::createFromFormat('m', $keyword)->format('m');
                            $query->orWhereRaw('MONTH(created_at) = ?', [$month]);
                            $hasMonth = true;
                        }
                        if (preg_match('/^\d{4}$/', $keyword)) {
                            $year = \Carbon\Carbon::createFromFormat('Y', $keyword)->format('Y');
                            $query->orWhereRaw('YEAR(created_at) = ?', [$year]);
                            $hasYear = true;
                        }

                        if ($hasDay || $hasMonth || $hasYear) {
                            return $query;
                        }
                    }
                } catch (\Exception $e) {
                    \Log::error('Invalid date format: ' . $keyword);
                }
            });

        if (
            \Laratrust::hasPermission('mobileservice_servicerequest end') ||
            \Laratrust::hasPermission('mobileservice_servicerequest technician assign') ||
            \Laratrust::hasPermission('mobileservice_servicerequest show') ||
            \Laratrust::hasPermission('mobileservice_servicerequest edit') ||
            \Laratrust::hasPermission('mobileservice_servicerequest delete')
        ) {
            $dataTable->addColumn('action', function (MobileServiceRequest $data) {
                return view('mobile-service-management::service-request.action', compact('data'));
            });
            $rowColumn[] = 'action';
        }
        return $dataTable->rawColumns($rowColumn);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(MobileServiceRequest $model): QueryBuilder
    {
        $requestData = $model->where('is_approve', 1)
            ->where('is_repair_end', 0)
            ->where('mobile_service_requests.workspace_id', getActiveWorkSpace())
            ->with('getServiceCreatedName', 'getServiceTrackingStatus', 'getAssignTechnician.getTechnicianDetails', 'getInvoiceStatus')
            ->Select('mobile_service_requests.*');


        if (Auth::user()->type != 'company') {
            $requestData->where('mobile_service_requests.created_by', Auth::user()->id);
        }

        return $requestData;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        $dataTable = $this->builder()
            ->setTableId('mobile-service-request-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0)
            ->language([
                "paginate"          => [
                    "next"     => '<i class="ti ti-chevron-right"></i>',
                    "previous" => '<i class="ti ti-chevron-left"></i>'
                ],
                'lengthMenu'        => "_MENU_" . __('Entries Per Page'),
                "searchPlaceholder" => __('Search...'),
                "search"            => "",
                "info"              => __('Showing _START_ to _END_ of _TOTAL_ entries')
            ])
            ->initComplete('function() {
                var table = this;
                var searchInput = $(\'#\'+table.api().table().container().id+\' label input[type="search"]\');
                searchInput.removeClass(\'form-control form-control-sm\');
                searchInput.addClass(\'dataTable-input\');
                var select = $(table.api().table().container()).find(".dataTables_length select").removeClass(\'custom-select custom-select-sm form-control form-control-sm\').addClass(\'dataTable-selector\');
            }');

        $exportButtonConfig = [
            'extend'    => 'collection',
            'className' => 'btn btn-light-secondary dropdown-toggle',
            'text'      => '<i class="ti ti-download me-2" data-bs-toggle="tooltip" data-bs-placement="top" data-bs-original-title="Export"></i>',
            'buttons'   => [
                [
                    'extend'        => 'print',
                    'text'          => '<i class="fas fa-print me-2"></i> ' . __('Print'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
                [
                    'extend'        => 'csv',
                    'text'          => '<i class="fas fa-file-csv me-2"></i> ' . __('CSV'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
                [
                    'extend'        => 'excel',
                    'text'          => '<i class="fas fa-file-excel me-2"></i> ' . __('Excel'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
            ],
        ];

        $buttonsConfig = array_merge([
            $exportButtonConfig,
            [
                'extend'    => 'reset',
                'className' => 'btn btn-light-danger',
            ],
            [
                'extend'    => 'reload',
                'className' => 'btn btn-light-warning',
            ],
        ]);

        $dataTable->parameters([
            "dom"          => "
        <'dataTable-top'<'dataTable-dropdown page-dropdown'l><'dataTable-botton table-btn dataTable-search tb-search  d-flex justify-content-end gap-2'Bf>>
        <'dataTable-container'<'col-sm-12'tr>>
        <'dataTable-bottom row'<'col-5'i><'col-7'p>>",
            'buttons'      => $buttonsConfig,
            "drawCallback" => 'function( settings ) {
                var tooltipTriggerList = [].slice.call(
                    document.querySelectorAll("[data-bs-toggle=tooltip]")
                  );
                  var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                    return new bootstrap.Tooltip(tooltipTriggerEl);
                  });
                  var popoverTriggerList = [].slice.call(
                    document.querySelectorAll("[data-bs-toggle=popover]")
                  );
                  var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
                    return new bootstrap.Popover(popoverTriggerEl);
                  });
                  var toastElList = [].slice.call(document.querySelectorAll(".toast"));
                  var toastList = toastElList.map(function (toastEl) {
                    return new bootstrap.Toast(toastEl);
                  });
            }'
        ]);

        $dataTable->language([
            'buttons' => [
                'create' => __('Create'),
                'export' => __('Export'),
                'print'  => __('Print'),
                'reset'  => __('Reset'),
                'reload' => __('Reload'),
                'excel'  => __('Excel'),
                'csv'    => __('CSV'),
            ]
        ]);

        return $dataTable;
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $column = [
            Column::make('id')->searchable(false)->visible(false)->exportable(false)->printable(false),
            Column::make('No')->title(__('No'))->data('DT_RowIndex')->name('DT_RowIndex')->searchable(false)->orderable(false),
            Column::make('service_id')->title(__('Service ID')),
            Column::make('customer_name')->title(__('Customer Name')),
            Column::make('priority')->title(__('Priority')),
            Column::make('tracking_status')->title(__('Tracking Status')),
            Column::make('is_technician_asign')->title(__('Technician')),
            Column::make('payment_status')->title(__('Payment Status'))->orderable(false),
            Column::make('getServiceCreatedName')->title(__('Created By'))->name('getServiceCreatedName.name'),
            Column::make('created_at')->title(__('Created')),
        ];
        if (
            \Laratrust::hasPermission('mobileservice_servicerequest end') ||
            \Laratrust::hasPermission('mobileservice_servicerequest technician assign') ||
            \Laratrust::hasPermission('mobileservice_servicerequest show') ||
            \Laratrust::hasPermission('mobileservice_servicerequest edit') ||
            \Laratrust::hasPermission('mobileservice_servicerequest delete')
        ) {
            $action = [
                Column::computed('action')
                    ->title(__('Action'))
                    ->exportable(false)
                    ->printable(false)
                    ->addClass('text-end')
                    ->width(60),
            ];

            $column = array_merge($column, $action);
        }

        return $column;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Mobile_Service_Request' . date('YmdHis');
    }
}
