<?php

namespace Workdo\MobileServiceManagement\Http\Controllers;


use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

use Workdo\MobileServiceManagement\Entities\MobileServiceCategory;

use Workdo\MobileServiceManagement\Events\CreateMobileServiceCategory;
use Workdo\MobileServiceManagement\Events\UpdateMobileServiceCategory;
use Workdo\MobileServiceManagement\Events\DestroyMobileServiceCategory;

class MobileServiceCategoryController extends Controller
{
    public function index(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory manage')) {
            $categoryData = MobileServiceCategory::where('created_by', creatorId())->where('workspace', getActiveWorkSpace())->get();
            return view('mobile-service-management::service-setup.service-category.index', compact('categoryData'));
        } elseif (Auth::user()->isAbleTo('mobileservice_devicebrand manage')) {
            return redirect()->route('mobileservice.brands.index');
        } elseif (Auth::user()->isAbleTo('mobileservice_trackingstatus manage')) {
            return redirect()->route('mobileservice.tracking-status.index');
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory create')) {
            return view('mobile-service-management::service-setup.service-category.create');
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory create')) {
            $validator = Validator::make($request->all(), [
                'category_name' => 'required|string'
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $category                = new MobileServiceCategory();
            $category->category_name = $request->category_name;
            $category->workspace     = getActiveWorkSpace();
            $category->created_by    = creatorId();
            $category->save();

            event(new CreateMobileServiceCategory($request, $category));

            return redirect()->back()->with('success', __('The service category has been created successfully'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(Request $request, $categoryId)
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory edit')) {
            $categoryData = MobileServiceCategory::where('id', $categoryId)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($categoryData) {
                return view('mobile-service-management::service-setup.service-category.edit', compact('categoryData'));
            } else {
                return response()->json(['error' => __('Service category not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, $categoryId)
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory edit')) {
            $validator = Validator::make($request->all(), [
                'category_name' => 'required|string'
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $categoryData = MobileServiceCategory::where('id', $categoryId)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($categoryData) {
                $categoryData->category_name = $request->category_name;
                $categoryData->save();

                event(new UpdateMobileServiceCategory($request, $categoryData));

                return redirect()->back()->with('success', __('The service category details are updated successfully'));
            } else {
                return redirect()->back()->with('error', __('Service category not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy($categoryId)
    {
        if (Auth::user()->isAbleTo('mobileservice_servicecategory delete')) {
            $categoryData = MobileServiceCategory::where('id', $categoryId)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($categoryData) {
                event(new DestroyMobileServiceCategory($categoryData));
                $categoryData->delete();

                return redirect()->back()->with('success', __('The category has been deleted'));
            } else {
                return redirect()->back()->with('error', __('Service category not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}
