<?php

namespace Workdo\MobileServiceManagement\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

use Workdo\MobileServiceManagement\DataTables\MobileServicePolicyDataTable;

use Workdo\MobileServiceManagement\Entities\MobileServicePolicy;

use Workdo\MobileServiceManagement\Events\CreateMobileServicePolicy;
use Workdo\MobileServiceManagement\Events\UpdateMobileServicePolicy;
use Workdo\MobileServiceManagement\Events\DestroyMobileServicePolicy;

class MobileServicePolicyController extends Controller
{
    public function index(MobileServicePolicyDataTable $datatable)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy manage')) {
            return $datatable->render('mobile-service-management::mobileservice-policy.index');
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if (Auth::user()->isAbleTo('mobileservice_policy create')) {
            return view('mobile-service-management::mobileservice-policy.create');
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy create')) {
            $validator = Validator::make($request->all(), [
                'policy_name'     => 'required|string',
                'description'     => 'required|string',
                'response_time'   => 'required|string',
                'resolution_time' => 'required|string',
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $mobileservice_policy                  = new MobileServicePolicy();
            $mobileservice_policy->policy_name     = $request->policy_name;
            $mobileservice_policy->description     = $request->description;
            $mobileservice_policy->response_time   = $request->response_time;
            $mobileservice_policy->resolution_time = $request->resolution_time;
            $mobileservice_policy->workspace       = getActiveWorkSpace();
            $mobileservice_policy->created_by      = creatorId();
            $mobileservice_policy->save();

            event(new CreateMobileServicePolicy($request, $mobileservice_policy));

            return redirect()->route('mobileservice-policy.index')->with('success', __('The policy has been created successfully'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy edit')) {
            $mobileservice_policy = MobileServicePolicy::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_policy) {
                return view('mobile-service-management::mobileservice-policy.edit', compact('mobileservice_policy'));
            } else {
                return response()->json(['error' => __('Policy not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy edit')) {
            $validator = Validator::make($request->all(), [
                'policy_name'     => 'required|string',
                'description'     => 'required|string',
                'response_time'   => 'required|string',
                'resolution_time' => 'required|string',
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $mobileservice_policy = MobileServicePolicy::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_policy) {
                $mobileservice_policy->policy_name     = $request->policy_name;
                $mobileservice_policy->description     = $request->description;
                $mobileservice_policy->response_time   = $request->response_time;
                $mobileservice_policy->resolution_time = $request->resolution_time;
                $mobileservice_policy->save();

                event(new UpdateMobileServicePolicy($request, $mobileservice_policy));

                return redirect()->route('mobileservice-policy.index')->with('success', __('The policy details are updated successfully'));
            } else {
                return redirect()->back()->with('error', __('Policy not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy delete')) {
            $mobileservice_policy = MobileServicePolicy::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_policy) {
                event(new DestroyMobileServicePolicy($mobileservice_policy));
                $mobileservice_policy->delete();

                return redirect()->back()->with('success', __('The policy has been deleted'));
            } else {
                return redirect()->back()->with('error', __('Policy not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function viewpolicy($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_policy manage')) {
            $mobileservice_policy = MobileServicePolicy::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_policy) {
                return view('mobile-service-management::mobileservice-policy.description', compact('mobileservice_policy'));
            } else {
                return response()->json(['error' => __('Policy not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }
}
